/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.raft.cluster.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import io.atomix.cluster.MemberId;
import io.atomix.protocols.raft.RaftError;
import io.atomix.protocols.raft.RaftServer;
import io.atomix.protocols.raft.cluster.RaftCluster;
import io.atomix.protocols.raft.cluster.RaftClusterEvent;
import io.atomix.protocols.raft.cluster.RaftClusterEventListener;
import io.atomix.protocols.raft.cluster.RaftMember;
import io.atomix.protocols.raft.cluster.impl.DefaultRaftMember;
import io.atomix.protocols.raft.cluster.impl.RaftMemberContext;
import io.atomix.protocols.raft.impl.RaftContext;
import io.atomix.protocols.raft.protocol.JoinRequest;
import io.atomix.protocols.raft.protocol.LeaveRequest;
import io.atomix.protocols.raft.protocol.RaftResponse;
import io.atomix.protocols.raft.storage.system.Configuration;
import io.atomix.utils.concurrent.Futures;
import io.atomix.utils.concurrent.Scheduled;
import io.atomix.utils.event.Event;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;

public final class RaftClusterContext
implements RaftCluster,
AutoCloseable {
    private final Logger log;
    private final RaftContext raft;
    private final DefaultRaftMember member;
    private volatile Configuration configuration;
    private final Map<MemberId, RaftMemberContext> membersMap = new ConcurrentHashMap<MemberId, RaftMemberContext>();
    private final Set<RaftMember> members = new CopyOnWriteArraySet<RaftMember>();
    private final List<RaftMemberContext> remoteMembers = new CopyOnWriteArrayList<RaftMemberContext>();
    private final Map<RaftMember.Type, List<RaftMemberContext>> memberTypes = new HashMap<RaftMember.Type, List<RaftMemberContext>>();
    private volatile Scheduled joinTimeout;
    private volatile CompletableFuture<Void> joinFuture;
    private volatile Scheduled leaveTimeout;
    private volatile CompletableFuture<Void> leaveFuture;
    private final Set<RaftClusterEventListener> listeners = new CopyOnWriteArraySet<RaftClusterEventListener>();

    public RaftClusterContext(MemberId localMemberId, RaftContext raft) {
        Instant time = Instant.now();
        this.member = new DefaultRaftMember(localMemberId, RaftMember.Type.PASSIVE, time).setCluster(this);
        this.raft = (RaftContext)Preconditions.checkNotNull((Object)raft, (Object)"context cannot be null");
        this.log = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftServer.class).addValue((Object)raft.getName()).build());
        this.configuration = raft.getMetaStore().loadConfiguration();
        if (this.configuration != null) {
            Instant updateTime = Instant.ofEpochMilli(this.configuration.time());
            for (RaftMember member : this.configuration.members()) {
                if (member.equals(this.member)) {
                    this.member.setType(member.getType());
                    this.members.add(this.member);
                    continue;
                }
                RaftMemberContext state = new RaftMemberContext(new DefaultRaftMember(member.memberId(), member.getType(), updateTime), this);
                state.resetState(raft.getLog());
                this.members.add(state.getMember());
                this.remoteMembers.add(state);
                this.membersMap.put(member.memberId(), state);
                List<RaftMemberContext> memberType = this.memberTypes.get((Object)member.getType());
                if (memberType == null) {
                    memberType = new CopyOnWriteArrayList<RaftMemberContext>();
                    this.memberTypes.put(member.getType(), memberType);
                }
                memberType.add(state);
            }
        }
    }

    public RaftContext getContext() {
        return this.raft;
    }

    public Configuration getConfiguration() {
        return this.configuration;
    }

    @Override
    public RaftMember getLeader() {
        return this.raft.getLeader();
    }

    @Override
    public long getTerm() {
        return this.raft.getTerm();
    }

    @Override
    public void addLeaderElectionListener(Consumer<RaftMember> callback) {
        this.raft.addLeaderElectionListener(callback);
    }

    @Override
    public void removeLeaderElectionListener(Consumer<RaftMember> listener) {
        this.raft.removeLeaderElectionListener(listener);
    }

    @Override
    public RaftMember getMember() {
        return this.member;
    }

    @Override
    public Collection<RaftMember> getMembers() {
        return new ArrayList<RaftMember>(this.members);
    }

    @Override
    public DefaultRaftMember getMember(MemberId id) {
        if (this.member.memberId().equals((Object)id)) {
            return this.member;
        }
        return this.getRemoteMember(id);
    }

    @Override
    public void addListener(RaftClusterEventListener listener) {
        this.listeners.add(listener);
    }

    @Override
    public void removeListener(RaftClusterEventListener listener) {
        this.listeners.remove(listener);
    }

    public int getQuorum() {
        return (int)Math.floor((double)(this.getActiveMemberStates().size() + 1) / 2.0) + 1;
    }

    public RaftMemberContext getMemberState(MemberId id) {
        return this.membersMap.get(id);
    }

    public DefaultRaftMember getRemoteMember(MemberId id) {
        RaftMemberContext member = this.membersMap.get(id);
        return member != null ? member.getMember() : null;
    }

    public List<RaftMemberContext> getRemoteMemberStates() {
        return this.remoteMembers;
    }

    public List<RaftMemberContext> getRemoteMemberStates(RaftMember.Type type) {
        List members = this.memberTypes.get((Object)type);
        return members != null ? members : Collections.EMPTY_LIST;
    }

    public List<RaftMemberContext> getActiveMemberStates() {
        return this.getRemoteMemberStates(RaftMember.Type.ACTIVE);
    }

    public List<RaftMemberContext> getActiveMemberStates(Comparator<RaftMemberContext> comparator) {
        ArrayList<RaftMemberContext> activeMembers = new ArrayList<RaftMemberContext>(this.getActiveMemberStates());
        activeMembers.sort(comparator);
        return activeMembers;
    }

    public List<RaftMemberContext> getPassiveMemberStates() {
        return this.getRemoteMemberStates(RaftMember.Type.PASSIVE);
    }

    public List<RaftMemberContext> getPassiveMemberStates(Comparator<RaftMemberContext> comparator) {
        ArrayList<RaftMemberContext> passiveMembers = new ArrayList<RaftMemberContext>(this.getPassiveMemberStates());
        passiveMembers.sort(comparator);
        return passiveMembers;
    }

    @Override
    public CompletableFuture<Void> bootstrap(Collection<MemberId> cluster) {
        if (this.joinFuture != null) {
            return this.joinFuture;
        }
        if (this.configuration == null) {
            this.member.setType(RaftMember.Type.ACTIVE);
            Set<RaftMember> activeMembers = cluster.stream().filter(m -> !m.equals((Object)this.member.memberId())).map(m -> new DefaultRaftMember((MemberId)m, RaftMember.Type.ACTIVE, this.member.getLastUpdated())).collect(Collectors.toSet());
            activeMembers.add(this.member);
            this.configure(new Configuration(0L, 0L, this.member.getLastUpdated().toEpochMilli(), activeMembers));
        }
        return this.join();
    }

    @Override
    public synchronized CompletableFuture<Void> listen(Collection<MemberId> cluster) {
        if (this.joinFuture != null) {
            return this.joinFuture;
        }
        if (this.configuration == null) {
            this.member.setType(RaftMember.Type.PASSIVE);
            Set<RaftMember> activeMembers = cluster.stream().filter(m -> !m.equals((Object)this.member.memberId())).map(m -> new DefaultRaftMember((MemberId)m, RaftMember.Type.ACTIVE, this.member.getLastUpdated())).collect(Collectors.toSet());
            if (activeMembers.isEmpty()) {
                return Futures.exceptionalFuture((Throwable)new IllegalStateException("cannot join empty cluster"));
            }
            this.configure(new Configuration(0L, 0L, this.member.getLastUpdated().toEpochMilli(), activeMembers));
        }
        return this.join();
    }

    @Override
    public synchronized CompletableFuture<Void> join(Collection<MemberId> cluster) {
        if (this.joinFuture != null) {
            return this.joinFuture;
        }
        if (this.configuration == null) {
            this.member.setType(RaftMember.Type.PROMOTABLE);
            Set<RaftMember> activeMembers = cluster.stream().filter(m -> !m.equals((Object)this.member.memberId())).map(m -> new DefaultRaftMember((MemberId)m, RaftMember.Type.ACTIVE, this.member.getLastUpdated())).collect(Collectors.toSet());
            if (activeMembers.isEmpty()) {
                return Futures.exceptionalFuture((Throwable)new IllegalStateException("cannot join empty cluster"));
            }
            this.configure(new Configuration(0L, 0L, this.member.getLastUpdated().toEpochMilli(), activeMembers));
        }
        return this.join().thenCompose(v -> {
            if (this.member.getType() == RaftMember.Type.ACTIVE) {
                return CompletableFuture.completedFuture(null);
            }
            return this.member.promote(RaftMember.Type.ACTIVE);
        });
    }

    private synchronized CompletableFuture<Void> join() {
        this.joinFuture = new CompletableFuture();
        this.raft.getThreadContext().execute(() -> {
            this.raft.transition(this.member.getType());
            List<RaftMemberContext> activeMembers = this.getActiveMemberStates();
            if (!activeMembers.isEmpty()) {
                this.join(this.getActiveMemberStates().iterator());
            } else {
                this.joinFuture.complete(null);
            }
        });
        return this.joinFuture.whenComplete((result, error) -> {
            this.joinFuture = null;
        });
    }

    private void join(Iterator<RaftMemberContext> iterator) {
        if (iterator.hasNext()) {
            this.cancelJoinTimer();
            this.joinTimeout = this.raft.getThreadContext().schedule(this.raft.getElectionTimeout().multipliedBy(2L), () -> this.join(iterator));
            RaftMemberContext member = iterator.next();
            this.log.debug("Attempting to join via {}", (Object)member.getMember().memberId());
            JoinRequest request = ((JoinRequest.Builder)JoinRequest.builder().withMember(new DefaultRaftMember(this.getMember().memberId(), this.getMember().getType(), this.getMember().getLastUpdated()))).build();
            this.raft.getProtocol().join(member.getMember().memberId(), request).whenCompleteAsync((response, error) -> {
                this.cancelJoinTimer();
                if (error == null) {
                    if (response.status() == RaftResponse.Status.OK) {
                        this.log.debug("Successfully joined via {}", (Object)member.getMember().memberId());
                        Configuration configuration = new Configuration(response.index(), response.term(), response.timestamp(), response.members());
                        this.configure(configuration).commit();
                        if (!this.members.contains(this.member)) {
                            this.joinFuture.completeExceptionally(new IllegalStateException("not a member of the cluster"));
                        } else if (this.joinFuture != null) {
                            this.joinFuture.complete(null);
                        }
                    } else if (response.error() == null || response.error().type() == RaftError.Type.CONFIGURATION_ERROR) {
                        this.log.debug("Failed to join {}", (Object)member.getMember().memberId());
                        this.resetJoinTimer();
                    } else {
                        this.log.debug("Failed to join {}", (Object)member.getMember().memberId());
                        this.join(iterator);
                    }
                } else {
                    this.log.debug("Failed to join {}", (Object)member.getMember().memberId());
                    this.join(iterator);
                }
            }, (Executor)this.raft.getThreadContext());
        } else {
            this.log.debug("Failed to join cluster, retrying...");
            this.resetJoinTimer();
        }
    }

    private void resetJoinTimer() {
        this.cancelJoinTimer();
        this.joinTimeout = this.raft.getThreadContext().schedule(this.raft.getElectionTimeout().multipliedBy(2L), () -> this.join(this.getActiveMemberStates().iterator()));
    }

    private void cancelJoinTimer() {
        if (this.joinTimeout != null) {
            this.log.trace("Cancelling join timeout");
            this.joinTimeout.cancel();
            this.joinTimeout = null;
        }
    }

    @Override
    public synchronized CompletableFuture<Void> leave() {
        if (this.leaveFuture != null) {
            return this.leaveFuture;
        }
        this.leaveFuture = new CompletableFuture();
        this.raft.getThreadContext().execute(() -> {
            this.cancelJoinTimer();
            if (this.joinFuture != null) {
                this.joinFuture.completeExceptionally(new IllegalStateException("failed to join cluster"));
            }
            if (this.getActiveMemberStates().isEmpty() && this.configuration.index() <= this.raft.getCommitIndex()) {
                this.log.trace("Single member cluster. Transitioning directly to inactive.");
                this.raft.transition(RaftServer.Role.INACTIVE);
                this.leaveFuture.complete(null);
            } else {
                this.leave(this.leaveFuture);
            }
        });
        return this.leaveFuture.whenComplete((result, error) -> {
            this.leaveFuture = null;
        });
    }

    private void leave(CompletableFuture<Void> future) {
        this.leaveTimeout = this.raft.getThreadContext().schedule(this.raft.getElectionTimeout(), () -> this.leave(future));
        this.raft.getRaftRole().onLeave(((LeaveRequest.Builder)LeaveRequest.builder().withMember(this.getMember())).build()).whenComplete((response, error) -> {
            this.cancelLeaveTimer();
            if (error == null && response.status() == RaftResponse.Status.OK) {
                Configuration configuration = new Configuration(response.index(), response.term(), response.timestamp(), response.members());
                this.configure(configuration).commit();
                future.complete(null);
            } else {
                this.leaveTimeout = this.raft.getThreadContext().schedule(this.raft.getElectionTimeout(), () -> this.leave(future));
            }
        });
    }

    private void cancelLeaveTimer() {
        if (this.leaveTimeout != null) {
            this.log.trace("Cancelling leave timeout");
            this.leaveTimeout.cancel();
            this.leaveTimeout = null;
        }
    }

    public RaftClusterContext reset() {
        this.configure(this.raft.getMetaStore().loadConfiguration());
        return this;
    }

    public RaftClusterContext commit() {
        this.raft.transition(this.member.getType());
        if (!this.configuration.members().contains(this.member) && this.leaveFuture != null) {
            this.leaveFuture.complete(null);
        }
        if (this.raft.getMetaStore().loadConfiguration().index() < this.configuration.index()) {
            this.raft.getMetaStore().storeConfiguration(this.configuration);
        }
        return this;
    }

    public RaftClusterContext configure(Configuration configuration) {
        Preconditions.checkNotNull((Object)configuration, (Object)"configuration cannot be null");
        if (this.configuration != null && configuration.index() <= this.configuration.index()) {
            return this;
        }
        Instant time = Instant.ofEpochMilli(configuration.time());
        boolean transition = false;
        for (RaftMember raftMember : configuration.members()) {
            if (raftMember.equals(this.member)) {
                transition = this.member.getType().ordinal() < raftMember.getType().ordinal();
                this.member.update(raftMember.getType(), time);
                this.members.add(this.member);
                continue;
            }
            RaftMemberContext state = this.membersMap.get(raftMember.memberId());
            if (state == null) {
                DefaultRaftMember defaultMember = new DefaultRaftMember(raftMember.memberId(), raftMember.getType(), time);
                state = new RaftMemberContext(defaultMember, this);
                state.resetState(this.raft.getLog());
                this.members.add(state.getMember());
                this.remoteMembers.add(state);
                this.membersMap.put(raftMember.memberId(), state);
                this.listeners.forEach(l -> l.event((Event)new RaftClusterEvent(RaftClusterEvent.Type.JOIN, defaultMember, time.toEpochMilli())));
            }
            if (state.getMember().getType() != raftMember.getType()) {
                state.getMember().update(raftMember.getType(), time);
                state.resetState(this.raft.getLog());
            }
            for (List list : this.memberTypes.values()) {
                list.remove(state);
            }
            List<RaftMemberContext> memberType = this.memberTypes.get((Object)raftMember.getType());
            if (memberType == null) {
                memberType = new CopyOnWriteArrayList<RaftMemberContext>();
                this.memberTypes.put(raftMember.getType(), memberType);
            }
            memberType.add(state);
        }
        if (transition) {
            this.raft.transition(this.member.getType());
        }
        int i = 0;
        while (i < this.remoteMembers.size()) {
            RaftMemberContext raftMemberContext = this.remoteMembers.get(i);
            if (!configuration.members().contains(raftMemberContext.getMember())) {
                this.members.remove(raftMemberContext.getMember());
                this.remoteMembers.remove(i);
                for (List<RaftMemberContext> memberType : this.memberTypes.values()) {
                    memberType.remove(raftMemberContext);
                }
                this.membersMap.remove(raftMemberContext.getMember().memberId());
                this.listeners.forEach(l -> l.event((Event)new RaftClusterEvent(RaftClusterEvent.Type.LEAVE, raftMemberContext.getMember(), time.toEpochMilli())));
                continue;
            }
            ++i;
        }
        if (!configuration.members().contains(this.member)) {
            this.members.remove(this.member);
        }
        this.configuration = configuration;
        if (this.raft.getCommitIndex() >= configuration.index()) {
            this.raft.getMetaStore().storeConfiguration(configuration);
        }
        return this;
    }

    @Override
    public void close() {
        for (RaftMemberContext member : this.remoteMembers) {
            member.getMember().close();
        }
        this.member.close();
        this.cancelJoinTimer();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("server", (Object)this.raft.getName()).toString();
    }
}

