/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.gson.JsonParseException;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GCSNotebookRepo
implements NotebookRepo {
    private static final Logger LOGGER = LoggerFactory.getLogger(GCSNotebookRepo.class);
    private String encoding;
    private String bucketName;
    private Optional<String> basePath;
    private Pattern notePathPattern;
    private Storage storage;

    public GCSNotebookRepo() {
    }

    @VisibleForTesting
    public GCSNotebookRepo(ZeppelinConfiguration zConf, Storage storage) throws IOException {
        this.init(zConf);
        this.storage = storage;
    }

    public void init(ZeppelinConfiguration zConf) throws IOException {
        this.encoding = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING);
        String gcsStorageDir = zConf.getGCSStorageDir();
        if (gcsStorageDir.isEmpty()) {
            throw new IOException("GCS storage directory must be set using 'zeppelin.notebook.gcs.dir'");
        }
        if (!gcsStorageDir.startsWith("gs://")) {
            throw new IOException(String.format("GCS storage directory '%s' must start with 'gs://'.", gcsStorageDir));
        }
        String storageDirWithoutScheme = gcsStorageDir.substring("gs://".length());
        List<String> pathComponents = Arrays.asList(storageDirWithoutScheme.split("/"));
        if (pathComponents.isEmpty()) {
            throw new IOException(String.format("GCS storage directory '%s' must be in the form gs://bucketname/path/to/dir", gcsStorageDir));
        }
        this.bucketName = pathComponents.get(0);
        this.basePath = pathComponents.size() > 1 ? Optional.of(StringUtils.join(pathComponents.subList(1, pathComponents.size()), (String)"/")) : Optional.empty();
        this.notePathPattern = this.basePath.isPresent() ? Pattern.compile("^" + Pattern.quote(this.basePath.get() + "/") + "(.+\\.zpln)$") : Pattern.compile("^(.+\\.zpln)$");
        GoogleCredentials credentials = GoogleCredentials.getApplicationDefault();
        String credentialJsonPath = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_GCS_CREDENTIALS_FILE);
        if (credentialJsonPath != null) {
            credentials = GoogleCredentials.fromStream((InputStream)new FileInputStream(credentialJsonPath));
        }
        this.storage = (Storage)((StorageOptions.Builder)StorageOptions.newBuilder().setCredentials((Credentials)credentials)).build().getService();
    }

    private BlobId makeBlobId(String noteId, String notePath) throws IOException {
        if (this.basePath.isPresent()) {
            return BlobId.of((String)this.bucketName, (String)(this.basePath.get() + "/" + this.buildNoteFileName(noteId, notePath)));
        }
        return BlobId.of((String)this.bucketName, (String)this.buildNoteFileName(noteId, notePath));
    }

    public Map<String, NoteInfo> list(AuthenticationInfo subject) throws IOException {
        try {
            HashMap<String, NoteInfo> infos = new HashMap<String, NoteInfo>();
            Iterable blobsUnderDir = this.basePath.isPresent() ? this.storage.list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)(this.basePath.get() + "/"))}).iterateAll() : this.storage.list(this.bucketName, new Storage.BlobListOption[0]).iterateAll();
            for (Blob b : blobsUnderDir) {
                Matcher matcher = this.notePathPattern.matcher(b.getName());
                if (!matcher.matches()) continue;
                String noteFileName = matcher.group(1);
                try {
                    String noteId = this.getNoteId(noteFileName);
                    String notePath = this.getNotePath("", noteFileName);
                    infos.put(noteId, new NoteInfo(noteId, notePath));
                }
                catch (IOException e) {
                    LOGGER.warn(e.getMessage());
                }
            }
            return infos;
        }
        catch (StorageException se) {
            throw new IOException("Could not list GCS directory: " + se.getMessage(), se);
        }
    }

    public Note get(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        byte[] contents;
        BlobId blobId = this.makeBlobId(noteId, notePath);
        try {
            contents = this.storage.readAllBytes(blobId, new Storage.BlobSourceOption[0]);
        }
        catch (StorageException se) {
            throw new IOException("Could not read " + blobId.toString() + ": " + se.getMessage(), se);
        }
        try {
            return Note.fromJson((String)noteId, (String)new String(contents, this.encoding));
        }
        catch (JsonParseException jpe) {
            throw new IOException("Could note parse as json " + blobId.toString() + jpe.getMessage(), jpe);
        }
    }

    public void save(Note note, AuthenticationInfo subject) throws IOException {
        BlobInfo info = BlobInfo.newBuilder((BlobId)this.makeBlobId(note.getId(), note.getPath())).setContentType("application/json").build();
        try {
            this.storage.create(info, note.toJson().getBytes(StandardCharsets.UTF_8), new Storage.BlobTargetOption[0]);
        }
        catch (StorageException se) {
            throw new IOException("Could not write " + info.toString() + ": " + se.getMessage(), se);
        }
    }

    public void move(String noteId, String notePath, String newNotePath, AuthenticationInfo subject) throws IOException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)noteId));
        BlobId sourceBlobId = this.makeBlobId(noteId, notePath);
        BlobId destinationBlobId = this.makeBlobId(noteId, newNotePath);
        try {
            this.storage.get(sourceBlobId).copyTo(destinationBlobId, new Blob.BlobSourceOption[0]);
        }
        catch (Exception se) {
            throw new IOException("Could not copy from " + sourceBlobId.toString() + " to " + destinationBlobId.toString() + ": " + se.getMessage(), se);
        }
        this.remove(noteId, notePath, subject);
    }

    public void move(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        if (!folderPath.endsWith("/")) {
            folderPath = folderPath + "/";
        }
        if (!newFolderPath.endsWith("/")) {
            newFolderPath = newFolderPath + "/";
        }
        if (this.basePath.isPresent()) {
            folderPath = this.basePath.get() + folderPath;
            newFolderPath = this.basePath.get() + newFolderPath;
        }
        String oldPath = folderPath;
        String newPath = newFolderPath;
        try {
            ArrayList toBeDeleted = new ArrayList();
            this.storage.list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)oldPath)}).getValues().forEach(note -> toBeDeleted.add(note.getBlobId()));
            if (toBeDeleted.isEmpty()) {
                throw new IOException("Empty folder or folder does not exist: " + oldPath);
            }
            for (BlobId note2 : toBeDeleted) {
                String newLocation = ("/" + note2.getName()).replaceFirst(oldPath, newPath).substring(1);
                this.storage.get(note2).copyTo(this.bucketName, newLocation, new Blob.BlobSourceOption[0]);
                this.storage.delete(note2);
            }
        }
        catch (Exception se) {
            throw new IOException("Could not copy from " + oldPath + " to " + newPath + ": " + se.getMessage(), se);
        }
    }

    public void remove(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)noteId));
        BlobId blobId = this.makeBlobId(noteId, notePath);
        try {
            boolean deleted = this.storage.delete(blobId);
            if (!deleted) {
                throw new IOException("Tried to remove nonexistent blob " + blobId.toString());
            }
        }
        catch (StorageException se) {
            throw new IOException("Could not remove " + blobId.toString() + ": " + se.getMessage(), se);
        }
    }

    public void remove(String folderPath, AuthenticationInfo subject) throws IOException {
        if (!folderPath.endsWith("/")) {
            folderPath = folderPath + "/";
        }
        if (this.basePath.isPresent()) {
            folderPath = this.basePath.get() + folderPath;
        }
        String oldPath = folderPath;
        try {
            ArrayList toBeDeleted = new ArrayList();
            this.storage.list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)oldPath)}).getValues().forEach(note -> toBeDeleted.add(note.getBlobId()));
            if (toBeDeleted.isEmpty()) {
                throw new IOException("Empty folder or folder does not exist: " + oldPath);
            }
            for (BlobId note2 : toBeDeleted) {
                this.storage.delete(note2);
            }
        }
        catch (Exception se) {
            throw new IOException("Could not delete from " + oldPath + ": " + se.getMessage(), se);
        }
    }

    public void close() {
    }

    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        LOGGER.warn("getSettings is not implemented for GCSNotebookRepo");
        return Collections.emptyList();
    }

    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        LOGGER.warn("updateSettings is not implemented for GCSNotebookRepo");
    }
}

