/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.exception.NotePathAlreadyExistsException;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class NoteManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(NoteManager.class);
    public static final String TRASH_FOLDER = "~Trash";
    private Folder root;
    private Folder trash;
    private NotebookRepo notebookRepo;
    private Map<String, String> notesInfo;

    @Inject
    public NoteManager(NotebookRepo notebookRepo) throws IOException {
        this.notebookRepo = notebookRepo;
        this.root = new Folder("/", notebookRepo);
        this.trash = this.root.getOrCreateFolder(TRASH_FOLDER);
        this.init();
    }

    private void init() throws IOException {
        this.notesInfo = this.notebookRepo.list(AuthenticationInfo.ANONYMOUS).values().stream().collect(Collectors.toMap(NoteInfo::getId, NoteInfo::getPath));
        this.notesInfo.entrySet().stream().forEach(entry -> {
            try {
                this.addOrUpdateNoteNode(new Note(new NoteInfo((String)entry.getKey(), (String)entry.getValue())));
            }
            catch (IOException e) {
                LOGGER.warn(e.getMessage());
            }
        });
    }

    public Map<String, String> getNotesInfo() {
        return this.notesInfo;
    }

    public Stream<Note> getNotesStream() {
        return this.notesInfo.values().stream().map(notePath -> {
            try {
                return this.getNoteNode((String)notePath).getNote();
            }
            catch (Exception e) {
                LOGGER.warn("Fail to load note: {}", notePath, (Object)e);
                return null;
            }
        }).filter(Objects::nonNull);
    }

    public void reloadNotes() throws IOException {
        this.root = new Folder("/", this.notebookRepo);
        this.trash = this.root.getOrCreateFolder(TRASH_FOLDER);
        this.init();
    }

    private void addOrUpdateNoteNode(Note note, boolean checkDuplicates) throws IOException {
        String notePath = note.getPath();
        if (checkDuplicates && !this.isNotePathAvailable(notePath)) {
            throw new NotePathAlreadyExistsException("Note '" + notePath + "' existed");
        }
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i])) continue;
            curFolder = curFolder.getOrCreateFolder(tokens[i]);
        }
        curFolder.addNote(tokens[tokens.length - 1], note);
        this.notesInfo.put(note.getId(), note.getPath());
    }

    private void addOrUpdateNoteNode(Note note) throws IOException {
        this.addOrUpdateNoteNode(note, false);
    }

    public boolean containsNote(String notePath) {
        try {
            this.getNoteNode(notePath);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public boolean containsFolder(String folderPath) {
        try {
            this.getFolder(folderPath);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public void saveNote(Note note, AuthenticationInfo subject) throws IOException {
        if (note.isRemoved()) {
            LOGGER.warn("Try to save note: {} when it is removed", (Object)note.getId());
        } else if (note.isLoaded() || !note.isSaved()) {
            this.addOrUpdateNoteNode(note);
            this.notebookRepo.save(note, subject);
            note.setSaved(true);
        } else {
            LOGGER.warn("Try to save note: {} when it is unloaded", (Object)note.getId());
        }
    }

    public void addNote(Note note, AuthenticationInfo subject) throws IOException {
        this.addOrUpdateNoteNode(note, true);
        note.setLoaded(true);
    }

    public void saveNote(Note note) throws IOException {
        this.saveNote(note, AuthenticationInfo.ANONYMOUS);
    }

    public void removeNote(String noteId, AuthenticationInfo subject) throws IOException {
        String notePath = this.notesInfo.remove(noteId);
        Folder folder = this.getOrCreateFolder(this.getFolderName(notePath));
        folder.removeNote(this.getNoteName(notePath));
        this.notebookRepo.remove(noteId, notePath, subject);
    }

    public void moveNote(String noteId, String newNotePath, AuthenticationInfo subject) throws IOException {
        String notePath = this.notesInfo.get(noteId);
        if (noteId == null) {
            throw new IOException("No metadata found for this note: " + noteId);
        }
        if (!this.isNotePathAvailable(newNotePath)) {
            throw new NotePathAlreadyExistsException("Note '" + newNotePath + "' existed");
        }
        NoteNode noteNode = this.getNoteNode(notePath);
        noteNode.getParent().removeNote(this.getNoteName(notePath));
        noteNode.setNotePath(newNotePath);
        String newParent = this.getFolderName(newNotePath);
        Folder newFolder = this.getOrCreateFolder(newParent);
        newFolder.addNoteNode(noteNode);
        this.notesInfo.put(noteId, newNotePath);
        this.notebookRepo.move(noteId, notePath, newNotePath, subject);
    }

    public void moveFolder(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        this.notebookRepo.move(folderPath, newFolderPath, subject);
        Folder folder = this.getFolder(folderPath);
        folder.getParent().removeFolder(folder.getName(), subject);
        Folder newFolder = this.getOrCreateFolder(newFolderPath);
        newFolder.getParent().addFolder(newFolder.getName(), folder);
        for (Note note : folder.getRawNotesRecursively()) {
            this.notesInfo.put(note.getId(), note.getPath());
        }
    }

    public List<Note> removeFolder(String folderPath, AuthenticationInfo subject) throws IOException {
        this.notebookRepo.remove(folderPath, subject);
        Folder folder = this.getFolder(folderPath);
        List<Note> notes = folder.getParent().removeFolder(folder.getName(), subject);
        for (Note note : notes) {
            this.notesInfo.remove(note.getId());
        }
        return notes;
    }

    public Note getNote(String noteId, boolean reload) throws IOException {
        String notePath = this.notesInfo.get(noteId);
        if (notePath == null) {
            return null;
        }
        NoteNode noteNode = this.getNoteNode(notePath);
        return noteNode.getNote(reload);
    }

    public Note getNote(String noteId) throws IOException {
        String notePath = this.notesInfo.get(noteId);
        if (notePath == null) {
            return null;
        }
        NoteNode noteNode = this.getNoteNode(notePath);
        return noteNode.getNote();
    }

    public Folder getOrCreateFolder(String folderName) {
        String[] tokens = folderName.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i])) continue;
            curFolder = curFolder.getOrCreateFolder(tokens[i]);
        }
        return curFolder;
    }

    private NoteNode getNoteNode(String notePath) throws IOException {
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            throw new IOException("Can not find note: " + notePath);
        }
        NoteNode noteNode = curFolder.getNote(tokens[tokens.length - 1]);
        if (noteNode == null) {
            throw new IOException("Can not find note: " + notePath);
        }
        return noteNode;
    }

    private Folder getFolder(String folderPath) throws IOException {
        String[] tokens = folderPath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            throw new IOException("Can not find folder: " + folderPath);
        }
        return curFolder;
    }

    public Folder getTrashFolder() {
        return this.trash;
    }

    private String getFolderName(String notePath) {
        int pos = notePath.lastIndexOf(47);
        return notePath.substring(0, pos);
    }

    private String getNoteName(String notePath) {
        int pos = notePath.lastIndexOf(47);
        return notePath.substring(pos + 1);
    }

    private boolean isNotePathAvailable(String notePath) {
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            return true;
        }
        return !curFolder.containsNote(tokens[tokens.length - 1]);
    }

    public static class NoteNode {
        private Folder parent;
        private Note note;
        private NotebookRepo notebookRepo;

        public NoteNode(Note note, Folder parent, NotebookRepo notebookRepo) {
            this.note = note;
            this.parent = parent;
            this.notebookRepo = notebookRepo;
        }

        public synchronized Note getNote() throws IOException {
            return this.getNote(false);
        }

        public synchronized Note getNote(boolean reload) throws IOException {
            if (!this.note.isLoaded() || reload) {
                this.note = this.notebookRepo.get(this.note.getId(), this.note.getPath(), AuthenticationInfo.ANONYMOUS);
                if (this.parent.toString().equals("/")) {
                    this.note.setPath("/" + this.note.getName());
                } else {
                    this.note.setPath(this.parent.toString() + "/" + this.note.getName());
                }
                this.note.setCronSupported(ZeppelinConfiguration.create());
                this.note.setLoaded(true);
            }
            return this.note;
        }

        public String getNoteId() {
            return this.note.getId();
        }

        public String getNoteName() {
            return this.note.getName();
        }

        public String getNotePath() {
            if (this.parent.getPath().equals("/")) {
                return this.parent.getPath() + this.note.getName();
            }
            return this.parent.getPath() + "/" + this.note.getName();
        }

        public Note getRawNote() {
            return this.note;
        }

        public Folder getParent() {
            return this.parent;
        }

        public String toString() {
            return this.getNotePath();
        }

        public void setParent(Folder parent) {
            this.parent = parent;
        }

        public void setNotePath(String notePath) {
            this.note.setPath(notePath);
        }

        public void updateNotePath() {
            this.note.setPath(this.getNotePath());
        }
    }

    public static class Folder {
        private String name;
        private Folder parent;
        private NotebookRepo notebookRepo;
        private Map<String, NoteNode> notes = new HashMap<String, NoteNode>();
        private Map<String, Folder> subFolders = new HashMap<String, Folder>();

        public Folder(String name, NotebookRepo notebookRepo) {
            this.name = name;
            this.notebookRepo = notebookRepo;
        }

        public Folder(String name, Folder parent, NotebookRepo notebookRepo) {
            this(name, notebookRepo);
            this.parent = parent;
        }

        public synchronized Folder getOrCreateFolder(String folderName) {
            if (StringUtils.isBlank((CharSequence)folderName)) {
                return this;
            }
            if (!this.subFolders.containsKey(folderName)) {
                this.subFolders.put(folderName, new Folder(folderName, this, this.notebookRepo));
            }
            return this.subFolders.get(folderName);
        }

        public Folder getParent() {
            return this.parent;
        }

        public void setParent(Folder parent) {
            this.parent = parent;
        }

        public String getName() {
            return this.name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Folder getFolder(String folderName) {
            return this.subFolders.get(folderName);
        }

        public Map<String, Folder> getFolders() {
            return this.subFolders;
        }

        public NoteNode getNote(String noteName) {
            return this.notes.get(noteName);
        }

        public void addNote(String noteName, Note note) {
            this.notes.put(noteName, new NoteNode(note, this, this.notebookRepo));
        }

        public void addFolder(String folderName, Folder folder) throws IOException {
            this.subFolders.put(folderName, folder);
            folder.setParent(this);
            folder.setName(folderName);
            for (NoteNode noteNode : folder.getNoteNodeRecursively()) {
                noteNode.updateNotePath();
            }
        }

        public boolean containsNote(String noteName) {
            return this.notes.containsKey(noteName);
        }

        public void addNoteNode(NoteNode noteNode) {
            this.notes.put(noteNode.getNoteName(), noteNode);
            noteNode.setParent(this);
        }

        public void removeNote(String noteName) {
            this.notes.remove(noteName);
        }

        public List<Note> removeFolder(String folderName, AuthenticationInfo subject) throws IOException {
            Folder folder = this.subFolders.remove(folderName);
            return folder.getRawNotesRecursively();
        }

        public List<Note> getRawNotesRecursively() {
            ArrayList<Note> notesInfo = new ArrayList<Note>();
            for (NoteNode noteNode : this.notes.values()) {
                notesInfo.add(noteNode.getRawNote());
            }
            for (Folder folder : this.subFolders.values()) {
                notesInfo.addAll(folder.getRawNotesRecursively());
            }
            return notesInfo;
        }

        public List<NoteNode> getNoteNodeRecursively() {
            ArrayList<NoteNode> noteNodeRecursively = new ArrayList<NoteNode>();
            noteNodeRecursively.addAll(this.notes.values());
            for (Folder folder : this.subFolders.values()) {
                noteNodeRecursively.addAll(folder.getNoteNodeRecursively());
            }
            return noteNodeRecursively;
        }

        public Map<String, NoteNode> getNotes() {
            return this.notes;
        }

        public String getPath() {
            if (this.name.equals("/")) {
                return this.name;
            }
            if (this.parent.name.equals("/")) {
                return "/" + this.name;
            }
            return this.parent.toString() + "/" + this.name;
        }

        public String toString() {
            return this.getPath();
        }
    }
}

