/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.util;

import com.google.common.math.LongMath;
import com.google.common.primitives.Ints;
import java.io.IOException;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Comparator;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.RamUsageEstimator;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.hash.MurmurHash3;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.unit.SizeValue;

public class BloomFilter {
    final BitArray bits;
    final int numHashFunctions;
    final Hashing hashing;

    public static BloomFilter create(int expectedInsertions, double fpp) {
        return BloomFilter.create(expectedInsertions, fpp, -1);
    }

    public static BloomFilter create(int expectedInsertions, double fpp, int numHashFunctions) {
        if (expectedInsertions == 0) {
            expectedInsertions = 1;
        }
        long numBits = BloomFilter.optimalNumOfBits(expectedInsertions, fpp);
        if (numHashFunctions == -1) {
            numHashFunctions = BloomFilter.optimalNumOfHashFunctions(expectedInsertions, numBits);
        }
        try {
            return new BloomFilter(new BitArray(numBits), numHashFunctions, Hashing.DEFAULT);
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Could not create BloomFilter of " + numBits + " bits", e);
        }
    }

    public static void skipBloom(IndexInput in) throws IOException {
        int version = in.readInt();
        int numLongs = in.readInt();
        in.seek(in.getFilePointer() + (long)(numLongs * 8) + 4L + 4L);
    }

    public static BloomFilter deserialize(DataInput in) throws IOException {
        int version = in.readInt();
        int numLongs = in.readInt();
        long[] data = new long[numLongs];
        for (int i = 0; i < numLongs; ++i) {
            data[i] = in.readLong();
        }
        int numberOfHashFunctions = in.readInt();
        int hashType = in.readInt();
        return new BloomFilter(new BitArray(data), numberOfHashFunctions, Hashing.fromType(hashType));
    }

    public static void serilaize(BloomFilter filter, DataOutput out) throws IOException {
        out.writeInt(0);
        BitArray bits = filter.bits;
        out.writeInt(bits.data.length);
        for (long l : bits.data) {
            out.writeLong(l);
        }
        out.writeInt(filter.numHashFunctions);
        out.writeInt(filter.hashing.type());
    }

    public static BloomFilter readFrom(StreamInput in) throws IOException {
        int version = in.readVInt();
        int numLongs = in.readVInt();
        long[] data = new long[numLongs];
        for (int i = 0; i < numLongs; ++i) {
            data[i] = in.readLong();
        }
        int numberOfHashFunctions = in.readVInt();
        int hashType = in.readVInt();
        return new BloomFilter(new BitArray(data), numberOfHashFunctions, Hashing.fromType(hashType));
    }

    public static void writeTo(BloomFilter filter, StreamOutput out) throws IOException {
        out.writeVInt(0);
        BitArray bits = filter.bits;
        out.writeVInt(bits.data.length);
        for (long l : bits.data) {
            out.writeLong(l);
        }
        out.writeVInt(filter.numHashFunctions);
        out.writeVInt(filter.hashing.type());
    }

    BloomFilter(BitArray bits, int numHashFunctions, Hashing hashing) {
        this.bits = bits;
        this.numHashFunctions = numHashFunctions;
        this.hashing = hashing;
        if (numHashFunctions > 255) {
            throw new IllegalArgumentException("Currently we don't allow BloomFilters that would use more than 255 hash functions");
        }
    }

    public boolean put(BytesRef value) {
        return this.hashing.put(value, this.numHashFunctions, this.bits);
    }

    public boolean mightContain(BytesRef value) {
        return this.hashing.mightContain(value, this.numHashFunctions, this.bits);
    }

    public int getNumHashFunctions() {
        return this.numHashFunctions;
    }

    public long getSizeInBytes() {
        return this.bits.ramBytesUsed();
    }

    public int hashCode() {
        return this.bits.hashCode() + this.numHashFunctions;
    }

    static int optimalNumOfHashFunctions(long n, long m) {
        return Math.max(1, (int)Math.round((double)(m / n) * Math.log(2.0)));
    }

    static long optimalNumOfBits(long n, double p) {
        if (p == 0.0) {
            p = Double.MIN_VALUE;
        }
        return (long)((double)(-n) * Math.log(p) / (Math.log(2.0) * Math.log(2.0)));
    }

    protected static long getblock(byte[] key, int offset, int index) {
        int i_8 = index << 3;
        int blockOffset = offset + i_8;
        return ((long)key[blockOffset + 0] & 0xFFL) + (((long)key[blockOffset + 1] & 0xFFL) << 8) + (((long)key[blockOffset + 2] & 0xFFL) << 16) + (((long)key[blockOffset + 3] & 0xFFL) << 24) + (((long)key[blockOffset + 4] & 0xFFL) << 32) + (((long)key[blockOffset + 5] & 0xFFL) << 40) + (((long)key[blockOffset + 6] & 0xFFL) << 48) + (((long)key[blockOffset + 7] & 0xFFL) << 56);
    }

    protected static long rotl64(long v, int n) {
        return v << n | v >>> 64 - n;
    }

    protected static long fmix(long k) {
        k ^= k >>> 33;
        k *= -49064778989728563L;
        k ^= k >>> 33;
        k *= -4265267296055464877L;
        k ^= k >>> 33;
        return k;
    }

    public static long hash3_x64_128(byte[] key, int offset, int length, long seed) {
        int nblocks = length >> 4;
        long h1 = seed;
        long h2 = seed;
        long c1 = -8663945395140668459L;
        long c2 = 5545529020109919103L;
        for (int i = 0; i < nblocks; ++i) {
            long k1 = BloomFilter.getblock(key, offset, i * 2 + 0);
            long k2 = BloomFilter.getblock(key, offset, i * 2 + 1);
            k1 *= c1;
            k1 = BloomFilter.rotl64(k1, 31);
            h1 ^= (k1 *= c2);
            h1 = BloomFilter.rotl64(h1, 27);
            h1 += h2;
            h1 = h1 * 5L + 1390208809L;
            k2 *= c2;
            k2 = BloomFilter.rotl64(k2, 33);
            h2 ^= (k2 *= c1);
            h2 = BloomFilter.rotl64(h2, 31);
            h2 += h1;
            h2 = h2 * 5L + 944331445L;
        }
        offset += nblocks * 16;
        long k1 = 0L;
        long k2 = 0L;
        switch (length & 0xF) {
            case 15: {
                k2 ^= (long)key[offset + 14] << 48;
            }
            case 14: {
                k2 ^= (long)key[offset + 13] << 40;
            }
            case 13: {
                k2 ^= (long)key[offset + 12] << 32;
            }
            case 12: {
                k2 ^= (long)key[offset + 11] << 24;
            }
            case 11: {
                k2 ^= (long)key[offset + 10] << 16;
            }
            case 10: {
                k2 ^= (long)key[offset + 9] << 8;
            }
            case 9: {
                k2 ^= (long)key[offset + 8] << 0;
                k2 *= c2;
                k2 = BloomFilter.rotl64(k2, 33);
                h2 ^= (k2 *= c1);
            }
            case 8: {
                k1 ^= (long)key[offset + 7] << 56;
            }
            case 7: {
                k1 ^= (long)key[offset + 6] << 48;
            }
            case 6: {
                k1 ^= (long)key[offset + 5] << 40;
            }
            case 5: {
                k1 ^= (long)key[offset + 4] << 32;
            }
            case 4: {
                k1 ^= (long)key[offset + 3] << 24;
            }
            case 3: {
                k1 ^= (long)key[offset + 2] << 16;
            }
            case 2: {
                k1 ^= (long)key[offset + 1] << 8;
            }
            case 1: {
                k1 ^= (long)key[offset];
                k1 *= c1;
                k1 = BloomFilter.rotl64(k1, 31);
                h1 ^= (k1 *= c2);
            }
        }
        h1 ^= (long)length;
        h1 += (h2 ^= (long)length);
        h2 += h1;
        h1 = BloomFilter.fmix(h1);
        h2 = BloomFilter.fmix(h2);
        h1 += h2;
        h2 += h1;
        return h1;
    }

    static enum Hashing {
        V0{

            @Override
            protected boolean put(BytesRef value, int numHashFunctions, BitArray bits) {
                long bitSize = bits.bitSize();
                long hash64 = BloomFilter.hash3_x64_128(value.bytes, value.offset, value.length, 0L);
                int hash1 = (int)hash64;
                int hash2 = (int)(hash64 >>> 32);
                boolean bitsChanged = false;
                for (int i = 1; i <= numHashFunctions; ++i) {
                    int nextHash = hash1 + i * hash2;
                    if (nextHash < 0) {
                        nextHash ^= 0xFFFFFFFF;
                    }
                    bitsChanged |= bits.set((long)nextHash % bitSize);
                }
                return bitsChanged;
            }

            @Override
            protected boolean mightContain(BytesRef value, int numHashFunctions, BitArray bits) {
                long bitSize = bits.bitSize();
                long hash64 = BloomFilter.hash3_x64_128(value.bytes, value.offset, value.length, 0L);
                int hash1 = (int)hash64;
                int hash2 = (int)(hash64 >>> 32);
                for (int i = 1; i <= numHashFunctions; ++i) {
                    int nextHash = hash1 + i * hash2;
                    if (nextHash < 0) {
                        nextHash ^= 0xFFFFFFFF;
                    }
                    if (bits.get((long)nextHash % bitSize)) continue;
                    return false;
                }
                return true;
            }

            @Override
            protected int type() {
                return 0;
            }
        }
        ,
        V1{

            @Override
            protected boolean put(BytesRef value, int numHashFunctions, BitArray bits) {
                long bitSize = bits.bitSize();
                MurmurHash3.Hash128 hash128 = MurmurHash3.hash128(value.bytes, value.offset, value.length, 0L, new MurmurHash3.Hash128());
                boolean bitsChanged = false;
                long combinedHash = hash128.h1;
                for (int i = 0; i < numHashFunctions; ++i) {
                    bitsChanged |= bits.set((combinedHash & Long.MAX_VALUE) % bitSize);
                    combinedHash += hash128.h2;
                }
                return bitsChanged;
            }

            @Override
            protected boolean mightContain(BytesRef value, int numHashFunctions, BitArray bits) {
                long bitSize = bits.bitSize();
                MurmurHash3.Hash128 hash128 = MurmurHash3.hash128(value.bytes, value.offset, value.length, 0L, new MurmurHash3.Hash128());
                long combinedHash = hash128.h1;
                for (int i = 0; i < numHashFunctions; ++i) {
                    if (!bits.get((combinedHash & Long.MAX_VALUE) % bitSize)) {
                        return false;
                    }
                    combinedHash += hash128.h2;
                }
                return true;
            }

            @Override
            protected int type() {
                return 1;
            }
        };

        public static final Hashing DEFAULT;

        protected abstract boolean put(BytesRef var1, int var2, BitArray var3);

        protected abstract boolean mightContain(BytesRef var1, int var2, BitArray var3);

        protected abstract int type();

        public static Hashing fromType(int type) {
            if (type == 0) {
                return V0;
            }
            if (type == 1) {
                return V1;
            }
            throw new IllegalArgumentException("no hashing type matching " + type);
        }

        static {
            DEFAULT = V1;
        }
    }

    static final class BitArray {
        final long[] data;
        final long bitSize;
        long bitCount;

        BitArray(long bits) {
            this(new long[Ints.checkedCast(LongMath.divide(bits, 64L, RoundingMode.CEILING))]);
        }

        BitArray(long[] data) {
            this.data = data;
            long bitCount = 0L;
            for (long value : data) {
                bitCount += (long)Long.bitCount(value);
            }
            this.bitCount = bitCount;
            this.bitSize = data.length * 64;
        }

        boolean set(long index) {
            if (!this.get(index)) {
                int n = (int)(index >>> 6);
                this.data[n] = this.data[n] | 1L << (int)index;
                ++this.bitCount;
                return true;
            }
            return false;
        }

        boolean get(long index) {
            return (this.data[(int)(index >>> 6)] & 1L << (int)index) != 0L;
        }

        long bitSize() {
            return this.bitSize;
        }

        long bitCount() {
            return this.bitCount;
        }

        BitArray copy() {
            return new BitArray((long[])this.data.clone());
        }

        void putAll(BitArray array) {
            this.bitCount = 0L;
            for (int i = 0; i < this.data.length; ++i) {
                int n = i;
                this.data[n] = this.data[n] | array.data[i];
                this.bitCount += (long)Long.bitCount(this.data[i]);
            }
        }

        public boolean equals(Object o) {
            if (o instanceof BitArray) {
                BitArray bitArray = (BitArray)o;
                return Arrays.equals(this.data, bitArray.data);
            }
            return false;
        }

        public int hashCode() {
            return Arrays.hashCode(this.data);
        }

        public long ramBytesUsed() {
            return 8 * this.data.length + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER + 16;
        }
    }

    public static class Factory {
        public static final Factory DEFAULT = Factory.buildDefault();
        private final Entry[] entries;

        private static Factory buildDefault() {
            return Factory.buildFromString("10k=0.01,1m=0.03");
        }

        public static Factory buildFromString(@Nullable String config) {
            if (config == null) {
                return Factory.buildDefault();
            }
            String[] sEntries = Strings.splitStringToArray(config, ',');
            if (sEntries.length == 0) {
                if (config.length() > 0) {
                    return new Factory(new Entry[]{new Entry(0, Double.parseDouble(config))});
                }
                return Factory.buildDefault();
            }
            Entry[] entries = new Entry[sEntries.length];
            for (int i = 0; i < sEntries.length; ++i) {
                int index = sEntries[i].indexOf(61);
                entries[i] = new Entry((int)SizeValue.parseSizeValue(sEntries[i].substring(0, index).trim()).singles(), Double.parseDouble(sEntries[i].substring(index + 1).trim()));
            }
            return new Factory(entries);
        }

        public Factory(Entry[] entries) {
            this.entries = entries;
            Arrays.sort(this.entries, new Comparator<Entry>(){

                @Override
                public int compare(Entry o1, Entry o2) {
                    return o2.expectedInsertions - o1.expectedInsertions;
                }
            });
        }

        public BloomFilter createFilter(int expectedInsertions) {
            for (Entry entry : this.entries) {
                if (expectedInsertions <= entry.expectedInsertions) continue;
                return BloomFilter.create(expectedInsertions, entry.fpp);
            }
            return BloomFilter.create(expectedInsertions, 0.03);
        }

        public static class Entry {
            public final int expectedInsertions;
            public final double fpp;

            Entry(int expectedInsertions, double fpp) {
                this.expectedInsertions = expectedInsertions;
                this.fpp = fpp;
            }
        }
    }
}

