/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.annotations.Beta;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableSet;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.io.ByteProcessor;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import com.google.common.io.CharSink;
import com.google.common.io.CharSource;
import com.google.common.io.CharStreams;
import com.google.common.io.Closer;
import com.google.common.io.FileWriteMode;
import com.google.common.io.InputSupplier;
import com.google.common.io.LineProcessor;
import com.google.common.io.OutputSupplier;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.zip.Checksum;

@Beta
public final class Files {
    private static final int TEMP_DIR_ATTEMPTS = 10000;

    private Files() {
    }

    public static BufferedReader newReader(File file, Charset charset) throws FileNotFoundException {
        Preconditions.checkNotNull(file);
        Preconditions.checkNotNull(charset);
        return new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(file), charset));
    }

    public static BufferedWriter newWriter(File file, Charset charset) throws FileNotFoundException {
        Preconditions.checkNotNull(file);
        Preconditions.checkNotNull(charset);
        return new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file), charset));
    }

    public static ByteSource asByteSource(File file) {
        return new FileByteSource(file);
    }

    public static ByteSink asByteSink(File file, FileWriteMode ... modes) {
        return new FileByteSink(file, modes);
    }

    public static CharSource asCharSource(File file, Charset charset) {
        return Files.asByteSource(file).asCharSource(charset);
    }

    public static CharSink asCharSink(File file, Charset charset, FileWriteMode ... modes) {
        return Files.asByteSink(file, modes).asCharSink(charset);
    }

    public static InputSupplier<FileInputStream> newInputStreamSupplier(File file) {
        return ByteStreams.asInputSupplier(Files.asByteSource(file));
    }

    public static OutputSupplier<FileOutputStream> newOutputStreamSupplier(File file) {
        return Files.newOutputStreamSupplier(file, false);
    }

    public static OutputSupplier<FileOutputStream> newOutputStreamSupplier(File file, boolean append2) {
        return ByteStreams.asOutputSupplier(Files.asByteSink(file, Files.modes(append2)));
    }

    private static FileWriteMode[] modes(boolean append2) {
        FileWriteMode[] fileWriteModeArray;
        if (append2) {
            FileWriteMode[] fileWriteModeArray2 = new FileWriteMode[1];
            fileWriteModeArray = fileWriteModeArray2;
            fileWriteModeArray2[0] = FileWriteMode.APPEND;
        } else {
            fileWriteModeArray = new FileWriteMode[]{};
        }
        return fileWriteModeArray;
    }

    public static InputSupplier<InputStreamReader> newReaderSupplier(File file, Charset charset) {
        return CharStreams.asInputSupplier(Files.asCharSource(file, charset));
    }

    public static OutputSupplier<OutputStreamWriter> newWriterSupplier(File file, Charset charset) {
        return Files.newWriterSupplier(file, charset, false);
    }

    public static OutputSupplier<OutputStreamWriter> newWriterSupplier(File file, Charset charset, boolean append2) {
        return CharStreams.asOutputSupplier(Files.asCharSink(file, charset, Files.modes(append2)));
    }

    public static byte[] toByteArray(File file) throws IOException {
        return Files.asByteSource(file).read();
    }

    public static String toString(File file, Charset charset) throws IOException {
        return Files.asCharSource(file, charset).read();
    }

    public static void copy(InputSupplier<? extends InputStream> from2, File to2) throws IOException {
        ByteStreams.asByteSource(from2).copyTo(Files.asByteSink(to2, new FileWriteMode[0]));
    }

    public static void write(byte[] from2, File to2) throws IOException {
        Files.asByteSink(to2, new FileWriteMode[0]).write(from2);
    }

    public static void copy(File from2, OutputSupplier<? extends OutputStream> to2) throws IOException {
        Files.asByteSource(from2).copyTo(ByteStreams.asByteSink(to2));
    }

    public static void copy(File from2, OutputStream to2) throws IOException {
        Files.asByteSource(from2).copyTo(to2);
    }

    public static void copy(File from2, File to2) throws IOException {
        Preconditions.checkArgument(!from2.equals(to2), "Source %s and destination %s must be different", from2, to2);
        Files.asByteSource(from2).copyTo(Files.asByteSink(to2, new FileWriteMode[0]));
    }

    public static <R extends Readable & Closeable> void copy(InputSupplier<R> from2, File to2, Charset charset) throws IOException {
        CharStreams.asCharSource(from2).copyTo(Files.asCharSink(to2, charset, new FileWriteMode[0]));
    }

    public static void write(CharSequence from2, File to2, Charset charset) throws IOException {
        Files.asCharSink(to2, charset, new FileWriteMode[0]).write(from2);
    }

    public static void append(CharSequence from2, File to2, Charset charset) throws IOException {
        Files.write(from2, to2, charset, true);
    }

    private static void write(CharSequence from2, File to2, Charset charset, boolean append2) throws IOException {
        Files.asCharSink(to2, charset, Files.modes(append2)).write(from2);
    }

    public static <W extends Appendable & Closeable> void copy(File from2, Charset charset, OutputSupplier<W> to2) throws IOException {
        Files.asCharSource(from2, charset).copyTo(CharStreams.asCharSink(to2));
    }

    public static void copy(File from2, Charset charset, Appendable to2) throws IOException {
        Files.asCharSource(from2, charset).copyTo(to2);
    }

    public static boolean equal(File file1, File file2) throws IOException {
        Preconditions.checkNotNull(file1);
        Preconditions.checkNotNull(file2);
        if (file1 == file2 || file1.equals(file2)) {
            return true;
        }
        long len1 = file1.length();
        long len2 = file2.length();
        if (len1 != 0L && len2 != 0L && len1 != len2) {
            return false;
        }
        return Files.asByteSource(file1).contentEquals(Files.asByteSource(file2));
    }

    public static File createTempDir() {
        File baseDir = new File(System.getProperty("java.io.tmpdir"));
        String baseName = System.currentTimeMillis() + "-";
        for (int counter2 = 0; counter2 < 10000; ++counter2) {
            File tempDir = new File(baseDir, baseName + counter2);
            if (!tempDir.mkdir()) continue;
            return tempDir;
        }
        throw new IllegalStateException("Failed to create directory within 10000 attempts (tried " + baseName + "0 to " + baseName + 9999 + ')');
    }

    public static void touch(File file) throws IOException {
        Preconditions.checkNotNull(file);
        if (!file.createNewFile() && !file.setLastModified(System.currentTimeMillis())) {
            throw new IOException("Unable to update modification time of " + file);
        }
    }

    public static void createParentDirs(File file) throws IOException {
        Preconditions.checkNotNull(file);
        File parent = file.getCanonicalFile().getParentFile();
        if (parent == null) {
            return;
        }
        parent.mkdirs();
        if (!parent.isDirectory()) {
            throw new IOException("Unable to create parent directories of " + file);
        }
    }

    public static void move(File from2, File to2) throws IOException {
        Preconditions.checkNotNull(from2);
        Preconditions.checkNotNull(to2);
        Preconditions.checkArgument(!from2.equals(to2), "Source %s and destination %s must be different", from2, to2);
        if (!from2.renameTo(to2)) {
            Files.copy(from2, to2);
            if (!from2.delete()) {
                if (!to2.delete()) {
                    throw new IOException("Unable to delete " + to2);
                }
                throw new IOException("Unable to delete " + from2);
            }
        }
    }

    public static String readFirstLine(File file, Charset charset) throws IOException {
        return Files.asCharSource(file, charset).readFirstLine();
    }

    public static List<String> readLines(File file, Charset charset) throws IOException {
        return CharStreams.readLines(Files.newReaderSupplier(file, charset));
    }

    public static <T> T readLines(File file, Charset charset, LineProcessor<T> callback) throws IOException {
        return CharStreams.readLines(Files.newReaderSupplier(file, charset), callback);
    }

    public static <T> T readBytes(File file, ByteProcessor<T> processor) throws IOException {
        return ByteStreams.readBytes(Files.newInputStreamSupplier(file), processor);
    }

    @Deprecated
    public static long getChecksum(File file, Checksum checksum) throws IOException {
        return ByteStreams.getChecksum(Files.newInputStreamSupplier(file), checksum);
    }

    public static HashCode hash(File file, HashFunction hashFunction) throws IOException {
        return Files.asByteSource(file).hash(hashFunction);
    }

    public static MappedByteBuffer map(File file) throws IOException {
        Preconditions.checkNotNull(file);
        return Files.map(file, FileChannel.MapMode.READ_ONLY);
    }

    public static MappedByteBuffer map(File file, FileChannel.MapMode mode2) throws IOException {
        Preconditions.checkNotNull(file);
        Preconditions.checkNotNull(mode2);
        if (!file.exists()) {
            throw new FileNotFoundException(file.toString());
        }
        return Files.map(file, mode2, file.length());
    }

    public static MappedByteBuffer map(File file, FileChannel.MapMode mode2, long size2) throws FileNotFoundException, IOException {
        Preconditions.checkNotNull(file);
        Preconditions.checkNotNull(mode2);
        Closer closer = Closer.create();
        try {
            RandomAccessFile raf = closer.register(new RandomAccessFile(file, mode2 == FileChannel.MapMode.READ_ONLY ? "r" : "rw"));
            MappedByteBuffer mappedByteBuffer = Files.map(raf, mode2, size2);
            return mappedByteBuffer;
        }
        catch (Throwable e) {
            throw closer.rethrow(e);
        }
        finally {
            closer.close();
        }
    }

    private static MappedByteBuffer map(RandomAccessFile raf, FileChannel.MapMode mode2, long size2) throws IOException {
        Closer closer = Closer.create();
        try {
            FileChannel channel = closer.register(raf.getChannel());
            MappedByteBuffer mappedByteBuffer = channel.map(mode2, 0L, size2);
            return mappedByteBuffer;
        }
        catch (Throwable e) {
            throw closer.rethrow(e);
        }
        finally {
            closer.close();
        }
    }

    public static String simplifyPath(String pathname) {
        Preconditions.checkNotNull(pathname);
        if (pathname.length() == 0) {
            return ".";
        }
        Iterable<String> components = Splitter.on('/').omitEmptyStrings().split(pathname);
        ArrayList<String> path = new ArrayList<String>();
        for (String component : components) {
            if (component.equals(".")) continue;
            if (component.equals("..")) {
                if (path.size() > 0 && !((String)path.get(path.size() - 1)).equals("..")) {
                    path.remove(path.size() - 1);
                    continue;
                }
                path.add("..");
                continue;
            }
            path.add(component);
        }
        String result2 = Joiner.on('/').join(path);
        if (pathname.charAt(0) == '/') {
            result2 = "/" + result2;
        }
        while (result2.startsWith("/../")) {
            result2 = result2.substring(3);
        }
        if (result2.equals("/..")) {
            result2 = "/";
        } else if ("".equals(result2)) {
            result2 = ".";
        }
        return result2;
    }

    public static String getFileExtension(String fullName) {
        Preconditions.checkNotNull(fullName);
        String fileName = new File(fullName).getName();
        int dotIndex = fileName.lastIndexOf(46);
        return dotIndex == -1 ? "" : fileName.substring(dotIndex + 1);
    }

    public static String getNameWithoutExtension(String file) {
        Preconditions.checkNotNull(file);
        String fileName = new File(file).getName();
        int dotIndex = fileName.lastIndexOf(46);
        return dotIndex == -1 ? fileName : fileName.substring(0, dotIndex);
    }

    private static final class FileByteSink
    extends ByteSink {
        private final File file;
        private final ImmutableSet<FileWriteMode> modes;

        private FileByteSink(File file, FileWriteMode ... modes) {
            this.file = Preconditions.checkNotNull(file);
            this.modes = ImmutableSet.copyOf(modes);
        }

        @Override
        public FileOutputStream openStream() throws IOException {
            return new FileOutputStream(this.file, this.modes.contains((Object)FileWriteMode.APPEND));
        }

        public String toString() {
            return "Files.asByteSink(" + this.file + ", " + this.modes + ")";
        }
    }

    private static final class FileByteSource
    extends ByteSource {
        private final File file;

        private FileByteSource(File file) {
            this.file = Preconditions.checkNotNull(file);
        }

        @Override
        public FileInputStream openStream() throws IOException {
            return new FileInputStream(this.file);
        }

        @Override
        public long size() throws IOException {
            if (!this.file.isFile()) {
                throw new FileNotFoundException(this.file.toString());
            }
            return this.file.length();
        }

        @Override
        public byte[] read() throws IOException {
            long size2 = this.file.length();
            if (size2 == 0L) {
                return super.read();
            }
            if (size2 > Integer.MAX_VALUE) {
                throw new OutOfMemoryError("file is too large to fit in a byte array: " + size2 + " bytes");
            }
            byte[] bytes2 = new byte[(int)size2];
            Closer closer = Closer.create();
            try {
                InputStream in = closer.register(this.openStream());
                int off = 0;
                int read2 = 0;
                while ((long)off < size2 && (read2 = in.read(bytes2, off, (int)size2 - off)) != -1) {
                    off += read2;
                }
                byte[] result2 = bytes2;
                if ((long)off < size2) {
                    result2 = Arrays.copyOf(bytes2, off);
                } else if (read2 != -1) {
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    ByteStreams.copy(in, (OutputStream)out);
                    byte[] moreBytes = out.toByteArray();
                    result2 = new byte[bytes2.length + moreBytes.length];
                    System.arraycopy(bytes2, 0, result2, 0, bytes2.length);
                    System.arraycopy(moreBytes, 0, result2, bytes2.length, moreBytes.length);
                }
                byte[] byArray = result2;
                return byArray;
            }
            catch (Throwable e) {
                throw closer.rethrow(e);
            }
            finally {
                closer.close();
            }
        }

        public String toString() {
            return "Files.asByteSource(" + this.file + ")";
        }
    }
}

