/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.gson.JsonParseException;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.OldNoteInfo;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.notebook.repo.OldNotebookRepo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OldGCSNotebookRepo
implements OldNotebookRepo {
    private static final Logger LOG = LoggerFactory.getLogger(OldGCSNotebookRepo.class);
    private String encoding;
    private String bucketName;
    private Optional<String> basePath;
    private Pattern noteNamePattern;
    private Storage storage;

    public OldGCSNotebookRepo() {
    }

    @VisibleForTesting
    public OldGCSNotebookRepo(ZeppelinConfiguration zConf, Storage storage) throws IOException {
        this.init(zConf);
        this.storage = storage;
    }

    public void init(ZeppelinConfiguration zConf) throws IOException {
        this.encoding = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_ENCODING);
        String gcsStorageDir = zConf.getGCSStorageDir();
        if (gcsStorageDir.isEmpty()) {
            throw new IOException("GCS storage directory must be set using 'zeppelin.notebook.gcs.dir'");
        }
        if (!gcsStorageDir.startsWith("gs://")) {
            throw new IOException(String.format("GCS storage directory '%s' must start with 'gs://'.", gcsStorageDir));
        }
        String storageDirWithoutScheme = gcsStorageDir.substring("gs://".length());
        List<String> pathComponents = Arrays.asList(storageDirWithoutScheme.split("/"));
        if (pathComponents.isEmpty()) {
            throw new IOException(String.format("GCS storage directory '%s' must be in the form gs://bucketname/path/to/dir", gcsStorageDir));
        }
        this.bucketName = pathComponents.get(0);
        this.basePath = pathComponents.size() > 1 ? Optional.of(StringUtils.join(pathComponents.subList(1, pathComponents.size()), (String)"/")) : Optional.empty();
        this.noteNamePattern = this.basePath.isPresent() ? Pattern.compile("^" + Pattern.quote(this.basePath.get() + "/") + "([^/]+)/note\\.json$") : Pattern.compile("^([^/]+)/note\\.json$");
        GoogleCredentials credentials = GoogleCredentials.getApplicationDefault();
        String credentialJsonPath = zConf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_GCS_CREDENTIALS_FILE);
        if (credentialJsonPath != null) {
            credentials = GoogleCredentials.fromStream((InputStream)new FileInputStream(credentialJsonPath));
        }
        this.storage = (Storage)((StorageOptions.Builder)StorageOptions.newBuilder().setCredentials((Credentials)credentials)).build().getService();
    }

    private BlobId makeBlobId(String noteId) {
        if (this.basePath.isPresent()) {
            return BlobId.of((String)this.bucketName, (String)(this.basePath.get() + "/" + noteId + "/note.json"));
        }
        return BlobId.of((String)this.bucketName, (String)(noteId + "/note.json"));
    }

    public List<OldNoteInfo> list(AuthenticationInfo subject) throws IOException {
        try {
            ArrayList<OldNoteInfo> infos = new ArrayList<OldNoteInfo>();
            Iterable blobsUnderDir = this.basePath.isPresent() ? this.storage.list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)(this.basePath.get() + "/"))}).iterateAll() : this.storage.list(this.bucketName, new Storage.BlobListOption[0]).iterateAll();
            for (Blob b : blobsUnderDir) {
                Matcher matcher = this.noteNamePattern.matcher(b.getName());
                if (!matcher.matches()) continue;
                infos.add(new OldNoteInfo(matcher.group(1), "", null));
            }
            return infos;
        }
        catch (StorageException se) {
            throw new IOException("Could not list GCS directory: " + se.getMessage(), se);
        }
    }

    public Note get(String noteId, AuthenticationInfo subject) throws IOException {
        byte[] contents;
        BlobId blobId = this.makeBlobId(noteId);
        try {
            contents = this.storage.readAllBytes(blobId, new Storage.BlobSourceOption[0]);
        }
        catch (StorageException se) {
            throw new IOException("Could not read " + blobId.toString() + ": " + se.getMessage(), se);
        }
        try {
            return Note.fromJson((String)noteId, (String)new String(contents, this.encoding));
        }
        catch (JsonParseException jpe) {
            throw new IOException("Could note parse as json " + blobId.toString() + jpe.getMessage(), jpe);
        }
    }

    public void save(Note note, AuthenticationInfo subject) throws IOException {
        BlobInfo info = BlobInfo.newBuilder((BlobId)this.makeBlobId(note.getId())).setContentType("application/json").build();
        try {
            this.storage.create(info, note.toJson().getBytes(StandardCharsets.UTF_8), new Storage.BlobTargetOption[0]);
        }
        catch (StorageException se) {
            throw new IOException("Could not write " + info.toString() + ": " + se.getMessage(), se);
        }
    }

    public void remove(String noteId, AuthenticationInfo subject) throws IOException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)noteId));
        BlobId blobId = this.makeBlobId(noteId);
        try {
            boolean deleted = this.storage.delete(blobId);
            if (!deleted) {
                throw new IOException("Tried to remove nonexistent blob " + blobId.toString());
            }
        }
        catch (StorageException se) {
            throw new IOException("Could not remove " + blobId.toString() + ": " + se.getMessage(), se);
        }
    }

    public void close() {
    }

    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        LOG.warn("getSettings is not implemented for GCSNotebookRepo");
        return Collections.emptyList();
    }

    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        LOG.warn("updateSettings is not implemented for GCSNotebookRepo");
    }
}

