/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.inject.Inject;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.OldNoteInfo;
import org.apache.zeppelin.notebook.Paragraph;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.notebook.repo.NotebookRepoWithSettings;
import org.apache.zeppelin.notebook.repo.NotebookRepoWithVersionControl;
import org.apache.zeppelin.notebook.repo.OldNotebookRepo;
import org.apache.zeppelin.plugin.PluginManager;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.apache.zeppelin.util.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NotebookRepoSync
implements NotebookRepoWithVersionControl {
    private static final Logger LOGGER = LoggerFactory.getLogger(NotebookRepoSync.class);
    private static final int MAX_REPO_NUM = 2;
    private static final String PUSH_KEY = "pushNoteIds";
    private static final String PULL_KEY = "pullNoteIds";
    private static final String DEL_DST_KEY = "delDstNoteIds";
    private static final String DEFAULT_STORAGE = "org.apache.zeppelin.notebook.repo.GitNotebookRepo";
    private List<NotebookRepo> repos = new ArrayList<NotebookRepo>();
    private boolean oneWaySync;

    @Inject
    public NotebookRepoSync(ZeppelinConfiguration conf) throws IOException {
        this.init(conf);
    }

    @Override
    public void init(ZeppelinConfiguration conf) throws IOException {
        String[] storageClassNames;
        this.oneWaySync = conf.getBoolean(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_ONE_WAY_SYNC);
        String allStorageClassNames = conf.getNotebookStorageClass().trim();
        if (allStorageClassNames.isEmpty()) {
            allStorageClassNames = DEFAULT_STORAGE;
            LOGGER.warn("Empty ZEPPELIN_NOTEBOOK_STORAGE conf parameter, using default {}", (Object)DEFAULT_STORAGE);
        }
        if ((storageClassNames = allStorageClassNames.split(",")).length > this.getMaxRepoNum()) {
            LOGGER.warn("Unsupported number {} of storage classes in ZEPPELIN_NOTEBOOK_STORAGE : {}\nfirst {} will be used", new Object[]{storageClassNames.length, allStorageClassNames, this.getMaxRepoNum()});
        }
        for (int i = 0; i < Math.min(storageClassNames.length, this.getMaxRepoNum()); ++i) {
            NotebookRepo notebookRepo = PluginManager.get().loadNotebookRepo(storageClassNames[i].trim());
            notebookRepo.init(conf);
            this.repos.add(notebookRepo);
        }
        if (this.getRepoCount() == 0) {
            LOGGER.info("No storage could be initialized, using default {} storage", (Object)DEFAULT_STORAGE);
            NotebookRepo defaultNotebookRepo = PluginManager.get().loadNotebookRepo(DEFAULT_STORAGE);
            defaultNotebookRepo.init(conf);
            this.repos.add(defaultNotebookRepo);
        }
        if (this.getRepoCount() > 1 && conf.isAnonymousAllowed()) {
            try {
                this.sync(AuthenticationInfo.ANONYMOUS);
            }
            catch (IOException e) {
                LOGGER.error("Couldn't sync anonymous mode on start ", (Throwable)e);
            }
        }
    }

    public void convertNoteFiles(ZeppelinConfiguration conf, boolean deleteOld) throws IOException {
        for (int i = 0; i < this.repos.size(); ++i) {
            NotebookRepo newNotebookRepo = this.repos.get(i);
            OldNotebookRepo oldNotebookRepo = PluginManager.get().loadOldNotebookRepo(newNotebookRepo.getClass().getCanonicalName());
            oldNotebookRepo.init(conf);
            List<OldNoteInfo> oldNotesInfo = oldNotebookRepo.list(AuthenticationInfo.ANONYMOUS);
            LOGGER.info("Convert old note file to new style, note count: {}", (Object)oldNotesInfo.size());
            LOGGER.info("Delete old note: {}", (Object)deleteOld);
            for (OldNoteInfo oldNoteInfo : oldNotesInfo) {
                LOGGER.info("Converting note, id: {}", (Object)oldNoteInfo.getId());
                Note note = oldNotebookRepo.get(oldNoteInfo.getId(), AuthenticationInfo.ANONYMOUS);
                note.setPath(note.getName());
                note.setVersion(Util.getVersion());
                newNotebookRepo.save(note, AuthenticationInfo.ANONYMOUS);
                if (newNotebookRepo instanceof NotebookRepoWithVersionControl) {
                    ((NotebookRepoWithVersionControl)newNotebookRepo).checkpoint(note.getId(), note.getPath(), "Upgrade note '" + note.getName() + "' to " + Util.getVersion(), AuthenticationInfo.ANONYMOUS);
                }
                if (!deleteOld) continue;
                oldNotebookRepo.remove(note.getId(), AuthenticationInfo.ANONYMOUS);
                LOGGER.info("Remote old note: {}", (Object)note.getId());
            }
        }
    }

    public List<NotebookRepoWithSettings> getNotebookRepos(AuthenticationInfo subject) {
        ArrayList reposSetting = Lists.newArrayList();
        for (NotebookRepo repo : this.repos) {
            NotebookRepoWithSettings repoWithSettings = NotebookRepoWithSettings.builder(repo.getClass().getSimpleName()).className(repo.getClass().getName()).settings(repo.getSettings(subject)).build();
            reposSetting.add(repoWithSettings);
        }
        return reposSetting;
    }

    public NotebookRepoWithSettings updateNotebookRepo(String name, Map<String, String> settings, AuthenticationInfo subject) {
        NotebookRepoWithSettings updatedSettings = NotebookRepoWithSettings.EMPTY;
        for (NotebookRepo repo : this.repos) {
            if (!repo.getClass().getName().equals(name)) continue;
            repo.updateSettings(settings, subject);
            updatedSettings = NotebookRepoWithSettings.builder(repo.getClass().getSimpleName()).className(repo.getClass().getName()).settings(repo.getSettings(subject)).build();
            break;
        }
        return updatedSettings;
    }

    @Override
    public Map<String, NoteInfo> list(AuthenticationInfo subject) throws IOException {
        return this.getRepo(0).list(subject);
    }

    List<NoteInfo> list(int repoIndex, AuthenticationInfo subject) throws IOException {
        return new ArrayList<NoteInfo>(this.getRepo(repoIndex).list(subject).values());
    }

    @Override
    public Note get(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        return this.getRepo(0).get(noteId, notePath, subject);
    }

    Note get(int repoIndex, String noteId, String noteName, AuthenticationInfo subject) throws IOException {
        return this.getRepo(repoIndex).get(noteId, noteName, subject);
    }

    @Override
    public void save(Note note, AuthenticationInfo subject) throws IOException {
        this.getRepo(0).save(note, subject);
        if (this.getRepoCount() > 1) {
            try {
                this.getRepo(1).save(note, subject);
            }
            catch (IOException e) {
                LOGGER.info("{}: Failed to write to secondary storage", (Object)e.getMessage());
            }
        }
    }

    void save(int repoIndex, Note note, AuthenticationInfo subject) throws IOException {
        this.getRepo(repoIndex).save(note, subject);
    }

    @Override
    public void move(String noteId, String notePath, String newNotePath, AuthenticationInfo subject) throws IOException {
        this.getRepo(0).move(noteId, notePath, newNotePath, subject);
        if (this.getRepoCount() > 1) {
            try {
                this.getRepo(1).move(noteId, notePath, newNotePath, subject);
            }
            catch (IOException e) {
                LOGGER.info("{}: Failed to write to secondary storage", (Object)e.getMessage());
            }
        }
    }

    @Override
    public void move(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        for (NotebookRepo repo : this.repos) {
            repo.move(folderPath, newFolderPath, subject);
        }
    }

    @Override
    public void remove(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        for (NotebookRepo repo : this.repos) {
            repo.remove(noteId, notePath, subject);
        }
    }

    @Override
    public void remove(String folderPath, AuthenticationInfo subject) throws IOException {
        for (NotebookRepo repo : this.repos) {
            repo.remove(folderPath, subject);
        }
    }

    void remove(int repoIndex, String noteId, String noteName, AuthenticationInfo subject) throws IOException {
        this.getRepo(repoIndex).remove(noteId, noteName, subject);
    }

    void sync(int sourceRepoIndex, int destRepoIndex, AuthenticationInfo subject) throws IOException {
        LOGGER.info("Sync started");
        NotebookRepo srcRepo = this.getRepo(sourceRepoIndex);
        NotebookRepo dstRepo = this.getRepo(destRepoIndex);
        ArrayList<NoteInfo> srcNotes = new ArrayList<NoteInfo>(srcRepo.list(subject).values());
        ArrayList<NoteInfo> dstNotes = new ArrayList<NoteInfo>(dstRepo.list(subject).values());
        Map<String, List<NoteInfo>> noteIds = this.notesCheckDiff(srcNotes, srcRepo, dstNotes, dstRepo, subject);
        List<NoteInfo> pushNoteIds = noteIds.get(PUSH_KEY);
        List<NoteInfo> pullNoteIds = noteIds.get(PULL_KEY);
        List<NoteInfo> delDstNoteIds = noteIds.get(DEL_DST_KEY);
        if (!pushNoteIds.isEmpty()) {
            LOGGER.info("The following notes will be pushed");
            for (NoteInfo noteInfo : pushNoteIds) {
                LOGGER.info("Note : {}", (Object)noteInfo.getId());
            }
            this.pushNotes(subject, pushNoteIds, srcRepo, dstRepo);
        } else {
            LOGGER.info("Nothing to push");
        }
        if (!pullNoteIds.isEmpty()) {
            LOGGER.info("The following notes will be pulled");
            for (NoteInfo noteInfo : pullNoteIds) {
                LOGGER.info("Note : {}", (Object)noteInfo);
            }
            this.pushNotes(subject, pullNoteIds, dstRepo, srcRepo);
        } else {
            LOGGER.info("Nothing to pull");
        }
        if (!delDstNoteIds.isEmpty()) {
            LOGGER.info("The following notes will be deleted from dest");
            for (NoteInfo noteInfo : delDstNoteIds) {
                LOGGER.info("Note : {}", (Object)noteInfo.getId());
            }
            this.deleteNotes(subject, delDstNoteIds, dstRepo);
        } else {
            LOGGER.info("Nothing to delete from dest");
        }
        LOGGER.info("Sync ended");
    }

    public void sync(AuthenticationInfo subject) throws IOException {
        this.sync(0, 1, subject);
    }

    private void pushNotes(AuthenticationInfo subject, List<NoteInfo> notesInfo, NotebookRepo localRepo, NotebookRepo remoteRepo) {
        for (NoteInfo noteInfo : notesInfo) {
            try {
                remoteRepo.save(localRepo.get(noteInfo.getId(), noteInfo.getPath(), subject), subject);
            }
            catch (IOException e) {
                LOGGER.error("Failed to push note to storage, moving onto next one", (Throwable)e);
            }
        }
    }

    private void deleteNotes(AuthenticationInfo subject, List<NoteInfo> noteInfos, NotebookRepo repo) throws IOException {
        for (NoteInfo noteInfo : noteInfos) {
            repo.remove(noteInfo.getId(), noteInfo.getPath(), subject);
        }
    }

    public int getRepoCount() {
        return this.repos.size();
    }

    int getMaxRepoNum() {
        return 2;
    }

    public NotebookRepo getRepo(int repoIndex) throws IOException {
        if (repoIndex < 0 || repoIndex >= this.getRepoCount()) {
            throw new IOException("Requested storage index " + repoIndex + " isn't initialized, repository count is " + this.getRepoCount());
        }
        return this.repos.get(repoIndex);
    }

    private Map<String, List<NoteInfo>> notesCheckDiff(List<NoteInfo> sourceNotes, NotebookRepo sourceRepo, List<NoteInfo> destNotes, NotebookRepo destRepo, AuthenticationInfo subject) {
        NoteInfo dnote;
        ArrayList<NoteInfo> pushIDs = new ArrayList<NoteInfo>();
        ArrayList<NoteInfo> pullIDs = new ArrayList<NoteInfo>();
        ArrayList<NoteInfo> delDstIDs = new ArrayList<NoteInfo>();
        for (NoteInfo snote : sourceNotes) {
            dnote = this.containsID(destNotes, snote.getId());
            if (dnote != null) {
                Date ddate;
                Date sdate;
                try {
                    sdate = this.lastModificationDate(sourceRepo.get(snote.getId(), snote.getPath(), subject));
                    ddate = this.lastModificationDate(destRepo.get(dnote.getId(), dnote.getPath(), subject));
                }
                catch (IOException e) {
                    LOGGER.error("Cannot access previously listed note {} from storage ", (Object)dnote.getId(), (Object)e);
                    continue;
                }
                if (sdate.compareTo(ddate) == 0) continue;
                if (sdate.after(ddate) || this.oneWaySync) {
                    pushIDs.add(snote);
                    LOGGER.info("Modified note is added to push list : {}", (Object)sdate);
                    continue;
                }
                LOGGER.info("Modified note is added to pull list : {}", (Object)ddate);
                pullIDs.add(snote);
                continue;
            }
            pushIDs.add(snote);
        }
        for (NoteInfo note : destNotes) {
            dnote = this.containsID(sourceNotes, note.getId());
            if (dnote != null) continue;
            if (this.oneWaySync) {
                LOGGER.info("Extraneous note is added to delete dest list : {}", (Object)note.getId());
                delDstIDs.add(note);
                continue;
            }
            LOGGER.info("Missing note is added to pull list : {}", (Object)note.getId());
            pullIDs.add(note);
        }
        HashMap<String, List<NoteInfo>> map = new HashMap<String, List<NoteInfo>>();
        map.put(PUSH_KEY, pushIDs);
        map.put(PULL_KEY, pullIDs);
        map.put(DEL_DST_KEY, delDstIDs);
        return map;
    }

    private NoteInfo containsID(List<NoteInfo> notes, String id) {
        for (NoteInfo note : notes) {
            if (!note.getId().equals(id)) continue;
            return note;
        }
        return null;
    }

    private Date lastModificationDate(Note note) {
        Date latest = new Date(0L);
        for (Paragraph paragraph : note.getParagraphs()) {
            Date tempCreated = paragraph.getDateCreated();
            Date tempStarted = paragraph.getDateStarted();
            Date tempFinished = paragraph.getDateFinished();
            if (tempCreated != null && tempCreated.after(latest)) {
                latest = tempCreated;
            }
            if (tempStarted != null && tempStarted.after(latest)) {
                latest = tempStarted;
            }
            if (tempFinished == null || !tempFinished.after(latest)) continue;
            latest = tempFinished;
        }
        return latest;
    }

    @Override
    public void close() {
        LOGGER.info("Closing all notebook storages");
        for (NotebookRepo repo : this.repos) {
            repo.close();
        }
    }

    public boolean isRevisionSupportedInDefaultRepo() {
        return this.isRevisionSupportedInRepo(0);
    }

    public boolean isRevisionSupportedInRepo(int repoIndex) {
        try {
            if (this.getRepo(repoIndex) instanceof NotebookRepoWithVersionControl) {
                return true;
            }
        }
        catch (IOException e) {
            LOGGER.error("Error getting default repo", (Throwable)e);
        }
        return false;
    }

    @Override
    public NotebookRepoWithVersionControl.Revision checkpoint(String noteId, String notePath, String checkpointMsg, AuthenticationInfo subject) throws IOException {
        int repoCount = this.getRepoCount();
        int repoBound = Math.min(repoCount, this.getMaxRepoNum());
        int errorCount = 0;
        StringBuilder errorMessage = new StringBuilder("");
        ArrayList<NotebookRepoWithVersionControl.Revision> allRepoCheckpoints = new ArrayList<NotebookRepoWithVersionControl.Revision>();
        NotebookRepoWithVersionControl.Revision rev = null;
        for (int i = 0; i < repoBound; ++i) {
            try {
                if (!this.isRevisionSupportedInRepo(i)) continue;
                allRepoCheckpoints.add(((NotebookRepoWithVersionControl)this.getRepo(i)).checkpoint(noteId, notePath, checkpointMsg, subject));
                continue;
            }
            catch (IOException e) {
                LOGGER.warn("Couldn't checkpoint in {} storage with index {} for note {}", new Object[]{this.getRepo(i).getClass(), i, noteId});
                errorMessage.append("Error on storage class " + this.getRepo(i).getClass().toString() + " with index " + i + " : " + e.getMessage() + "\n");
                ++errorCount;
            }
        }
        if (errorCount == repoBound) {
            throw new IOException(errorMessage.toString());
        }
        if (!allRepoCheckpoints.isEmpty()) {
            rev = (NotebookRepoWithVersionControl.Revision)allRepoCheckpoints.get(0);
            if (allRepoCheckpoints.size() > 1 && rev == null) {
                rev = (NotebookRepoWithVersionControl.Revision)allRepoCheckpoints.get(1);
            }
        }
        return rev;
    }

    @Override
    public Note get(String noteId, String notePath, String revId, AuthenticationInfo subject) {
        Note revisionNote = null;
        try {
            if (this.isRevisionSupportedInDefaultRepo()) {
                revisionNote = ((NotebookRepoWithVersionControl)this.getRepo(0)).get(noteId, notePath, revId, subject);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to get revision {} of note {}", new Object[]{revId, noteId, e});
        }
        return revisionNote;
    }

    @Override
    public List<NotebookRepoWithVersionControl.Revision> revisionHistory(String noteId, String notePath, AuthenticationInfo subject) {
        List<NotebookRepoWithVersionControl.Revision> revisions = Collections.emptyList();
        try {
            if (this.isRevisionSupportedInDefaultRepo()) {
                revisions = ((NotebookRepoWithVersionControl)this.getRepo(0)).revisionHistory(noteId, notePath, subject);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to list revision history", (Throwable)e);
        }
        return revisions;
    }

    @Override
    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        List<NotebookRepoSettingsInfo> repoSettings = Collections.emptyList();
        try {
            repoSettings = this.getRepo(0).getSettings(subject);
        }
        catch (IOException e) {
            LOGGER.error("Cannot get notebook repo settings", (Throwable)e);
        }
        return repoSettings;
    }

    @Override
    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        try {
            this.getRepo(0).updateSettings(settings, subject);
        }
        catch (IOException e) {
            LOGGER.error("Cannot update notebook repo settings", (Throwable)e);
        }
    }

    @Override
    public Note setNoteRevision(String noteId, String notePath, String revId, AuthenticationInfo subject) throws IOException {
        int repoCount = this.getRepoCount();
        int repoBound = Math.min(repoCount, this.getMaxRepoNum());
        Note currentNote = null;
        Note revisionNote = null;
        for (int i = 0; i < repoBound; ++i) {
            try {
                if (this.isRevisionSupportedInRepo(i)) {
                    currentNote = ((NotebookRepoWithVersionControl)this.getRepo(i)).setNoteRevision(noteId, notePath, revId, subject);
                }
            }
            catch (IOException e) {
                currentNote = null;
            }
            if (currentNote == null || revisionNote != null) continue;
            revisionNote = currentNote;
        }
        return revisionNote;
    }
}

