/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sedona.common.geography;

import com.google.common.geometry.S2Builder;
import com.google.common.geometry.S2BuilderLayer;
import com.google.common.geometry.S2Error;
import com.google.common.geometry.S2LatLng;
import com.google.common.geometry.S2Loop;
import com.google.common.geometry.S2Point;
import com.google.common.geometry.S2PointVectorLayer;
import com.google.common.geometry.S2Polygon;
import com.google.common.geometry.S2PolygonLayer;
import com.google.common.geometry.S2Polyline;
import com.google.common.geometry.S2PolylineLayer;
import com.google.common.geometry.S2PolylineVectorLayer;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.sedona.common.S2Geography.Geography;
import org.apache.sedona.common.S2Geography.GeographyCollection;
import org.apache.sedona.common.S2Geography.MultiPolygonGeography;
import org.apache.sedona.common.S2Geography.PointGeography;
import org.apache.sedona.common.S2Geography.PolygonGeography;
import org.apache.sedona.common.S2Geography.PolylineGeography;
import org.apache.sedona.common.S2Geography.SinglePointGeography;
import org.apache.sedona.common.S2Geography.SinglePolylineGeography;
import org.apache.sedona.common.S2Geography.WKBReader;
import org.apache.sedona.common.S2Geography.WKTReader;
import org.apache.sedona.common.utils.GeoHashDecoder;
import org.locationtech.jts.algorithm.Orientation;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateArrays;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.io.ParseException;

public class Constructors {
    public static Geography geogFromWKB(byte[] wkb) throws ParseException {
        return new WKBReader().read(wkb);
    }

    public static Geography geogFromWKB(byte[] wkb, int SRID) throws ParseException {
        Geography geog = Constructors.geogFromWKB(wkb);
        geog.setSRID(SRID);
        return geog;
    }

    public static Geography geogFromWKT(String wkt, int srid) throws ParseException {
        Geography geog = new WKTReader().read(wkt);
        geog.setSRID(srid);
        return geog;
    }

    public static Geography geogFromEWKT(String ewkt) throws ParseException {
        if (ewkt == null) {
            return null;
        }
        int SRID = 0;
        String wkt = ewkt;
        int index = ewkt.indexOf("SRID=");
        if (index != -1) {
            int semicolonIndex = ewkt.indexOf(59, index);
            if (semicolonIndex != -1) {
                SRID = Integer.parseInt(ewkt.substring(index + 5, semicolonIndex));
                wkt = ewkt.substring(semicolonIndex + 1);
            } else {
                throw new ParseException("Invalid EWKT string");
            }
        }
        return Constructors.geogFromWKT(wkt, SRID);
    }

    public static Geography geogCollFromText(String wkt, int srid) throws ParseException {
        if (wkt == null || !wkt.startsWith("GEOMETRYCOLLECTION")) {
            return null;
        }
        return Constructors.geogFromWKT(wkt, srid);
    }

    public static Geography geogFromGeoHash(String geoHash, Integer precision) {
        try {
            return GeoHashDecoder.decodeGeog(geoHash, precision);
        }
        catch (GeoHashDecoder.InvalidGeoHashException e) {
            return null;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Geometry geogToGeometry(Geography geography) {
        GeometryFactory geometryFactory = new GeometryFactory(new PrecisionModel(), geography.getSRID());
        return Constructors.geogToGeometry(geography, geometryFactory);
    }

    public static Geometry geogToGeometry(Geography geography, GeometryFactory geometryFactory) {
        if (geography == null) {
            return null;
        }
        Geography.GeographyKind kind = Geography.GeographyKind.fromKind(geography.getKind());
        switch (kind) {
            case SINGLEPOINT: 
            case POINT: {
                return Constructors.pointToGeom(geography, geometryFactory);
            }
            case SINGLEPOLYLINE: 
            case POLYLINE: {
                return Constructors.polylineToGeom(geography, geometryFactory);
            }
            case POLYGON: 
            case MULTIPOLYGON: {
                return Constructors.polygonToGeom(geography, geometryFactory);
            }
            case GEOGRAPHY_COLLECTION: {
                return Constructors.collectionToGeom(geography, geometryFactory);
            }
        }
        throw new IllegalArgumentException("Unsupported Geography type: " + kind);
    }

    private static Geometry pointToGeom(Geography g, GeometryFactory gf) {
        if (g instanceof SinglePointGeography) {
            S2Point p = ((SinglePointGeography)g).getPoints().get(0);
            S2LatLng ll = S2LatLng.fromPoint((S2Point)p);
            return gf.createPoint(new Coordinate(ll.lngDegrees(), ll.latDegrees()));
        }
        if (g instanceof PointGeography) {
            List<S2Point> pts = ((PointGeography)g).getPoints();
            Coordinate[] cs = new Coordinate[pts.size()];
            for (int i = 0; i < pts.size(); ++i) {
                S2LatLng ll = S2LatLng.fromPoint((S2Point)pts.get(i));
                cs[i] = new Coordinate(ll.lngDegrees(), ll.latDegrees());
            }
            return gf.createMultiPointFromCoords(cs);
        }
        return null;
    }

    private static Geometry polylineToGeom(Geography g, GeometryFactory gf) {
        if (g instanceof SinglePolylineGeography) {
            S2Polyline line = ((SinglePolylineGeography)g).getPolylines().get(0);
            int n = line.numVertices();
            Coordinate[] cs = new Coordinate[n];
            for (int k = 0; k < n; ++k) {
                S2LatLng ll = S2LatLng.fromPoint((S2Point)line.vertex(k));
                cs[k] = new Coordinate(ll.lngDegrees(), ll.latDegrees());
            }
            return gf.createLineString(cs);
        }
        if (g instanceof PolylineGeography) {
            List<S2Polyline> lines = ((PolylineGeography)g).getPolylines();
            LineString[] lss = new LineString[lines.size()];
            for (int i = 0; i < lines.size(); ++i) {
                S2Polyline pl = lines.get(i);
                int n = pl.numVertices();
                Coordinate[] cs = new Coordinate[n];
                for (int k = 0; k < n; ++k) {
                    S2LatLng ll = S2LatLng.fromPoint((S2Point)pl.vertex(k));
                    cs[k] = new Coordinate(ll.lngDegrees(), ll.latDegrees());
                }
                lss[i] = gf.createLineString(cs);
            }
            return gf.createMultiLineString(lss);
        }
        return null;
    }

    private static Geometry polygonToGeom(Geography g, GeometryFactory gf) {
        if (g instanceof PolygonGeography) {
            S2Polygon s2p = ((PolygonGeography)g).polygon;
            return Constructors.s2LoopsToJts(s2p.getLoops(), gf);
        }
        if (g instanceof MultiPolygonGeography) {
            List<Geography> parts = ((MultiPolygonGeography)g).getFeatures();
            Polygon[] polys = new Polygon[parts.size()];
            for (int i = 0; i < parts.size(); ++i) {
                polys[i] = (Polygon)Constructors.s2LoopsToJts(((PolygonGeography)parts.get((int)i)).polygon.getLoops(), gf);
            }
            return gf.createMultiPolygon(polys);
        }
        return null;
    }

    private static Geometry s2LoopsToJts(List<S2Loop> loops, GeometryFactory gf) {
        if (loops == null || loops.isEmpty()) {
            return gf.createPolygon();
        }
        ArrayList<LinearRing> shells = new ArrayList<LinearRing>();
        ArrayList holesPerShell = new ArrayList();
        ArrayList<int[]> shellStack = new ArrayList<int[]>();
        for (S2Loop L : loops) {
            int n = L.numVertices();
            if (n < 3) continue;
            Coordinate[] cs = new Coordinate[n + 1];
            for (int i = 0; i < n; ++i) {
                S2LatLng ll = S2LatLng.fromPoint((S2Point)L.vertex(i)).normalized();
                cs[i] = new Coordinate(ll.lngDegrees(), ll.latDegrees());
            }
            cs[n] = cs[0];
            if (cs.length < 4 || cs[0].equals2D(cs[1]) || cs[1].equals2D(cs[2])) continue;
            LinearRing ring = gf.createLinearRing(cs);
            boolean isShell = (L.depth() & 1) == 0;
            int depth = L.depth();
            while (!shellStack.isEmpty() && ((int[])shellStack.get(shellStack.size() - 1))[1] >= depth) {
                shellStack.remove(shellStack.size() - 1);
            }
            if (isShell) {
                shells.add(ring);
                holesPerShell.add(new ArrayList());
                shellStack.add(new int[]{shells.size() - 1, depth});
                continue;
            }
            ring = Constructors.ensureOrientation(ring, false, gf);
            if (shellStack.isEmpty()) continue;
            int[] shellContainer = (int[])shellStack.get(shellStack.size() - 1);
            ((List)holesPerShell.get(shellContainer[0])).add(ring);
        }
        if (shells.isEmpty()) {
            return gf.createPolygon();
        }
        if (shells.size() == 1) {
            Polygon polygon = gf.createPolygon((LinearRing)shells.get(0), ((List)holesPerShell.get(0)).toArray(new LinearRing[0]));
            return polygon;
        }
        Polygon[] polys = new Polygon[shells.size()];
        for (int i = 0; i < shells.size(); ++i) {
            polys[i] = gf.createPolygon((LinearRing)shells.get(i), ((List)holesPerShell.get(i)).toArray(new LinearRing[0]));
        }
        return gf.createMultiPolygon(polys);
    }

    private static LinearRing ensureOrientation(LinearRing ring, boolean wantCCW, GeometryFactory gf) {
        boolean isCCW = Orientation.isCCW((Coordinate[])ring.getCoordinates());
        if (isCCW != wantCCW) {
            Coordinate[] cs = CoordinateArrays.copyDeep((Coordinate[])ring.getCoordinates());
            CoordinateArrays.reverse((Coordinate[])cs);
            return gf.createLinearRing(cs);
        }
        return ring;
    }

    private static Geometry collectionToGeom(Geography g, GeometryFactory gf) {
        List<Geography> parts = ((GeographyCollection)g).getFeatures();
        Geometry[] gs = new Geometry[parts.size()];
        for (int i = 0; i < parts.size(); ++i) {
            gs[i] = Constructors.geogToGeometry(parts.get(i));
        }
        return gf.createGeometryCollection(gs);
    }

    public static Geography geomToGeography(Geometry geom) {
        Geography geography;
        if (geom == null) {
            return null;
        }
        if (geom instanceof Point) {
            geography = Constructors.pointToGeog((Point)geom);
        } else if (geom instanceof MultiPoint) {
            geography = Constructors.mPointToGeog((MultiPoint)geom);
        } else if (geom instanceof LineString) {
            geography = Constructors.lineToGeog((LineString)geom);
        } else if (geom instanceof MultiLineString) {
            geography = Constructors.mLineToGeog((MultiLineString)geom);
        } else if (geom instanceof Polygon) {
            geography = Constructors.polyToGeog((Polygon)geom);
        } else if (geom instanceof MultiPolygon) {
            geography = Constructors.mPolyToGeog((MultiPolygon)geom);
        } else if (geom instanceof GeometryCollection) {
            geography = Constructors.geomCollToGeog((GeometryCollection)geom);
        } else {
            throw new UnsupportedOperationException("Geometry type is not supported: " + geom.getClass().getSimpleName());
        }
        geography.setSRID(geom.getSRID());
        return geography;
    }

    private static Geography pointToGeog(Point geom) throws IllegalArgumentException {
        Coordinate[] pts = geom.getCoordinates();
        if (pts.length == 0 || Double.isNaN(pts[0].x) || Double.isNaN(pts[0].y)) {
            return new SinglePointGeography();
        }
        double lon = pts[0].x;
        double lat = pts[0].y;
        S2Point s2Point = S2LatLng.fromDegrees((double)lat, (double)lon).toPoint();
        return new SinglePointGeography(s2Point);
    }

    private static Geography mPointToGeog(MultiPoint geom) throws IllegalArgumentException {
        Coordinate[] pts = geom.getCoordinates();
        List<S2Point> points = Constructors.toS2Points(pts);
        S2Builder builder = new S2Builder.Builder().build();
        S2PointVectorLayer layer = new S2PointVectorLayer();
        builder.startLayer((S2BuilderLayer)layer);
        for (S2Point pt : points) {
            builder.addPoint(pt);
        }
        S2Error error = new S2Error();
        if (!builder.build(error)) {
            throw new IllegalArgumentException("Failed to build S2 point layer: " + error.text());
        }
        return new PointGeography(layer.getPointVector());
    }

    private static Geography lineToGeog(LineString geom) throws IllegalArgumentException {
        List<S2Point> pts = Constructors.toS2Points(geom.getCoordinates());
        if (pts.size() < 2) {
            return new SinglePolylineGeography();
        }
        S2Builder builder = new S2Builder.Builder().build();
        S2PolylineLayer layer = new S2PolylineLayer();
        builder.startLayer((S2BuilderLayer)layer);
        builder.addPolyline(new S2Polyline(pts));
        S2Error error = new S2Error();
        if (!builder.build(error)) {
            throw new IllegalArgumentException("Failed to build S2 polyline: " + error.text());
        }
        S2Polyline s2poly = layer.getPolyline();
        return new SinglePolylineGeography(s2poly);
    }

    private static Geography mLineToGeog(MultiLineString geom) throws IllegalArgumentException {
        S2Builder builder = new S2Builder.Builder().build();
        S2PolylineVectorLayer.Options opts = new S2PolylineVectorLayer.Options().setDuplicateEdges(S2Builder.GraphOptions.DuplicateEdges.MERGE);
        S2PolylineVectorLayer vectorLayer = new S2PolylineVectorLayer(opts);
        builder.startLayer((S2BuilderLayer)vectorLayer);
        for (int i = 0; i < geom.getNumGeometries(); ++i) {
            LineString ls = (LineString)geom.getGeometryN(i);
            List<S2Point> pts = Constructors.toS2Points(ls.getCoordinates());
            if (pts.size() < 2) continue;
            builder.addPolyline(new S2Polyline(pts));
        }
        S2Error error = new S2Error();
        if (!builder.build(error)) {
            throw new IllegalArgumentException("Failed to build S2 polyline: " + error.text());
        }
        return new PolylineGeography(vectorLayer.getPolylines());
    }

    private static Geography polyToGeog(Polygon geom) throws IllegalArgumentException {
        S2Polygon s2poly = Constructors.toS2Polygon(geom);
        return new PolygonGeography(s2poly);
    }

    private static Geography mPolyToGeog(MultiPolygon geom) throws IllegalArgumentException {
        ArrayList<S2Polygon> polys = new ArrayList<S2Polygon>();
        for (int i = 0; i < geom.getNumGeometries(); ++i) {
            Polygon p = (Polygon)geom.getGeometryN(i);
            S2Polygon s2 = Constructors.toS2Polygon(p);
            if (s2 == null || s2.isEmpty()) continue;
            polys.add(s2);
        }
        return new MultiPolygonGeography(Geography.GeographyKind.MULTIPOLYGON, Collections.unmodifiableList(polys));
    }

    private static Geography geomCollToGeog(GeometryCollection geom) {
        ArrayList<Geography> features = new ArrayList<Geography>();
        for (int i = 0; i < geom.getNumGeometries(); ++i) {
            Geometry g = geom.getGeometryN(i);
            Geography sub = Constructors.geomToGeography(g);
            if (sub == null) continue;
            features.add(sub);
        }
        return new GeographyCollection(features);
    }

    private static List<S2Point> toS2Points(Coordinate[] coords) throws IllegalArgumentException {
        ArrayList<S2Point> points = new ArrayList<S2Point>(coords.length);
        for (int i = 0; i < coords.length; ++i) {
            double lon = coords[i].x;
            double lat = coords[i].y;
            if (Double.isNaN(lat) || Double.isNaN(lon)) continue;
            S2Point s2Point = S2LatLng.fromDegrees((double)lat, (double)lon).toPoint();
            if (!points.isEmpty() && ((S2Point)points.get(points.size() - 1)).equals((Object)s2Point)) continue;
            points.add(s2Point);
        }
        return points;
    }

    private static S2Loop toS2Loop(LinearRing ring) throws IllegalArgumentException {
        Coordinate[] coords = ring.getCoordinates();
        if (coords == null || coords.length < 4) {
            return null;
        }
        List<S2Point> pts = Constructors.toS2Points(coords);
        if (pts.size() < 3) {
            return null;
        }
        if (pts.get(0).equals((Object)pts.get(pts.size() - 1))) {
            pts.remove(pts.size() - 1);
        }
        S2Loop loop = new S2Loop(pts);
        loop.normalize();
        return loop;
    }

    private static S2Polygon toS2Polygon(Polygon poly) throws IllegalArgumentException {
        ArrayList<S2Loop> loops = new ArrayList<S2Loop>();
        S2Loop shell = Constructors.toS2Loop(poly.getExteriorRing());
        if (shell != null) {
            loops.add(shell);
        }
        for (int i = 0; i < poly.getNumInteriorRing(); ++i) {
            S2Loop hole = Constructors.toS2Loop(poly.getInteriorRingN(i));
            if (hole == null) continue;
            loops.add(hole);
        }
        if (loops.isEmpty()) {
            return null;
        }
        S2Builder builder = new S2Builder.Builder().build();
        S2PolygonLayer polyLayer = new S2PolygonLayer();
        builder.startLayer((S2BuilderLayer)polyLayer);
        for (S2Loop loop : loops) {
            builder.addLoop(loop);
        }
        S2Error error = new S2Error();
        if (!builder.build(error)) {
            throw new IllegalArgumentException("S2Builder failed: " + error.text());
        }
        return polyLayer.getPolygon();
    }
}

