/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import org.apache.iceberg.ExpireSnapshots;
import org.apache.iceberg.GenericManifestFile;
import org.apache.iceberg.ManifestEntry;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.ManifestFiles;
import org.apache.iceberg.ManifestReader;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.avro.Avro;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.relocated.com.google.common.util.concurrent.MoreExecutors;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.SnapshotUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RemoveSnapshots
implements ExpireSnapshots {
    private static final Logger LOG = LoggerFactory.getLogger(RemoveSnapshots.class);
    private static final ExecutorService DEFAULT_DELETE_EXECUTOR_SERVICE = MoreExecutors.newDirectExecutorService();
    private final Consumer<String> defaultDelete = new Consumer<String>(){

        @Override
        public void accept(String file) {
            RemoveSnapshots.this.ops.io().deleteFile(file);
        }
    };
    private final TableOperations ops;
    private final Set<Long> idsToRemove = Sets.newHashSet();
    private boolean cleanExpiredFiles = true;
    private TableMetadata base;
    private long expireOlderThan;
    private int minNumSnapshots;
    private Consumer<String> deleteFunc = this.defaultDelete;
    private ExecutorService deleteExecutorService = DEFAULT_DELETE_EXECUTOR_SERVICE;
    private static final Schema MANIFEST_PROJECTION = ManifestFile.schema().select("manifest_path", "added_snapshot_id", "deleted_data_files_count");

    RemoveSnapshots(TableOperations ops) {
        this.ops = ops;
        this.base = ops.current();
        long maxSnapshotAgeMs = PropertyUtil.propertyAsLong(this.base.properties(), "history.expire.max-snapshot-age-ms", 432000000L);
        this.expireOlderThan = System.currentTimeMillis() - maxSnapshotAgeMs;
        this.minNumSnapshots = PropertyUtil.propertyAsInt(this.base.properties(), "history.expire.min-snapshots-to-keep", 1);
        ValidationException.check(PropertyUtil.propertyAsBoolean(this.base.properties(), "gc.enabled", true), "Cannot expire snapshots: GC is disabled (deleting files may corrupt other tables)", new Object[0]);
    }

    @Override
    public ExpireSnapshots cleanExpiredFiles(boolean clean) {
        this.cleanExpiredFiles = clean;
        return this;
    }

    @Override
    public ExpireSnapshots expireSnapshotId(long expireSnapshotId) {
        LOG.info("Expiring snapshot with id: {}", (Object)expireSnapshotId);
        this.idsToRemove.add(expireSnapshotId);
        return this;
    }

    @Override
    public ExpireSnapshots expireOlderThan(long timestampMillis) {
        LOG.info("Expiring snapshots older than: {} ({})", (Object)new Date(timestampMillis), (Object)timestampMillis);
        this.expireOlderThan = timestampMillis;
        return this;
    }

    @Override
    public ExpireSnapshots retainLast(int numSnapshots) {
        Preconditions.checkArgument(1 <= numSnapshots, "Number of snapshots to retain must be at least 1, cannot be: %s", numSnapshots);
        this.minNumSnapshots = numSnapshots;
        return this;
    }

    @Override
    public ExpireSnapshots deleteWith(Consumer<String> newDeleteFunc) {
        this.deleteFunc = newDeleteFunc;
        return this;
    }

    @Override
    public ExpireSnapshots executeDeleteWith(ExecutorService executorService) {
        this.deleteExecutorService = executorService;
        return this;
    }

    @Override
    public List<Snapshot> apply() {
        TableMetadata updated = this.internalApply();
        ArrayList<Snapshot> removed = Lists.newArrayList(this.base.snapshots());
        removed.removeAll(updated.snapshots());
        return removed;
    }

    private TableMetadata internalApply() {
        this.base = this.ops.refresh();
        HashSet<Long> idsToRetain = Sets.newHashSet();
        List<Long> ancestorIds = SnapshotUtil.ancestorIds(this.base.currentSnapshot(), this.base::snapshot);
        if (this.minNumSnapshots >= ancestorIds.size()) {
            idsToRetain.addAll(ancestorIds);
        } else {
            idsToRetain.addAll(ancestorIds.subList(0, this.minNumSnapshots));
        }
        TableMetadata updateMeta = this.base.removeSnapshotsIf(snapshot -> this.idsToRemove.contains(snapshot.snapshotId()) || snapshot.timestampMillis() < this.expireOlderThan && !idsToRetain.contains(snapshot.snapshotId()));
        List<Snapshot> updateSnapshots = updateMeta.snapshots();
        List<Snapshot> baseSnapshots = this.base.snapshots();
        return updateSnapshots.size() != baseSnapshots.size() ? updateMeta : this.base;
    }

    @Override
    public void commit() {
        Tasks.foreach(this.ops).retry(this.base.propertyAsInt("commit.retry.num-retries", 4)).exponentialBackoff(this.base.propertyAsInt("commit.retry.min-wait-ms", 100), this.base.propertyAsInt("commit.retry.max-wait-ms", 60000), this.base.propertyAsInt("commit.retry.total-timeout-ms", 1800000), 2.0).onlyRetryOn((Class<Exception>)CommitFailedException.class).run(item -> {
            TableMetadata updated = this.internalApply();
            this.ops.commit(this.base, updated);
        });
        LOG.info("Committed snapshot changes");
        if (this.cleanExpiredFiles) {
            this.cleanExpiredSnapshots();
        } else {
            LOG.info("Cleaning up manifest and data files disabled, leaving them in place");
        }
    }

    private void cleanExpiredSnapshots() {
        TableMetadata current = this.ops.refresh();
        HashSet<Long> validIds = Sets.newHashSet();
        for (Snapshot snapshot : current.snapshots()) {
            validIds.add(snapshot.snapshotId());
        }
        HashSet<Long> expiredIds = Sets.newHashSet();
        for (Snapshot snapshot : this.base.snapshots()) {
            long snapshotId = snapshot.snapshotId();
            if (validIds.contains(snapshotId)) continue;
            LOG.info("Expired snapshot: {}", (Object)snapshot);
            expiredIds.add(snapshotId);
        }
        if (expiredIds.isEmpty()) {
            return;
        }
        LOG.info("Committed snapshot changes; cleaning up expired manifests and data files.");
        this.removeExpiredFiles(current.snapshots(), validIds, expiredIds);
    }

    private void removeExpiredFiles(List<Snapshot> snapshots, Set<Long> validIds, Set<Long> expiredIds) {
        HashSet<Long> ancestorIds = Sets.newHashSet(SnapshotUtil.ancestorIds(this.base.currentSnapshot(), this.base::snapshot));
        HashSet<Long> pickedAncestorSnapshotIds = Sets.newHashSet();
        Iterator iterator = ancestorIds.iterator();
        while (iterator.hasNext()) {
            long snapshotId = (Long)iterator.next();
            String sourceSnapshotId = this.base.snapshot(snapshotId).summary().get("source-snapshot-id");
            if (sourceSnapshotId == null) continue;
            pickedAncestorSnapshotIds.add(Long.parseLong(sourceSnapshotId));
        }
        HashSet validManifests = Sets.newHashSet();
        HashSet<ManifestFile> manifestsToScan = Sets.newHashSet();
        Tasks.foreach(snapshots).retry(3).suppressFailureWhenFinished().onFailure((snapshot, exc) -> LOG.warn("Failed on snapshot {} while reading manifest list: {}", snapshot.snapshotId(), snapshot.manifestListLocation(), exc)).run(snapshot -> {
            try {
                CloseableIterable<ManifestFile> manifests = this.readManifestFiles((Snapshot)snapshot);
                Throwable throwable = null;
                try {
                    for (ManifestFile manifest : manifests) {
                        validManifests.add(manifest.path());
                        long snapshotId = manifest.snapshotId();
                        boolean fromValidSnapshots = validIds.contains(snapshotId);
                        boolean isFromAncestor = ancestorIds.contains(snapshotId);
                        boolean isPicked = pickedAncestorSnapshotIds.contains(snapshotId);
                        if (fromValidSnapshots || !isFromAncestor && !isPicked || !manifest.hasDeletedFiles()) continue;
                        manifestsToScan.add(manifest.copy());
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (manifests != null) {
                        RemoveSnapshots.$closeResource(throwable, manifests);
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to close manifest list: %s", snapshot.manifestListLocation());
            }
        });
        HashSet<String> manifestListsToDelete = Sets.newHashSet();
        HashSet<String> manifestsToDelete = Sets.newHashSet();
        HashSet<ManifestFile> manifestsToRevert = Sets.newHashSet();
        Tasks.foreach(this.base.snapshots()).retry(3).suppressFailureWhenFinished().onFailure((snapshot, exc) -> LOG.warn("Failed on snapshot {} while reading manifest list: {}", snapshot.snapshotId(), snapshot.manifestListLocation(), exc)).run(snapshot -> {
            long snapshotId = snapshot.snapshotId();
            if (!validIds.contains(snapshotId)) {
                if (pickedAncestorSnapshotIds.contains(snapshotId)) {
                    return;
                }
                long sourceSnapshotId = PropertyUtil.propertyAsLong(snapshot.summary(), "source-snapshot-id", -1L);
                if (ancestorIds.contains(sourceSnapshotId)) {
                    return;
                }
                if (pickedAncestorSnapshotIds.contains(sourceSnapshotId)) {
                    return;
                }
                try {
                    CloseableIterable<ManifestFile> manifests = this.readManifestFiles((Snapshot)snapshot);
                    Throwable throwable = null;
                    try {
                        for (ManifestFile manifest : manifests) {
                            if (validManifests.contains(manifest.path())) continue;
                            manifestsToDelete.add(manifest.path());
                            boolean isFromAncestor = ancestorIds.contains(manifest.snapshotId());
                            boolean isFromExpiringSnapshot = expiredIds.contains(manifest.snapshotId());
                            if (isFromAncestor && manifest.hasDeletedFiles()) {
                                manifestsToScan.add(manifest.copy());
                            }
                            if (isFromAncestor || !isFromExpiringSnapshot || !manifest.hasAddedFiles()) continue;
                            manifestsToRevert.add(manifest.copy());
                        }
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (manifests != null) {
                            RemoveSnapshots.$closeResource(throwable, manifests);
                        }
                    }
                }
                catch (IOException e) {
                    throw new RuntimeIOException(e, "Failed to close manifest list: %s", snapshot.manifestListLocation());
                }
                if (snapshot.manifestListLocation() != null) {
                    manifestListsToDelete.add(snapshot.manifestListLocation());
                }
            }
        });
        this.deleteDataFiles(manifestsToScan, manifestsToRevert, validIds);
        this.deleteMetadataFiles(manifestsToDelete, manifestListsToDelete);
    }

    private void deleteMetadataFiles(Set<String> manifestsToDelete, Set<String> manifestListsToDelete) {
        LOG.warn("Manifests to delete: {}", (Object)Joiner.on(", ").join(manifestsToDelete));
        LOG.warn("Manifests Lists to delete: {}", (Object)Joiner.on(", ").join(manifestListsToDelete));
        Tasks.foreach(manifestsToDelete).executeWith(this.deleteExecutorService).retry(3).stopRetryOn(NotFoundException.class).suppressFailureWhenFinished().onFailure((manifest, exc) -> LOG.warn("Delete failed for manifest: {}", manifest, (Object)exc)).run(this.deleteFunc::accept);
        Tasks.foreach(manifestListsToDelete).executeWith(this.deleteExecutorService).retry(3).stopRetryOn(NotFoundException.class).suppressFailureWhenFinished().onFailure((list, exc) -> LOG.warn("Delete failed for manifest list: {}", list, (Object)exc)).run(this.deleteFunc::accept);
    }

    private void deleteDataFiles(Set<ManifestFile> manifestsToScan, Set<ManifestFile> manifestsToRevert, Set<Long> validIds) {
        Set<String> filesToDelete = this.findFilesToDelete(manifestsToScan, manifestsToRevert, validIds);
        Tasks.foreach(filesToDelete).executeWith(this.deleteExecutorService).retry(3).stopRetryOn(NotFoundException.class).suppressFailureWhenFinished().onFailure((file, exc) -> LOG.warn("Delete failed for data file: {}", file, (Object)exc)).run(file -> this.deleteFunc.accept((String)file));
    }

    private Set<String> findFilesToDelete(Set<ManifestFile> manifestsToScan, Set<ManifestFile> manifestsToRevert, Set<Long> validIds) {
        ConcurrentHashMap.KeySetView filesToDelete = ConcurrentHashMap.newKeySet();
        Tasks.foreach(manifestsToScan).retry(3).suppressFailureWhenFinished().executeWith(ThreadPools.getWorkerPool()).onFailure((item, exc) -> LOG.warn("Failed to get deleted files: this may cause orphaned data files", exc)).run(manifest -> {
            try {
                ManifestReader<?> reader = ManifestFiles.open(manifest, this.ops.io(), this.ops.current().specsById());
                Throwable throwable = null;
                try {
                    for (ManifestEntry manifestEntry : reader.entries()) {
                        if (manifestEntry.status() != ManifestEntry.Status.DELETED || validIds.contains(manifestEntry.snapshotId())) continue;
                        filesToDelete.add(manifestEntry.file().path().toString());
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (reader != null) {
                        RemoveSnapshots.$closeResource(throwable, reader);
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to read manifest file: %s", manifest);
            }
        });
        Tasks.foreach(manifestsToRevert).retry(3).suppressFailureWhenFinished().executeWith(ThreadPools.getWorkerPool()).onFailure((item, exc) -> LOG.warn("Failed to get added files: this may cause orphaned data files", exc)).run(manifest -> {
            try {
                ManifestReader<?> reader = ManifestFiles.open(manifest, this.ops.io(), this.ops.current().specsById());
                Throwable throwable = null;
                try {
                    for (ManifestEntry manifestEntry : reader.entries()) {
                        if (manifestEntry.status() != ManifestEntry.Status.ADDED) continue;
                        filesToDelete.add(manifestEntry.file().path().toString());
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (reader != null) {
                        RemoveSnapshots.$closeResource(throwable, reader);
                    }
                }
            }
            catch (IOException e) {
                throw new RuntimeIOException(e, "Failed to read manifest file: %s", manifest);
            }
        });
        return filesToDelete;
    }

    private CloseableIterable<ManifestFile> readManifestFiles(Snapshot snapshot) {
        if (snapshot.manifestListLocation() != null) {
            return Avro.read(this.ops.io().newInputFile(snapshot.manifestListLocation())).rename("manifest_file", GenericManifestFile.class.getName()).classLoader(GenericManifestFile.class.getClassLoader()).project(MANIFEST_PROJECTION).reuseContainers(true).build();
        }
        return CloseableIterable.withNoopClose(snapshot.allManifests());
    }

    private static /* synthetic */ /* end resource */ void $closeResource(Throwable x0, AutoCloseable x1) {
        if (x0 != null) {
            try {
                x1.close();
            }
            catch (Throwable throwable) {
                x0.addSuppressed(throwable);
            }
        } else {
            x1.close();
        }
    }
}

