/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.InvalidRecordException;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.utils.ByteBufferOutputStream;
import org.apache.kafka.common.utils.ByteUtils;
import org.apache.kafka.common.utils.Checksums;
import org.apache.kafka.common.utils.Crc32;
import org.apache.kafka.common.utils.Utils;

public final class LegacyRecord {
    public static final int CRC_OFFSET = 0;
    public static final int CRC_LENGTH = 4;
    public static final int MAGIC_OFFSET = 4;
    public static final int MAGIC_LENGTH = 1;
    public static final int ATTRIBUTES_OFFSET = 5;
    public static final int ATTRIBUTES_LENGTH = 1;
    public static final int TIMESTAMP_OFFSET = 6;
    public static final int TIMESTAMP_LENGTH = 8;
    public static final int KEY_SIZE_OFFSET_V0 = 6;
    public static final int KEY_SIZE_OFFSET_V1 = 14;
    public static final int KEY_SIZE_LENGTH = 4;
    public static final int KEY_OFFSET_V0 = 10;
    public static final int KEY_OFFSET_V1 = 18;
    public static final int VALUE_SIZE_LENGTH = 4;
    public static final int HEADER_SIZE_V0 = 6;
    public static final int HEADER_SIZE_V1 = 14;
    public static final int RECORD_OVERHEAD_V0 = 14;
    public static final int RECORD_OVERHEAD_V1 = 22;
    private static final int COMPRESSION_CODEC_MASK = 7;
    private static final byte TIMESTAMP_TYPE_MASK = 8;
    public static final long NO_TIMESTAMP = -1L;
    private final ByteBuffer buffer;
    private final Long wrapperRecordTimestamp;
    private final TimestampType wrapperRecordTimestampType;

    public LegacyRecord(ByteBuffer buffer) {
        this(buffer, null, null);
    }

    public LegacyRecord(ByteBuffer buffer, Long wrapperRecordTimestamp, TimestampType wrapperRecordTimestampType) {
        this.buffer = buffer;
        this.wrapperRecordTimestamp = wrapperRecordTimestamp;
        this.wrapperRecordTimestampType = wrapperRecordTimestampType;
    }

    public long computeChecksum() {
        return Crc32.crc32(this.buffer, 4, this.buffer.limit() - 4);
    }

    public long checksum() {
        return ByteUtils.readUnsignedInt(this.buffer, 0);
    }

    public boolean isValid() {
        return this.sizeInBytes() >= 14 && this.checksum() == this.computeChecksum();
    }

    public Long wrapperRecordTimestamp() {
        return this.wrapperRecordTimestamp;
    }

    public TimestampType wrapperRecordTimestampType() {
        return this.wrapperRecordTimestampType;
    }

    public void ensureValid() {
        if (this.sizeInBytes() < 14) {
            throw new InvalidRecordException("Record is corrupt (crc could not be retrieved as the record is too small, size = " + this.sizeInBytes() + ")");
        }
        if (!this.isValid()) {
            throw new InvalidRecordException("Record is corrupt (stored crc = " + this.checksum() + ", computed crc = " + this.computeChecksum() + ")");
        }
    }

    public int sizeInBytes() {
        return this.buffer.limit();
    }

    public int keySize() {
        if (this.magic() == 0) {
            return this.buffer.getInt(6);
        }
        return this.buffer.getInt(14);
    }

    public boolean hasKey() {
        return this.keySize() >= 0;
    }

    private int valueSizeOffset() {
        if (this.magic() == 0) {
            return 10 + Math.max(0, this.keySize());
        }
        return 18 + Math.max(0, this.keySize());
    }

    public int valueSize() {
        return this.buffer.getInt(this.valueSizeOffset());
    }

    public boolean hasNullValue() {
        return this.valueSize() < 0;
    }

    public byte magic() {
        return this.buffer.get(4);
    }

    public byte attributes() {
        return this.buffer.get(5);
    }

    public long timestamp() {
        if (this.magic() == 0) {
            return -1L;
        }
        if (this.wrapperRecordTimestampType == TimestampType.LOG_APPEND_TIME && this.wrapperRecordTimestamp != null) {
            return this.wrapperRecordTimestamp;
        }
        return this.buffer.getLong(6);
    }

    public TimestampType timestampType() {
        return LegacyRecord.timestampType(this.magic(), this.wrapperRecordTimestampType, this.attributes());
    }

    public CompressionType compressionType() {
        return CompressionType.forId(this.buffer.get(5) & 7);
    }

    public ByteBuffer value() {
        return Utils.sizeDelimited(this.buffer, this.valueSizeOffset());
    }

    public ByteBuffer key() {
        if (this.magic() == 0) {
            return Utils.sizeDelimited(this.buffer, 6);
        }
        return Utils.sizeDelimited(this.buffer, 14);
    }

    public ByteBuffer buffer() {
        return this.buffer;
    }

    public String toString() {
        if (this.magic() > 0) {
            return String.format("Record(magic=%d, attributes=%d, compression=%s, crc=%d, %s=%d, key=%d bytes, value=%d bytes)", new Object[]{this.magic(), this.attributes(), this.compressionType(), this.checksum(), this.timestampType(), this.timestamp(), this.key() == null ? 0 : this.key().limit(), this.value() == null ? 0 : this.value().limit()});
        }
        return String.format("Record(magic=%d, attributes=%d, compression=%s, crc=%d, key=%d bytes, value=%d bytes)", new Object[]{this.magic(), this.attributes(), this.compressionType(), this.checksum(), this.key() == null ? 0 : this.key().limit(), this.value() == null ? 0 : this.value().limit()});
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null) {
            return false;
        }
        if (!other.getClass().equals(LegacyRecord.class)) {
            return false;
        }
        LegacyRecord record = (LegacyRecord)other;
        return this.buffer.equals(record.buffer);
    }

    public int hashCode() {
        return this.buffer.hashCode();
    }

    public static LegacyRecord create(byte magic, long timestamp, byte[] key, byte[] value, CompressionType compressionType, TimestampType timestampType) {
        int keySize = key == null ? 0 : key.length;
        int valueSize = value == null ? 0 : value.length;
        ByteBuffer buffer = ByteBuffer.allocate(LegacyRecord.recordSize(magic, keySize, valueSize));
        LegacyRecord.write(buffer, magic, timestamp, Utils.wrapNullable(key), Utils.wrapNullable(value), compressionType, timestampType);
        buffer.rewind();
        return new LegacyRecord(buffer);
    }

    public static LegacyRecord create(byte magic, long timestamp, byte[] key, byte[] value) {
        return LegacyRecord.create(magic, timestamp, key, value, CompressionType.NONE, TimestampType.CREATE_TIME);
    }

    public static void writeCompressedRecordHeader(ByteBuffer buffer, byte magic, int recordSize, long timestamp, CompressionType compressionType, TimestampType timestampType) {
        int recordPosition = buffer.position();
        int valueSize = recordSize - LegacyRecord.recordOverhead(magic);
        LegacyRecord.write(buffer, magic, timestamp, null, null, compressionType, timestampType);
        buffer.position(recordPosition);
        buffer.putInt(recordPosition + LegacyRecord.keyOffset(magic), valueSize);
        long crc = Crc32.crc32(buffer, 4, recordSize - 4);
        ByteUtils.writeUnsignedInt(buffer, recordPosition + 0, crc);
    }

    private static void write(ByteBuffer buffer, byte magic, long timestamp, ByteBuffer key, ByteBuffer value, CompressionType compressionType, TimestampType timestampType) {
        try {
            DataOutputStream out = new DataOutputStream(new ByteBufferOutputStream(buffer));
            LegacyRecord.write(out, magic, timestamp, key, value, compressionType, timestampType);
        }
        catch (IOException e) {
            throw new KafkaException(e);
        }
    }

    public static long write(DataOutputStream out, byte magic, long timestamp, byte[] key, byte[] value, CompressionType compressionType, TimestampType timestampType) throws IOException {
        return LegacyRecord.write(out, magic, timestamp, Utils.wrapNullable(key), Utils.wrapNullable(value), compressionType, timestampType);
    }

    public static long write(DataOutputStream out, byte magic, long timestamp, ByteBuffer key, ByteBuffer value, CompressionType compressionType, TimestampType timestampType) throws IOException {
        byte attributes = LegacyRecord.computeAttributes(magic, compressionType, timestampType);
        long crc = LegacyRecord.computeChecksum(magic, attributes, timestamp, key, value);
        LegacyRecord.write(out, magic, crc, attributes, timestamp, key, value);
        return crc;
    }

    public static void write(DataOutputStream out, byte magic, long crc, byte attributes, long timestamp, byte[] key, byte[] value) throws IOException {
        LegacyRecord.write(out, magic, crc, attributes, timestamp, Utils.wrapNullable(key), Utils.wrapNullable(value));
    }

    private static void write(DataOutputStream out, byte magic, long crc, byte attributes, long timestamp, ByteBuffer key, ByteBuffer value) throws IOException {
        int size2;
        if (magic != 0 && magic != 1) {
            throw new IllegalArgumentException("Invalid magic value " + magic);
        }
        if (timestamp < 0L && timestamp != -1L) {
            throw new IllegalArgumentException("Invalid message timestamp " + timestamp);
        }
        out.writeInt((int)(crc & 0xFFFFFFFFL));
        out.writeByte(magic);
        out.writeByte(attributes);
        if (magic > 0) {
            out.writeLong(timestamp);
        }
        if (key == null) {
            out.writeInt(-1);
        } else {
            size2 = key.remaining();
            out.writeInt(size2);
            Utils.writeTo(out, key, size2);
        }
        if (value == null) {
            out.writeInt(-1);
        } else {
            size2 = value.remaining();
            out.writeInt(size2);
            Utils.writeTo(out, value, size2);
        }
    }

    static int recordSize(byte magic, ByteBuffer key, ByteBuffer value) {
        return LegacyRecord.recordSize(magic, key == null ? 0 : key.limit(), value == null ? 0 : value.limit());
    }

    public static int recordSize(byte magic, int keySize, int valueSize) {
        return LegacyRecord.recordOverhead(magic) + keySize + valueSize;
    }

    public static byte computeAttributes(byte magic, CompressionType type, TimestampType timestampType) {
        byte attributes = 0;
        if (type.id > 0) {
            attributes = (byte)(attributes | 7 & type.id);
        }
        if (magic > 0) {
            if (timestampType == TimestampType.NO_TIMESTAMP_TYPE) {
                throw new IllegalArgumentException("Timestamp type must be provided to compute attributes for message format v1");
            }
            if (timestampType == TimestampType.LOG_APPEND_TIME) {
                attributes = (byte)(attributes | 8);
            }
        }
        return attributes;
    }

    public static long computeChecksum(byte magic, byte attributes, long timestamp, byte[] key, byte[] value) {
        return LegacyRecord.computeChecksum(magic, attributes, timestamp, Utils.wrapNullable(key), Utils.wrapNullable(value));
    }

    private static long computeChecksum(byte magic, byte attributes, long timestamp, ByteBuffer key, ByteBuffer value) {
        int size2;
        Crc32 crc = new Crc32();
        crc.update(magic);
        crc.update(attributes);
        if (magic > 0) {
            Checksums.updateLong(crc, timestamp);
        }
        if (key == null) {
            Checksums.updateInt(crc, -1);
        } else {
            size2 = key.remaining();
            Checksums.updateInt(crc, size2);
            Checksums.update(crc, key, size2);
        }
        if (value == null) {
            Checksums.updateInt(crc, -1);
        } else {
            size2 = value.remaining();
            Checksums.updateInt(crc, size2);
            Checksums.update(crc, value, size2);
        }
        return crc.getValue();
    }

    static int recordOverhead(byte magic) {
        if (magic == 0) {
            return 14;
        }
        if (magic == 1) {
            return 22;
        }
        throw new IllegalArgumentException("Invalid magic used in LegacyRecord: " + magic);
    }

    static int headerSize(byte magic) {
        if (magic == 0) {
            return 6;
        }
        if (magic == 1) {
            return 14;
        }
        throw new IllegalArgumentException("Invalid magic used in LegacyRecord: " + magic);
    }

    private static int keyOffset(byte magic) {
        if (magic == 0) {
            return 10;
        }
        if (magic == 1) {
            return 18;
        }
        throw new IllegalArgumentException("Invalid magic used in LegacyRecord: " + magic);
    }

    public static TimestampType timestampType(byte magic, TimestampType wrapperRecordTimestampType, byte attributes) {
        if (magic == 0) {
            return TimestampType.NO_TIMESTAMP_TYPE;
        }
        if (wrapperRecordTimestampType != null) {
            return wrapperRecordTimestampType;
        }
        return (attributes & 8) == 0 ? TimestampType.CREATE_TIME : TimestampType.LOG_APPEND_TIME;
    }
}

