/*
 * Decompiled with CFR 0.152.
 */
package io.openmessaging.storage.dledger.snapshot;

import io.openmessaging.storage.dledger.DLedgerConfig;
import io.openmessaging.storage.dledger.DLedgerServer;
import io.openmessaging.storage.dledger.entry.DLedgerEntry;
import io.openmessaging.storage.dledger.exception.DLedgerException;
import io.openmessaging.storage.dledger.protocol.DLedgerResponseCode;
import io.openmessaging.storage.dledger.snapshot.SnapshotMeta;
import io.openmessaging.storage.dledger.snapshot.SnapshotReader;
import io.openmessaging.storage.dledger.snapshot.SnapshotStatus;
import io.openmessaging.storage.dledger.snapshot.SnapshotStore;
import io.openmessaging.storage.dledger.snapshot.SnapshotWriter;
import io.openmessaging.storage.dledger.snapshot.file.FileSnapshotStore;
import io.openmessaging.storage.dledger.snapshot.hook.LoadSnapshotHook;
import io.openmessaging.storage.dledger.snapshot.hook.SaveSnapshotHook;
import io.openmessaging.storage.dledger.utils.IOUtils;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager {
    private static Logger logger = LoggerFactory.getLogger(SnapshotManager.class);
    public static final String SNAPSHOT_META_FILE = "snapshot_meta";
    public static final String SNAPSHOT_DATA_FILE = "data";
    public static final String SNAPSHOT_DIR_PREFIX = "snapshot_";
    public static final String SNAPSHOT_TEMP_DIR = "tmp";
    private DLedgerServer dLedgerServer;
    private long lastSnapshotIndex;
    private long lastSnapshotTerm;
    private final SnapshotStore snapshotStore;
    private volatile boolean savingSnapshot;
    private volatile boolean loadingSnapshot;

    public SnapshotManager(DLedgerServer dLedgerServer) {
        this.dLedgerServer = dLedgerServer;
        this.snapshotStore = new FileSnapshotStore(this.dLedgerServer.getDLedgerConfig().getSnapshotStoreBaseDir());
    }

    public boolean isSavingSnapshot() {
        return this.savingSnapshot;
    }

    public boolean isLoadingSnapshot() {
        return this.loadingSnapshot;
    }

    public void saveSnapshot(DLedgerEntry dLedgerEntry) {
        if (this.savingSnapshot) {
            return;
        }
        if (dLedgerEntry.getIndex() - this.lastSnapshotIndex <= (long)this.dLedgerServer.getDLedgerConfig().getSnapshotThreshold()) {
            return;
        }
        SnapshotWriter writer = this.snapshotStore.createSnapshotWriter();
        if (writer == null) {
            return;
        }
        this.savingSnapshot = true;
        SaveSnapshotAfterHook saveSnapshotAfter = new SaveSnapshotAfterHook(writer, dLedgerEntry);
        if (!this.dLedgerServer.getFsmCaller().onSnapshotSave(saveSnapshotAfter)) {
            logger.error("Unable to call statemachine onSnapshotSave");
            saveSnapshotAfter.doCallBack(SnapshotStatus.FAIL);
        }
    }

    private void saveSnapshotAfter(SnapshotWriter writer, SnapshotMeta snapshotMeta, DLedgerEntry dLedgerEntry, SnapshotStatus status) {
        int res = status.getCode();
        if (res == SnapshotStatus.SUCCESS.getCode()) {
            writer.setSnapshotMeta(snapshotMeta);
        }
        try {
            writer.save(status);
        }
        catch (IOException e) {
            logger.error("Unable to close snapshot writer", (Throwable)e);
            res = SnapshotStatus.FAIL.getCode();
        }
        if (res == SnapshotStatus.SUCCESS.getCode()) {
            this.lastSnapshotIndex = snapshotMeta.getLastIncludedIndex();
            this.lastSnapshotTerm = snapshotMeta.getLastIncludedTerm();
            logger.info("Snapshot {} saved successfully", (Object)snapshotMeta);
            CompletableFuture.runAsync(() -> this.truncatePrefix(dLedgerEntry));
        } else {
            logger.error("Unable to save snapshot");
        }
        this.savingSnapshot = false;
    }

    private void truncatePrefix(DLedgerEntry entry) {
        this.deleteExpiredSnapshot();
        this.dLedgerServer.getFsmCaller().getdLedgerStore().resetOffsetAfterSnapshot(entry);
    }

    private void deleteExpiredSnapshot() {
        DLedgerConfig config = this.dLedgerServer.getDLedgerConfig();
        File[] snapshotFiles = new File(config.getSnapshotStoreBaseDir()).listFiles();
        if (snapshotFiles != null && snapshotFiles.length > config.getMaxSnapshotReservedNum()) {
            long minSnapshotIdx = Long.MAX_VALUE;
            for (File file : snapshotFiles) {
                String fileName = file.getName();
                if (!fileName.startsWith(SNAPSHOT_DIR_PREFIX)) continue;
                minSnapshotIdx = Math.min(Long.parseLong(fileName.substring(SNAPSHOT_DIR_PREFIX.length())), minSnapshotIdx);
            }
            String deleteFilePath = config.getSnapshotStoreBaseDir() + File.separator + SNAPSHOT_DIR_PREFIX + minSnapshotIdx;
            try {
                IOUtils.deleteFile(new File(deleteFilePath));
            }
            catch (IOException e) {
                logger.error("Unable to remove expired snapshot: {}", (Object)deleteFilePath, (Object)e);
            }
        }
    }

    public void loadSnapshot() {
        if (this.loadingSnapshot) {
            return;
        }
        SnapshotReader reader = this.snapshotStore.createSnapshotReader();
        if (reader == null) {
            return;
        }
        this.loadingSnapshot = true;
        LoadSnapshotAfterHook loadSnapshotAfter = new LoadSnapshotAfterHook(reader);
        if (!this.dLedgerServer.getFsmCaller().onSnapshotLoad(loadSnapshotAfter)) {
            this.dLedgerServer.getFsmCaller().setError(this.dLedgerServer, new DLedgerException(DLedgerResponseCode.LOAD_SNAPSHOT_ERROR, "Unable to call statemachine onSnapshotLoad"));
        }
    }

    private void loadSnapshotAfter(SnapshotReader reader, SnapshotMeta snapshotMeta, SnapshotStatus status) {
        if (status.getCode() == SnapshotStatus.SUCCESS.getCode()) {
            this.lastSnapshotIndex = snapshotMeta.getLastIncludedIndex();
            this.lastSnapshotTerm = snapshotMeta.getLastIncludedTerm();
            this.loadingSnapshot = false;
            logger.info("Snapshot {} loaded successfully", (Object)snapshotMeta);
        } else {
            if (status.getCode() == SnapshotStatus.EXPIRED.getCode()) {
                this.loadingSnapshot = false;
                return;
            }
            boolean failed = false;
            try {
                IOUtils.deleteFile(new File(reader.getSnapshotStorePath()));
            }
            catch (IOException e) {
                logger.error("Unable to remove error snapshot: {}", (Object)reader.getSnapshotStorePath(), (Object)e);
                failed = true;
            }
            DLedgerConfig config = this.dLedgerServer.getDLedgerConfig();
            if (Objects.requireNonNull(new File(config.getSnapshotStoreBaseDir()).listFiles()).length == 0) {
                logger.error("No snapshot for recovering state machine: {}", (Object)config.getSnapshotStoreBaseDir());
                failed = true;
            }
            if (failed) {
                if (this.dLedgerServer.getFsmCaller().getdLedgerStore().getLedgerBeginIndex() == 0L) {
                    this.loadingSnapshot = false;
                    return;
                }
                this.dLedgerServer.getFsmCaller().setError(this.dLedgerServer, new DLedgerException(DLedgerResponseCode.LOAD_SNAPSHOT_ERROR, "Fail to recover state machine"));
                return;
            }
            logger.warn("Load snapshot from {} failed. Start recovering from the previous snapshot", (Object)reader.getSnapshotStorePath());
            this.loadingSnapshot = false;
            this.loadSnapshot();
        }
    }

    private class LoadSnapshotAfterHook
    implements LoadSnapshotHook {
        SnapshotReader reader;
        SnapshotMeta snapshotMeta;

        public LoadSnapshotAfterHook(SnapshotReader reader) {
            this.reader = reader;
        }

        @Override
        public void doCallBack(SnapshotStatus status) {
            SnapshotManager.this.loadSnapshotAfter(this.reader, this.snapshotMeta, status);
        }

        @Override
        public void registerSnapshotMeta(SnapshotMeta snapshotMeta) {
            this.snapshotMeta = snapshotMeta;
        }

        @Override
        public SnapshotReader getSnapshotReader() {
            return this.reader;
        }
    }

    private class SaveSnapshotAfterHook
    implements SaveSnapshotHook {
        SnapshotWriter writer;
        DLedgerEntry dLedgerEntry;
        SnapshotMeta snapshotMeta;

        public SaveSnapshotAfterHook(SnapshotWriter writer, DLedgerEntry dLedgerEntry) {
            this.writer = writer;
            this.dLedgerEntry = dLedgerEntry;
        }

        @Override
        public void doCallBack(SnapshotStatus status) {
            SnapshotManager.this.saveSnapshotAfter(this.writer, this.snapshotMeta, this.dLedgerEntry, status);
        }

        @Override
        public void registerSnapshotMeta(SnapshotMeta snapshotMeta) {
            this.snapshotMeta = snapshotMeta;
            this.writer.setSnapshotMeta(snapshotMeta);
        }

        @Override
        public SnapshotWriter getSnapshotWriter() {
            return this.writer;
        }

        @Override
        public DLedgerEntry getSnapshotEntry() {
            return this.dLedgerEntry;
        }
    }
}

