/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.service;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.ServiceException;
import java.io.IOException;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.utils.BackgroundService;
import org.apache.hadoop.hdds.utils.BackgroundTask;
import org.apache.hadoop.hdds.utils.BackgroundTaskQueue;
import org.apache.hadoop.hdds.utils.BackgroundTaskResult;
import org.apache.hadoop.ozone.ClientVersion;
import org.apache.hadoop.ozone.om.ExpiredOpenKeys;
import org.apache.hadoop.ozone.om.KeyManager;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.helpers.BucketLayout;
import org.apache.hadoop.ozone.om.ratis.utils.OzoneManagerRatisUtils;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.util.Time;
import org.apache.ratis.protocol.ClientId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenKeyCleanupService
extends BackgroundService {
    private static final Logger LOG = LoggerFactory.getLogger(OpenKeyCleanupService.class);
    private static final int OPEN_KEY_DELETING_CORE_POOL_SIZE = 1;
    private final OzoneManager ozoneManager;
    private final KeyManager keyManager;
    private final ClientId clientId = ClientId.randomId();
    private final Duration expireThreshold;
    private final Duration leaseThreshold;
    private final int cleanupLimitPerTask;
    private final AtomicLong submittedOpenKeyCount;
    private final AtomicLong callId;
    private final AtomicBoolean suspended;

    public OpenKeyCleanupService(long interval, TimeUnit unit, long timeout, OzoneManager ozoneManager, ConfigurationSource conf) {
        super("OpenKeyCleanupService", interval, unit, 1, timeout, ozoneManager.getThreadNamePrefix());
        this.ozoneManager = ozoneManager;
        this.keyManager = ozoneManager.getKeyManager();
        long expireMillis = conf.getTimeDuration("ozone.om.open.key.expire.threshold", "7d", TimeUnit.MILLISECONDS);
        this.expireThreshold = Duration.ofMillis(expireMillis);
        long leaseHardMillis = conf.getTimeDuration("ozone.om.lease.hard.limit", "7d", TimeUnit.MILLISECONDS);
        long leaseSoftMillis = conf.getTimeDuration("ozone.om.lease.soft.limit", "60s", TimeUnit.MILLISECONDS);
        if (leaseHardMillis < leaseSoftMillis) {
            String msg = "Hard lease limit cannot be less than Soft lease limit. LeaseHardLimit: " + leaseHardMillis + " LeaseSoftLimit: " + leaseSoftMillis;
            throw new IllegalArgumentException(msg);
        }
        this.leaseThreshold = Duration.ofMillis(leaseHardMillis);
        this.cleanupLimitPerTask = conf.getInt("ozone.om.open.key.cleanup.limit.per.task", 1000);
        this.submittedOpenKeyCount = new AtomicLong(0L);
        this.callId = new AtomicLong(0L);
        this.suspended = new AtomicBoolean(false);
    }

    @VisibleForTesting
    public void suspend() {
        this.suspended.set(true);
    }

    @VisibleForTesting
    public void resume() {
        this.suspended.set(false);
    }

    @VisibleForTesting
    public long getSubmittedOpenKeyCount() {
        return this.submittedOpenKeyCount.get();
    }

    public BackgroundTaskQueue getTasks() {
        BackgroundTaskQueue queue = new BackgroundTaskQueue();
        queue.add((BackgroundTask)new OpenKeyCleanupTask(BucketLayout.DEFAULT));
        queue.add((BackgroundTask)new OpenKeyCleanupTask(BucketLayout.FILE_SYSTEM_OPTIMIZED));
        return queue;
    }

    private boolean shouldRun() {
        return !this.suspended.get() && this.ozoneManager.isLeaderReady();
    }

    private class OpenKeyCleanupTask
    implements BackgroundTask {
        private final BucketLayout bucketLayout;

        OpenKeyCleanupTask(BucketLayout bucketLayout) {
            this.bucketLayout = bucketLayout;
        }

        public int getPriority() {
            return 0;
        }

        public BackgroundTaskResult call() throws Exception {
            OzoneManagerProtocolProtos.OMRequest omRequest;
            OzoneManagerProtocolProtos.OMResponse response;
            ExpiredOpenKeys expiredOpenKeys;
            if (!OpenKeyCleanupService.this.shouldRun()) {
                return BackgroundTaskResult.EmptyTaskResult.newResult();
            }
            LOG.debug("Running OpenKeyCleanupService");
            long startTime = Time.monotonicNow();
            try {
                expiredOpenKeys = OpenKeyCleanupService.this.keyManager.getExpiredOpenKeys(OpenKeyCleanupService.this.expireThreshold, OpenKeyCleanupService.this.cleanupLimitPerTask, this.bucketLayout, OpenKeyCleanupService.this.leaseThreshold);
            }
            catch (IOException e) {
                LOG.error("Unable to get hanging open keys, retry in next interval", (Throwable)e);
                return BackgroundTaskResult.EmptyTaskResult.newResult();
            }
            Collection openKeyBuckets = expiredOpenKeys.getOpenKeyBuckets();
            int numOpenKeys = openKeyBuckets.stream().mapToInt(OzoneManagerProtocolProtos.OpenKeyBucket.Builder::getKeysCount).sum();
            if (!openKeyBuckets.isEmpty() && (response = this.submitRequest(omRequest = this.createDeleteOpenKeysRequest(openKeyBuckets.stream()))) != null && response.getSuccess()) {
                OpenKeyCleanupService.this.ozoneManager.getMetrics().incNumOpenKeysCleaned(numOpenKeys);
                if (LOG.isDebugEnabled()) {
                    StringBuilder sb = new StringBuilder();
                    for (OzoneManagerProtocolProtos.OpenKeyBucket.Builder openKey : openKeyBuckets) {
                        sb.append(openKey.getVolumeName()).append("/").append(openKey.getBucketName()).append(": ").append(openKey.getKeysList().stream().map(OzoneManagerProtocolProtos.OpenKey::getName).collect(Collectors.toList())).append("\n");
                    }
                    LOG.debug("Non-hsync'ed openKeys being deleted in current iteration: \n" + sb);
                }
            }
            List hsyncKeys = expiredOpenKeys.getHsyncKeys();
            int numHsyncKeys = hsyncKeys.size();
            if (!hsyncKeys.isEmpty()) {
                hsyncKeys.forEach(b -> {
                    OzoneManagerProtocolProtos.OMResponse response = this.submitRequest(this.createCommitKeyRequest((OzoneManagerProtocolProtos.CommitKeyRequest.Builder)b));
                    if (response != null && response.getSuccess()) {
                        OpenKeyCleanupService.this.ozoneManager.getMetrics().incNumOpenKeysHSyncCleaned();
                        if (LOG.isDebugEnabled()) {
                            StringBuilder sb = new StringBuilder();
                            for (OzoneManagerProtocolProtos.CommitKeyRequest.Builder openKey : hsyncKeys) {
                                sb.append(openKey.getKeyArgs().getVolumeName()).append("/").append(openKey.getKeyArgs().getBucketName()).append(": ").append(openKey.getKeyArgs().getKeyName()).append(", ");
                            }
                            LOG.debug("hsync'ed openKeys committed in current iteration: \n" + sb);
                        }
                    }
                });
            }
            long timeTaken = Time.monotonicNow() - startTime;
            LOG.info("Number of expired open keys submitted for deletion: {}, for commit: {}, cleanupLimit: {}, elapsed time: {}ms", new Object[]{numOpenKeys, numHsyncKeys, OpenKeyCleanupService.this.cleanupLimitPerTask, timeTaken});
            OpenKeyCleanupService.this.ozoneManager.getPerfMetrics().setOpenKeyCleanupServiceLatencyMs(timeTaken);
            int numKeys = numOpenKeys + numHsyncKeys;
            OpenKeyCleanupService.this.submittedOpenKeyCount.addAndGet(numKeys);
            return () -> numKeys;
        }

        private OzoneManagerProtocolProtos.OMRequest createCommitKeyRequest(OzoneManagerProtocolProtos.CommitKeyRequest.Builder request) {
            return OzoneManagerProtocolProtos.OMRequest.newBuilder().setCmdType(OzoneManagerProtocolProtos.Type.CommitKey).setCommitKeyRequest(request).setClientId(OpenKeyCleanupService.this.clientId.toString()).setVersion(ClientVersion.CURRENT_VERSION).build();
        }

        private OzoneManagerProtocolProtos.OMRequest createDeleteOpenKeysRequest(Stream<OzoneManagerProtocolProtos.OpenKeyBucket.Builder> openKeyBuckets) {
            OzoneManagerProtocolProtos.DeleteOpenKeysRequest.Builder request = OzoneManagerProtocolProtos.DeleteOpenKeysRequest.newBuilder().setBucketLayout(this.bucketLayout.toProto());
            openKeyBuckets.forEach(arg_0 -> ((OzoneManagerProtocolProtos.DeleteOpenKeysRequest.Builder)request).addOpenKeysPerBucket(arg_0));
            OzoneManagerProtocolProtos.OMRequest omRequest = OzoneManagerProtocolProtos.OMRequest.newBuilder().setCmdType(OzoneManagerProtocolProtos.Type.DeleteOpenKeys).setDeleteOpenKeysRequest(request).setClientId(OpenKeyCleanupService.this.clientId.toString()).build();
            return omRequest;
        }

        private OzoneManagerProtocolProtos.OMResponse submitRequest(OzoneManagerProtocolProtos.OMRequest omRequest) {
            try {
                return OzoneManagerRatisUtils.submitRequest(OpenKeyCleanupService.this.ozoneManager, omRequest, OpenKeyCleanupService.this.clientId, OpenKeyCleanupService.this.callId.incrementAndGet());
            }
            catch (ServiceException e) {
                LOG.error("Open key " + omRequest.getCmdType() + " request failed. Will retry at next run.", (Throwable)e);
                return null;
            }
        }
    }
}

