/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.ConfigurationException;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.ozone.conf.OMClientConfig;
import org.apache.hadoop.ozone.ha.ConfUtils;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.BucketLayout;
import org.apache.hadoop.ozone.om.helpers.OMNodeDetails;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.RepeatedOmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.ServiceInfo;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class OmUtils {
    public static final Logger LOG = LoggerFactory.getLogger(OmUtils.class);
    private static final SecureRandom SRAND = new SecureRandom();
    private static byte[] randomBytes = new byte[32];
    private static final long TRANSACTION_ID_SHIFT = 8L;
    public static final long EPOCH_ID_SHIFT = 62L;
    public static final long REVERSE_EPOCH_ID_SHIFT = 2L;
    public static final long MAX_TRXN_ID = 0x3FFFFFFFFFFFFEL;
    public static final int EPOCH_WHEN_RATIS_NOT_ENABLED = 1;
    public static final int EPOCH_WHEN_RATIS_ENABLED = 2;

    private OmUtils() {
    }

    public static InetSocketAddress getOmAddress(ConfigurationSource conf) {
        return NetUtils.createSocketAddr((String)OmUtils.getOmRpcAddress(conf));
    }

    public static Map<String, List<InetSocketAddress>> getOmHAAddressesById(ConfigurationSource conf) {
        HashMap<String, List<InetSocketAddress>> result = new HashMap<String, List<InetSocketAddress>>();
        for (String serviceId : conf.getTrimmedStringCollection("ozone.om.service.ids")) {
            result.computeIfAbsent(serviceId, x -> new ArrayList());
            for (String nodeId : OmUtils.getActiveOMNodeIds(conf, serviceId)) {
                String rpcAddr = OmUtils.getOmRpcAddress(conf, ConfUtils.addKeySuffixes((String)"ozone.om.address", (String[])new String[]{serviceId, nodeId}));
                if (rpcAddr != null) {
                    ((List)result.get(serviceId)).add(NetUtils.createSocketAddr((String)rpcAddr));
                    continue;
                }
                LOG.warn("Address undefined for nodeId: {} for service {}", (Object)nodeId, (Object)serviceId);
            }
        }
        return result;
    }

    public static String getOmRpcAddress(ConfigurationSource conf) {
        Optional host = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{"ozone.om.address"});
        return host.orElse("0.0.0.0") + ":" + OmUtils.getOmRpcPort(conf);
    }

    public static String getOmRpcAddress(ConfigurationSource conf, String confKey) {
        Optional host = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{confKey});
        if (host.isPresent()) {
            return (String)host.get() + ":" + HddsUtils.getPortNumberFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{confKey}).orElse(9862);
        }
        return null;
    }

    public static InetSocketAddress getOmAddressForClients(ConfigurationSource conf) {
        Optional host = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{"ozone.om.address"});
        if (!host.isPresent()) {
            throw new IllegalArgumentException("ozone.om.address must be defined. See https://wiki.apache.org/hadoop/Ozone#Configuration for details on configuring Ozone.");
        }
        return NetUtils.createSocketAddr((String)((String)host.get() + ":" + OmUtils.getOmRpcPort(conf)));
    }

    public static boolean isServiceIdsDefined(ConfigurationSource conf) {
        String val = conf.get("ozone.om.service.ids");
        return val != null && val.length() > 0;
    }

    public static boolean isOmHAServiceId(ConfigurationSource conf, String serviceId) {
        Collection omServiceIds = conf.getTrimmedStringCollection("ozone.om.service.ids");
        return omServiceIds.contains(serviceId);
    }

    public static int getOmRpcPort(ConfigurationSource conf) {
        return HddsUtils.getPortNumberFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{"ozone.om.address"}).orElse(9862);
    }

    public static boolean isReadOnly(OzoneManagerProtocolProtos.OMRequest omRequest) {
        OzoneManagerProtocolProtos.Type cmdType = omRequest.getCmdType();
        switch (cmdType) {
            case CheckVolumeAccess: 
            case InfoVolume: 
            case ListVolume: 
            case InfoBucket: 
            case ListBuckets: 
            case LookupKey: 
            case ListKeys: 
            case ListKeysLight: 
            case ListTrash: 
            case ServiceList: 
            case ListMultiPartUploadParts: 
            case GetFileStatus: 
            case LookupFile: 
            case ListStatus: 
            case ListStatusLight: 
            case GetAcl: 
            case DBUpdates: 
            case ListMultipartUploads: 
            case FinalizeUpgradeProgress: 
            case PrepareStatus: 
            case GetS3VolumeContext: 
            case ListTenant: 
            case TenantGetUserInfo: 
            case TenantListUser: 
            case ListSnapshot: 
            case RefetchSecretKey: 
            case RangerBGSync: 
            case GetKeyInfo: 
            case SnapshotDiff: 
            case CancelSnapshotDiff: 
            case ListSnapshotDiffJobs: 
            case TransferLeadership: 
            case SetSafeMode: 
            case PrintCompactionLogDag: 
            case GetSnapshotInfo: {
                return true;
            }
            case CreateVolume: 
            case SetVolumeProperty: 
            case DeleteVolume: 
            case CreateBucket: 
            case SetBucketProperty: 
            case DeleteBucket: 
            case CreateKey: 
            case RenameKey: 
            case RenameKeys: 
            case DeleteKey: 
            case DeleteKeys: 
            case CommitKey: 
            case AllocateBlock: 
            case InitiateMultiPartUpload: 
            case CommitMultiPartUpload: 
            case CompleteMultiPartUpload: 
            case AbortMultiPartUpload: 
            case GetS3Secret: 
            case GetDelegationToken: 
            case RenewDelegationToken: 
            case CancelDelegationToken: 
            case CreateDirectory: 
            case CreateFile: 
            case RemoveAcl: 
            case SetAcl: 
            case AddAcl: 
            case PurgeKeys: 
            case RecoverTrash: 
            case FinalizeUpgrade: 
            case Prepare: 
            case CancelPrepare: 
            case DeleteOpenKeys: 
            case SetS3Secret: 
            case RevokeS3Secret: 
            case PurgeDirectories: 
            case PurgePaths: 
            case CreateTenant: 
            case DeleteTenant: 
            case TenantAssignUserAccessId: 
            case TenantRevokeUserAccessId: 
            case TenantAssignAdmin: 
            case TenantRevokeAdmin: 
            case SetRangerServiceVersion: 
            case CreateSnapshot: 
            case DeleteSnapshot: 
            case SnapshotMoveDeletedKeys: 
            case SnapshotPurge: 
            case RecoverLease: 
            case SetTimes: 
            case AbortExpiredMultiPartUploads: 
            case SetSnapshotProperty: 
            case UnknownCommand: {
                return false;
            }
            case EchoRPC: {
                return omRequest.getEchoRPCRequest().getReadOnly();
            }
        }
        LOG.error("CmdType {} is not categorized as readOnly or not.", (Object)cmdType);
        return false;
    }

    public static byte[] getSHADigest() throws IOException {
        try {
            SRAND.nextBytes(randomBytes);
            MessageDigest sha = MessageDigest.getInstance("SHA-256");
            return sha.digest(randomBytes);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new IOException("Error creating an instance of SHA-256 digest.\nThis could possibly indicate a faulty JRE");
        }
    }

    public static Collection<String> getActiveOMNodeIds(ConfigurationSource conf, String omServiceId) {
        String nodeIdsKey = ConfUtils.addSuffix((String)"ozone.om.nodes", (String)omServiceId);
        Collection nodeIds = conf.getTrimmedStringCollection(nodeIdsKey);
        String decommNodesKey = ConfUtils.addKeySuffixes((String)"ozone.om.decommissioned.nodes", (String[])new String[]{omServiceId});
        Collection decommNodeIds = conf.getTrimmedStringCollection(decommNodesKey);
        nodeIds.removeAll(decommNodeIds);
        return nodeIds;
    }

    private static Collection<String> getAllOMNodeIds(ConfigurationSource conf, String omServiceId) {
        HashSet<String> nodeIds = new HashSet<String>();
        String nodeIdsKey = ConfUtils.addSuffix((String)"ozone.om.nodes", (String)omServiceId);
        String decommNodesKey = ConfUtils.addKeySuffixes((String)"ozone.om.decommissioned.nodes", (String[])new String[]{omServiceId});
        nodeIds.addAll(conf.getTrimmedStringCollection(nodeIdsKey));
        nodeIds.addAll(conf.getTrimmedStringCollection(decommNodesKey));
        return nodeIds;
    }

    public static Collection<String> emptyAsSingletonNull(Collection<String> coll) {
        if (coll == null || coll.isEmpty()) {
            return Collections.singletonList(null);
        }
        return coll;
    }

    public static String getHttpAddressForOMPeerNode(ConfigurationSource conf, String omServiceId, String omNodeId, String omNodeHostAddr) {
        Optional bindHost = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.http-bind-host", (String[])new String[]{omServiceId, omNodeId})});
        OptionalInt addressPort = HddsUtils.getPortNumberFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.http-address", (String[])new String[]{omServiceId, omNodeId})});
        Optional addressHost = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.http-address", (String[])new String[]{omServiceId, omNodeId})});
        String hostName = bindHost.orElse(addressHost.orElse(omNodeHostAddr));
        return hostName + ":" + addressPort.orElse(9874);
    }

    public static String getHttpsAddressForOMPeerNode(ConfigurationSource conf, String omServiceId, String omNodeId, String omNodeHostAddr) {
        Optional bindHost = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.https-bind-host", (String[])new String[]{omServiceId, omNodeId})});
        OptionalInt addressPort = HddsUtils.getPortNumberFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.https-address", (String[])new String[]{omServiceId, omNodeId})});
        Optional addressHost = HddsUtils.getHostNameFromConfigKeys((ConfigurationSource)conf, (String[])new String[]{ConfUtils.addKeySuffixes((String)"ozone.om.https-address", (String[])new String[]{omServiceId, omNodeId})});
        String hostName = bindHost.orElse(addressHost.orElse(omNodeHostAddr));
        return hostName + ":" + addressPort.orElse(9875);
    }

    public static File createOMDir(String dirPath) {
        File dirFile = new File(dirPath);
        if (!dirFile.mkdirs() && !dirFile.exists()) {
            throw new IllegalArgumentException("Unable to create path: " + dirFile);
        }
        return dirFile;
    }

    public static RepeatedOmKeyInfo prepareKeyForDelete(OmKeyInfo keyInfo, long trxnLogIndex, boolean isRatisEnabled) {
        if (Boolean.parseBoolean(keyInfo.getMetadata().get("gdprEnabled"))) {
            keyInfo.getMetadata().remove("gdprEnabled");
            keyInfo.getMetadata().remove("algorithm");
            keyInfo.getMetadata().remove("secret");
            keyInfo.clearFileEncryptionInfo();
        }
        keyInfo.setUpdateID(trxnLogIndex, isRatisEnabled);
        return new RepeatedOmKeyInfo(keyInfo);
    }

    public static void validateVolumeName(String volumeName, boolean isStrictS3) throws OMException {
        try {
            HddsClientUtils.verifyResourceName((String)volumeName, (boolean)isStrictS3);
        }
        catch (IllegalArgumentException e) {
            throw new OMException("Invalid volume name: " + volumeName, OMException.ResultCodes.INVALID_VOLUME_NAME);
        }
    }

    public static void validateBucketName(String bucketName, boolean isStrictS3) throws OMException {
        try {
            HddsClientUtils.verifyResourceName((String)bucketName, (boolean)isStrictS3);
        }
        catch (IllegalArgumentException e) {
            throw new OMException("Invalid bucket name: " + bucketName, OMException.ResultCodes.INVALID_BUCKET_NAME);
        }
    }

    public static BucketLayout validateBucketLayout(String bucketLayoutString) {
        boolean bucketLayoutValid = Arrays.stream(BucketLayout.values()).anyMatch(layout -> layout.name().equals(bucketLayoutString));
        if (bucketLayoutValid) {
            return BucketLayout.fromString(bucketLayoutString);
        }
        throw new ConfigurationException(bucketLayoutString + " is not a valid default bucket layout. Supported values are " + Arrays.stream(BucketLayout.values()).map(Enum::toString).collect(Collectors.joining(", ")));
    }

    public static void validateSnapshotName(String snapshotName) throws OMException {
        if (snapshotName == null) {
            return;
        }
        try {
            HddsClientUtils.verifyResourceName((String)snapshotName);
        }
        catch (IllegalArgumentException e) {
            throw new OMException("Invalid snapshot name: " + snapshotName, OMException.ResultCodes.INVALID_SNAPSHOT_ERROR);
        }
    }

    public static long getOMClientRpcTimeOut(ConfigurationSource configuration) {
        return ((OMClientConfig)configuration.getObject(OMClientConfig.class)).getRpcTimeOut();
    }

    public static int getOMEpoch(boolean isRatisEnabled) {
        return isRatisEnabled ? 2 : 1;
    }

    public static long getObjectIdFromTxId(long epoch, long txId) {
        Preconditions.checkArgument((txId <= 0x3FFFFFFFFFFFFEL ? 1 : 0) != 0, (Object)"TransactionID exceeds max limit of 18014398509481982");
        return OmUtils.addEpochToTxId(epoch, txId);
    }

    public static long addEpochToTxId(long epoch, long txId) {
        long lsb54 = txId << 8;
        long msb2 = epoch << 62;
        return msb2 | lsb54;
    }

    @VisibleForTesting
    public static long getTxIdFromObjectId(long objectId) {
        return (0x1FFFFFFFFFFFFFFFL & objectId) >> 8;
    }

    public static void validateKeyName(String keyName) throws OMException {
        try {
            HddsClientUtils.verifyKeyName((String)keyName);
        }
        catch (IllegalArgumentException e) {
            throw new OMException(e.getMessage(), OMException.ResultCodes.INVALID_KEY_NAME);
        }
    }

    public static void verifyKeyNameWithSnapshotReservedWord(String keyName) throws OMException {
        if (keyName != null && keyName.startsWith(".snapshot")) {
            if (keyName.length() > ".snapshot".length()) {
                if (keyName.substring(".snapshot".length()).startsWith("/")) {
                    throw new OMException("Cannot create key under path reserved for snapshot: .snapshot/", OMException.ResultCodes.INVALID_KEY_NAME);
                }
            } else {
                throw new OMException("Cannot create key with reserved name: .snapshot", OMException.ResultCodes.INVALID_KEY_NAME);
            }
        }
    }

    public static String getOzoneManagerServiceId(OzoneConfiguration conf) throws IOException {
        String localOMServiceId = conf.get("ozone.om.internal.service.id");
        List<String> omServiceIds = conf.getTrimmedStringCollection("ozone.om.service.ids");
        if (localOMServiceId == null) {
            LOG.info("{} is not defined, falling back to {} to find serviceID for OzoneManager if it is HA enabled cluster", (Object)"ozone.om.internal.service.id", (Object)"ozone.om.service.ids");
            if (omServiceIds.size() > 1) {
                throw new IOException(String.format("More than 1 OzoneManager ServiceID (%s) configured : %s, but %s is not configured.", "ozone.om.service.ids", omServiceIds.toString(), "ozone.om.internal.service.id"));
            }
        } else {
            if (!omServiceIds.contains(localOMServiceId)) {
                throw new IOException(String.format("Cannot find the internal service id %s in %s", localOMServiceId, omServiceIds.toString()));
            }
            omServiceIds = Collections.singletonList(localOMServiceId);
        }
        if (omServiceIds.isEmpty()) {
            LOG.info("No OzoneManager ServiceID configured.");
            return null;
        }
        String serviceId = (String)omServiceIds.iterator().next();
        LOG.info("Using OzoneManager ServiceID '{}'.", (Object)serviceId);
        return serviceId;
    }

    public static String normalizeKey(String keyName, boolean preserveTrailingSlash) {
        if (!StringUtils.isBlank((CharSequence)keyName)) {
            String normalizedKeyName = keyName.startsWith("/") ? new Path(keyName).toUri().getPath() : new Path("/" + keyName).toUri().getPath();
            if (!keyName.equals(normalizedKeyName) && LOG.isDebugEnabled()) {
                LOG.debug("Normalized key {} to {} ", (Object)keyName, (Object)normalizedKeyName.substring(1));
            }
            if (preserveTrailingSlash && keyName.endsWith("/")) {
                return normalizedKeyName.substring(1) + "/";
            }
            return normalizedKeyName.substring(1);
        }
        return keyName;
    }

    public static String normalizePathUptoBucket(String path) {
        if (path == null || path.isEmpty()) {
            return "/";
        }
        path = path.replaceAll("^/*", "");
        String[] segments = path.split("/", -1);
        String volumeName = segments[0];
        String bucketName = segments.length > 1 ? segments[1] : "";
        StringBuilder normalizedPath = new StringBuilder(volumeName);
        if (!bucketName.isEmpty()) {
            normalizedPath.append("/").append(bucketName);
        }
        if (segments.length > 2) {
            normalizedPath.append("/").append(String.join((CharSequence)"/", Arrays.copyOfRange(segments, 2, segments.length)));
        }
        return normalizedPath.toString();
    }

    public static Set<String> getOmHostsFromConfig(OzoneConfiguration conf, String omServiceId) {
        Collection<String> omNodeIds = OmUtils.getActiveOMNodeIds((ConfigurationSource)conf, omServiceId);
        HashSet<String> omHosts = new HashSet<String>();
        for (String nodeId : OmUtils.emptyAsSingletonNull(omNodeIds)) {
            String rpcAddrKey = ConfUtils.addKeySuffixes((String)"ozone.om.address", (String[])new String[]{omServiceId, nodeId});
            String rpcAddrStr = OmUtils.getOmRpcAddress((ConfigurationSource)conf, rpcAddrKey);
            Optional hostName = HddsUtils.getHostName((String)rpcAddrStr);
            hostName.ifPresent(omHosts::add);
        }
        return omHosts;
    }

    public static List<OMNodeDetails> getAllOMHAAddresses(OzoneConfiguration conf, String omServiceId, boolean includeDecommissionedNodes) {
        ArrayList<OMNodeDetails> omNodesList = new ArrayList<OMNodeDetails>();
        Collection<String> omNodeIds = includeDecommissionedNodes ? OmUtils.getAllOMNodeIds((ConfigurationSource)conf, omServiceId) : OmUtils.getActiveOMNodeIds((ConfigurationSource)conf, omServiceId);
        Collection decommissionedNodeIds = conf.getTrimmedStringCollection(ConfUtils.addKeySuffixes((String)"ozone.om.decommissioned.nodes", (String[])new String[]{omServiceId}));
        if (omNodeIds.isEmpty()) {
            return Collections.emptyList();
        }
        for (String nodeId : omNodeIds) {
            try {
                OMNodeDetails omNodeDetails = OMNodeDetails.getOMNodeDetailsFromConf(conf, omServiceId, nodeId);
                if (decommissionedNodeIds.contains(omNodeDetails.getNodeId())) {
                    omNodeDetails.setDecommissioningState();
                }
                omNodesList.add(omNodeDetails);
            }
            catch (IOException e) {
                String omRpcAddressStr = OMNodeDetails.getOMNodeAddressFromConf(conf, omServiceId, nodeId);
                LOG.error("OM {} is present in config file but it's address {} could not be resolved. Hence, OM {} is not added to list of peer nodes.", new Object[]{nodeId, omRpcAddressStr, nodeId});
            }
        }
        return omNodesList;
    }

    public static String getOMAddressListPrintString(List<OMNodeDetails> omList) {
        if (omList.isEmpty()) {
            return null;
        }
        StringBuilder printString = new StringBuilder();
        printString.append("OM");
        if (omList.size() == 1) {
            printString.append(" [");
        } else {
            printString.append("(s) [");
        }
        printString.append(omList.get(0).getOMPrintInfo());
        for (int i = 1; i < omList.size(); ++i) {
            printString.append(",").append(omList.get(i).getOMPrintInfo());
        }
        printString.append("]");
        return printString.toString();
    }

    public static boolean isBucketSnapshotIndicator(String key) {
        return key.startsWith(".snapshot") && key.split("/").length == 2;
    }

    public static List<List<String>> format(List<ServiceInfo> nodes, int port, String leaderId) {
        ArrayList<List<String>> omInfoList = new ArrayList<List<String>>();
        List omNodes = nodes.stream().filter(node -> node.getNodeType() == HddsProtos.NodeType.OM).sorted(Comparator.comparing(ServiceInfo::getHostname)).collect(Collectors.toList());
        for (ServiceInfo info : omNodes) {
            if (info.getNodeType() != HddsProtos.NodeType.OM) continue;
            String role = info.getOmRoleInfo().getNodeId().equals(leaderId) ? "LEADER" : "FOLLOWER";
            ArrayList<String> omInfo = new ArrayList<String>();
            omInfo.add(info.getHostname());
            omInfo.add(info.getOmRoleInfo().getNodeId());
            omInfo.add(String.valueOf(port));
            omInfo.add(role);
            omInfoList.add(omInfo);
        }
        return omInfoList;
    }

    public static void resolveOmHost(String omHost, int omPort) throws IOException {
        InetSocketAddress omHostAddress = NetUtils.createSocketAddr((String)omHost, (int)omPort);
        if (omHostAddress.isUnresolved()) {
            throw new IOException("Cannot resolve OM host " + omHost + " in the URI", new UnknownHostException());
        }
        try {
            if (!omHostAddress.getAddress().isReachable(5000)) {
                throw new IOException("OM host " + omHost + " unreachable in the URI");
            }
        }
        catch (IOException e) {
            LOG.error("Failure in resolving OM host address", (Throwable)e);
            throw e;
        }
    }
}

