/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.replication;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.conf.Config;
import org.apache.hadoop.hdds.conf.ConfigGroup;
import org.apache.hadoop.hdds.conf.ConfigTag;
import org.apache.hadoop.hdds.conf.ConfigType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.PostConstruct;
import org.apache.hadoop.hdds.conf.ReconfigurableConfig;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ReplicationManagerReport;
import org.apache.hadoop.hdds.scm.container.balancer.MoveManager;
import org.apache.hadoop.hdds.scm.container.replication.AbstractOverReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.CommandTargetOverloadedException;
import org.apache.hadoop.hdds.scm.container.replication.ContainerCheckRequest;
import org.apache.hadoop.hdds.scm.container.replication.ContainerHealthResult;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaOp;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaPendingOps;
import org.apache.hadoop.hdds.scm.container.replication.ECContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ECMisReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.ECOverReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.ECUnderReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.LegacyReplicationManager;
import org.apache.hadoop.hdds.scm.container.replication.NullReplicationQueue;
import org.apache.hadoop.hdds.scm.container.replication.OverReplicatedProcessor;
import org.apache.hadoop.hdds.scm.container.replication.RatisContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.RatisMisReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.RatisOverReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.RatisUnderReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerMetrics;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationQueue;
import org.apache.hadoop.hdds.scm.container.replication.UnderReplicatedProcessor;
import org.apache.hadoop.hdds.scm.container.replication.UnhealthyReplicationHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.ClosedWithUnhealthyReplicasHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.ClosingContainerHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.DeletingContainerHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.ECMisReplicationCheckHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.ECReplicationCheckHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.EmptyContainerHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.HealthCheck;
import org.apache.hadoop.hdds.scm.container.replication.health.MismatchedReplicasHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.OpenContainerHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.QuasiClosedContainerHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.RatisReplicationCheckHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.RatisUnhealthyReplicationCheckHandler;
import org.apache.hadoop.hdds.scm.container.replication.health.VulnerableUnhealthyReplicasHandler;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMService;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.container.replication.ReplicationServer;
import org.apache.hadoop.ozone.protocol.commands.CloseContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.DeleteContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.ReconstructECContainersCommand;
import org.apache.hadoop.ozone.protocol.commands.ReplicateContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.ExitUtil;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicationManager
implements SCMService {
    public static final Logger LOG = LoggerFactory.getLogger(ReplicationManager.class);
    private final ContainerManager containerManager;
    private final SCMContext scmContext;
    private final ReplicationManagerConfiguration rmConf;
    private final ReplicationServer.ReplicationConfig replicationServerConf;
    private final NodeManager nodeManager;
    private Thread replicationMonitor;
    private volatile boolean running;
    private ReplicationManagerReport containerReport;
    private ReplicationManagerMetrics metrics;
    private final LegacyReplicationManager legacyReplicationManager;
    private final Map<DatanodeDetails, Integer> excludedNodes = new ConcurrentHashMap<DatanodeDetails, Integer>();
    private final Lock serviceLock = new ReentrantLock();
    private SCMService.ServiceStatus serviceStatus = SCMService.ServiceStatus.PAUSING;
    private final long waitTimeInMillis;
    private long lastTimeToBeReadyInMillis = 0L;
    private final Clock clock;
    private final ContainerReplicaPendingOps containerReplicaPendingOps;
    private final ECReplicationCheckHandler ecReplicationCheckHandler;
    private final ECMisReplicationCheckHandler ecMisReplicationCheckHandler;
    private final RatisReplicationCheckHandler ratisReplicationCheckHandler;
    private final EventPublisher eventPublisher;
    private final AtomicReference<ReplicationQueue> replicationQueue = new AtomicReference<ReplicationQueue>(new ReplicationQueue());
    private final ECUnderReplicationHandler ecUnderReplicationHandler;
    private final ECOverReplicationHandler ecOverReplicationHandler;
    private final ECMisReplicationHandler ecMisReplicationHandler;
    private final RatisUnderReplicationHandler ratisUnderReplicationHandler;
    private final RatisOverReplicationHandler ratisOverReplicationHandler;
    private final RatisMisReplicationHandler ratisMisReplicationHandler;
    private Thread underReplicatedProcessorThread;
    private Thread overReplicatedProcessorThread;
    private final UnderReplicatedProcessor underReplicatedProcessor;
    private final OverReplicatedProcessor overReplicatedProcessor;
    private final HealthCheck containerCheckChain;
    private final ReplicationQueue nullReplicationQueue = new NullReplicationQueue();

    public ReplicationManager(ConfigurationSource conf, ContainerManager containerManager, PlacementPolicy ratisContainerPlacement, PlacementPolicy ecContainerPlacement, EventPublisher eventPublisher, SCMContext scmContext, NodeManager nodeManager, Clock clock, LegacyReplicationManager legacyReplicationManager, ContainerReplicaPendingOps replicaPendingOps) throws IOException {
        this.containerManager = containerManager;
        this.scmContext = scmContext;
        this.rmConf = (ReplicationManagerConfiguration)((Object)conf.getObject(ReplicationManagerConfiguration.class));
        this.replicationServerConf = (ReplicationServer.ReplicationConfig)conf.getObject(ReplicationServer.ReplicationConfig.class);
        this.running = false;
        this.clock = clock;
        this.containerReport = new ReplicationManagerReport();
        this.eventPublisher = eventPublisher;
        this.waitTimeInMillis = conf.getTimeDuration("hdds.scm.wait.time.after.safemode.exit", "5m", TimeUnit.MILLISECONDS);
        this.containerReplicaPendingOps = replicaPendingOps;
        this.legacyReplicationManager = legacyReplicationManager;
        this.ecReplicationCheckHandler = new ECReplicationCheckHandler();
        this.ecMisReplicationCheckHandler = new ECMisReplicationCheckHandler(ecContainerPlacement);
        this.ratisReplicationCheckHandler = new RatisReplicationCheckHandler(ratisContainerPlacement, this);
        this.nodeManager = nodeManager;
        this.metrics = ReplicationManagerMetrics.create(this);
        this.ecUnderReplicationHandler = new ECUnderReplicationHandler(ecContainerPlacement, conf, this);
        this.ecOverReplicationHandler = new ECOverReplicationHandler(ecContainerPlacement, this);
        this.ecMisReplicationHandler = new ECMisReplicationHandler(ecContainerPlacement, conf, this);
        this.ratisUnderReplicationHandler = new RatisUnderReplicationHandler(ratisContainerPlacement, conf, this);
        this.ratisOverReplicationHandler = new RatisOverReplicationHandler(ratisContainerPlacement, this);
        this.ratisMisReplicationHandler = new RatisMisReplicationHandler(ratisContainerPlacement, conf, this);
        this.underReplicatedProcessor = new UnderReplicatedProcessor(this, this.rmConf::getUnderReplicatedInterval);
        this.overReplicatedProcessor = new OverReplicatedProcessor(this, this.rmConf::getOverReplicatedInterval);
        this.containerCheckChain = new OpenContainerHandler(this);
        this.containerCheckChain.addNext(new ClosingContainerHandler(this, clock)).addNext(new QuasiClosedContainerHandler(this)).addNext(new MismatchedReplicasHandler(this)).addNext(new EmptyContainerHandler(this)).addNext(new DeletingContainerHandler(this)).addNext(this.ecReplicationCheckHandler).addNext(this.ratisReplicationCheckHandler).addNext(new ClosedWithUnhealthyReplicasHandler(this)).addNext(this.ecMisReplicationCheckHandler).addNext(new RatisUnhealthyReplicationCheckHandler()).addNext(new VulnerableUnhealthyReplicasHandler(this));
        this.start();
    }

    @Override
    public synchronized void start() {
        if (!this.isRunning()) {
            LOG.info("Starting Replication Monitor Thread.");
            this.running = true;
            this.metrics = ReplicationManagerMetrics.create(this);
            if (this.rmConf.isLegacyEnabled()) {
                this.legacyReplicationManager.setMetrics(this.metrics);
            }
            this.containerReplicaPendingOps.setReplicationMetrics(this.metrics);
            this.startSubServices();
        } else {
            LOG.info("Replication Monitor Thread is already running.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isRunning() {
        if (!this.running) {
            ReplicationManager replicationManager = this;
            synchronized (replicationManager) {
                return this.replicationMonitor != null && this.replicationMonitor.isAlive();
            }
        }
        return true;
    }

    @Override
    public synchronized void stop() {
        if (this.running) {
            LOG.info("Stopping Replication Monitor Thread.");
            this.underReplicatedProcessorThread.interrupt();
            this.overReplicatedProcessorThread.interrupt();
            this.running = false;
            if (this.rmConf.isLegacyEnabled()) {
                this.legacyReplicationManager.clearInflightActions();
            }
            this.metrics.unRegister();
            this.replicationMonitor.interrupt();
        } else {
            LOG.info("Replication Monitor Thread is not running.");
        }
    }

    @VisibleForTesting
    protected void startSubServices() {
        String prefix = this.scmContext.threadNamePrefix();
        this.replicationMonitor = new Thread(this::run);
        this.replicationMonitor.setName(prefix + "ReplicationMonitor");
        this.replicationMonitor.setDaemon(true);
        this.replicationMonitor.start();
        this.underReplicatedProcessorThread = new Thread(this.underReplicatedProcessor);
        this.underReplicatedProcessorThread.setName(prefix + "UnderReplicatedProcessor");
        this.underReplicatedProcessorThread.setDaemon(true);
        this.underReplicatedProcessorThread.start();
        this.overReplicatedProcessorThread = new Thread(this.overReplicatedProcessor);
        this.overReplicatedProcessorThread.setName(prefix + "OverReplicatedProcessor");
        this.overReplicatedProcessorThread.setDaemon(true);
        this.overReplicatedProcessorThread.start();
    }

    public synchronized void processAll() {
        if (!this.shouldRun()) {
            LOG.info("Replication Manager is not ready to run until {}ms after safemode exit", (Object)this.waitTimeInMillis);
            return;
        }
        long start = this.clock.millis();
        List<ContainerInfo> containers = this.containerManager.getContainers();
        ReplicationManagerReport report = new ReplicationManagerReport();
        ReplicationQueue newRepQueue = new ReplicationQueue();
        for (ContainerInfo c : containers) {
            if (!this.shouldRun()) break;
            report.increment(c.getState());
            if (this.rmConf.isLegacyEnabled() && !ReplicationManager.isEC(c.getReplicationConfig())) {
                this.legacyReplicationManager.processContainer(c, report);
                continue;
            }
            try {
                this.processContainer(c, newRepQueue, report);
            }
            catch (ContainerNotFoundException e) {
                LOG.error("Container {} not found", (Object)c.getContainerID(), (Object)e);
            }
        }
        report.setComplete();
        this.replicationQueue.set(newRepQueue);
        this.containerReport = report;
        LOG.info("Replication Monitor Thread took {} milliseconds for processing {} containers.", (Object)(this.clock.millis() - start), (Object)containers.size());
    }

    public void sendCloseContainerEvent(ContainerID containerID) {
        this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)containerID);
    }

    public long getReplicationInFlightLimit() {
        double factor = this.rmConf.getInflightReplicationLimitFactor();
        if (factor <= 0.0) {
            return 0L;
        }
        int healthyNodes = this.nodeManager.getNodeCount(null, HddsProtos.NodeState.HEALTHY);
        return (long)Math.ceil((double)(healthyNodes * this.rmConf.getDatanodeReplicationLimit()) * factor);
    }

    public long getInflightReplicationCount() {
        return this.containerReplicaPendingOps.getPendingOpCount(ContainerReplicaOp.PendingOpType.ADD);
    }

    public void sendDeleteCommand(ContainerInfo container, int replicaIndex, DatanodeDetails datanode, boolean force) throws NotLeaderException {
        LOG.debug("Sending delete command for container {} and index {} on {}", new Object[]{container, replicaIndex, datanode});
        DeleteContainerCommand deleteCommand = new DeleteContainerCommand(container.containerID(), force);
        deleteCommand.setReplicaIndex(replicaIndex);
        this.sendDatanodeCommand((SCMCommand<?>)deleteCommand, container, datanode);
    }

    public void sendDeleteCommand(ContainerInfo container, int replicaIndex, DatanodeDetails datanode, boolean force, long scmDeadlineEpochMs) throws NotLeaderException {
        LOG.debug("Sending delete command for container {} and index {} on {} with SCM deadline {}.", new Object[]{container, replicaIndex, datanode, scmDeadlineEpochMs});
        DeleteContainerCommand deleteCommand = new DeleteContainerCommand(container.containerID(), force);
        deleteCommand.setReplicaIndex(replicaIndex);
        this.sendDatanodeCommand((SCMCommand<?>)deleteCommand, container, datanode, scmDeadlineEpochMs);
    }

    public void sendThrottledDeleteCommand(ContainerInfo container, int replicaIndex, DatanodeDetails datanode, boolean force) throws NotLeaderException, CommandTargetOverloadedException {
        try {
            int commandCount = this.nodeManager.getTotalDatanodeCommandCount(datanode, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.deleteContainerCommand);
            int deleteLimit = this.rmConf.getDatanodeDeleteLimit();
            if (commandCount >= deleteLimit) {
                this.metrics.incrDeleteContainerCmdsDeferredTotal();
                throw new CommandTargetOverloadedException("Cannot schedule a delete container command for container " + container.containerID() + " on datanode " + datanode + " as it has too many pending delete commands (" + commandCount + " > " + deleteLimit + ")");
            }
            this.sendDeleteCommand(container, replicaIndex, datanode, force);
        }
        catch (NodeNotFoundException e) {
            throw new IllegalArgumentException("Datanode " + datanode + " not found in NodeManager. Should not happen");
        }
    }

    public void sendThrottledReplicationCommand(ContainerInfo containerInfo, List<DatanodeDetails> sources, DatanodeDetails target, int replicaIndex) throws CommandTargetOverloadedException, NotLeaderException {
        long containerID = containerInfo.getContainerID();
        List<Pair<Integer, DatanodeDetails>> sourceWithCmds = this.getAvailableDatanodesForReplication(sources);
        if (sourceWithCmds.isEmpty()) {
            this.metrics.incrReplicateContainerCmdsDeferredTotal();
            throw new CommandTargetOverloadedException("No sources with capacity available for replication of container " + containerID + " to " + target);
        }
        DatanodeDetails source = this.selectAndOptionallyExcludeDatanode(1, sourceWithCmds);
        ReplicateContainerCommand cmd = ReplicateContainerCommand.toTarget((long)containerID, (DatanodeDetails)target);
        cmd.setReplicaIndex(replicaIndex);
        this.sendDatanodeCommand((SCMCommand<?>)cmd, containerInfo, source);
    }

    public void sendThrottledReconstructionCommand(ContainerInfo containerInfo, ReconstructECContainersCommand command) throws CommandTargetOverloadedException, NotLeaderException {
        List targets = command.getTargetDatanodes();
        List<Pair<Integer, DatanodeDetails>> targetWithCmds = this.getAvailableDatanodesForReplication(targets);
        if (targetWithCmds.isEmpty()) {
            this.metrics.incrECReconstructionCmdsDeferredTotal();
            throw new CommandTargetOverloadedException("No target with capacity available for reconstruction of " + containerInfo.getContainerID());
        }
        DatanodeDetails target = this.selectAndOptionallyExcludeDatanode(this.rmConf.getReconstructionCommandWeight(), targetWithCmds);
        this.sendDatanodeCommand((SCMCommand<?>)command, containerInfo, target);
    }

    private DatanodeDetails selectAndOptionallyExcludeDatanode(int additionalCmdCount, List<Pair<Integer, DatanodeDetails>> datanodes) {
        if (datanodes.isEmpty()) {
            return null;
        }
        datanodes.sort(Comparator.comparingInt(Pair::getLeft));
        DatanodeDetails datanode = (DatanodeDetails)datanodes.get(0).getRight();
        int currentCount = (Integer)datanodes.get(0).getLeft();
        if (currentCount + additionalCmdCount >= this.getReplicationLimit(datanode)) {
            this.addExcludedNode(datanode);
        }
        return datanode;
    }

    private List<Pair<Integer, DatanodeDetails>> getAvailableDatanodesForReplication(List<DatanodeDetails> datanodes) {
        ArrayList<Pair<Integer, DatanodeDetails>> datanodeWithCommandCount = new ArrayList<Pair<Integer, DatanodeDetails>>();
        for (DatanodeDetails dn : datanodes) {
            try {
                int totalCount = this.getQueuedReplicationCount(dn);
                int replicationLimit = this.getReplicationLimit(dn);
                if (totalCount >= replicationLimit) {
                    LOG.debug("Datanode {} has reached the maximum of {} queued commands for state {}: {}", new Object[]{dn, replicationLimit, dn.getPersistedOpState(), totalCount});
                    this.addExcludedNode(dn);
                    continue;
                }
                datanodeWithCommandCount.add((Pair<Integer, DatanodeDetails>)Pair.of((Object)totalCount, (Object)dn));
            }
            catch (NodeNotFoundException e) {
                LOG.error("Node {} not found in NodeManager. Should not happen", (Object)dn, (Object)e);
            }
        }
        return datanodeWithCommandCount;
    }

    private int getQueuedReplicationCount(DatanodeDetails datanode) throws NodeNotFoundException {
        Map<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> counts = this.nodeManager.getTotalDatanodeCommandCounts(datanode, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.replicateContainerCommand, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.reconstructECContainersCommand);
        int replicateCount = counts.get(StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.replicateContainerCommand);
        int reconstructCount = counts.get(StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.reconstructECContainersCommand);
        return replicateCount + reconstructCount * this.rmConf.getReconstructionCommandWeight();
    }

    public void sendLowPriorityReplicateContainerCommand(ContainerInfo container, int replicaIndex, DatanodeDetails source, DatanodeDetails target, long scmDeadlineEpochMs) throws NotLeaderException {
        ReplicateContainerCommand command = ReplicateContainerCommand.toTarget((long)container.getContainerID(), (DatanodeDetails)target);
        command.setReplicaIndex(replicaIndex);
        command.setPriority(StorageContainerDatanodeProtocolProtos.ReplicationCommandPriority.LOW);
        this.sendDatanodeCommand((SCMCommand<?>)command, container, source, scmDeadlineEpochMs);
    }

    public void sendDatanodeCommand(SCMCommand<?> command, ContainerInfo containerInfo, DatanodeDetails target) throws NotLeaderException {
        long scmDeadline = this.clock.millis() + this.rmConf.eventTimeout;
        this.sendDatanodeCommand(command, containerInfo, target, scmDeadline);
    }

    public void sendDatanodeCommand(SCMCommand<?> command, ContainerInfo containerInfo, DatanodeDetails target, long scmDeadlineEpochMs) throws NotLeaderException {
        long datanodeDeadline = scmDeadlineEpochMs - this.rmConf.getDatanodeTimeoutOffset();
        LOG.info("Sending command [{}] for container {} to {} with datanode deadline {} and scm deadline {}", new Object[]{command, containerInfo, target, datanodeDeadline, scmDeadlineEpochMs});
        command.setTerm(this.getScmTerm());
        command.setDeadline(datanodeDeadline);
        this.nodeManager.addDatanodeCommand(target.getUuid(), command);
        this.adjustPendingOpsAndMetrics(containerInfo, command, target, scmDeadlineEpochMs);
    }

    private void adjustPendingOpsAndMetrics(ContainerInfo containerInfo, SCMCommand<?> cmd, DatanodeDetails targetDatanode, long scmDeadlineEpochMs) {
        if (cmd.getType() == StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.deleteContainerCommand) {
            DeleteContainerCommand rcc = (DeleteContainerCommand)cmd;
            this.containerReplicaPendingOps.scheduleDeleteReplica(containerInfo.containerID(), targetDatanode, rcc.getReplicaIndex(), scmDeadlineEpochMs);
            if (rcc.getReplicaIndex() > 0) {
                this.getMetrics().incrEcDeletionCmdsSentTotal();
            } else if (rcc.getReplicaIndex() == 0) {
                this.getMetrics().incrDeletionCmdsSentTotal();
                this.getMetrics().incrDeletionBytesTotal(containerInfo.getUsedBytes());
            }
        } else if (cmd.getType() == StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.reconstructECContainersCommand) {
            ReconstructECContainersCommand rcc = (ReconstructECContainersCommand)cmd;
            List targets = rcc.getTargetDatanodes();
            byte[] targetIndexes = rcc.getMissingContainerIndexes();
            for (int i = 0; i < targetIndexes.length; ++i) {
                this.containerReplicaPendingOps.scheduleAddReplica(containerInfo.containerID(), (DatanodeDetails)targets.get(i), targetIndexes[i], scmDeadlineEpochMs);
            }
            this.getMetrics().incrEcReconstructionCmdsSentTotal();
        } else if (cmd.getType() == StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.replicateContainerCommand) {
            ReplicateContainerCommand rcc = (ReplicateContainerCommand)cmd;
            if (rcc.getTargetDatanode() == null) {
                this.containerReplicaPendingOps.scheduleAddReplica(containerInfo.containerID(), targetDatanode, rcc.getReplicaIndex(), scmDeadlineEpochMs);
            } else {
                this.containerReplicaPendingOps.scheduleAddReplica(containerInfo.containerID(), rcc.getTargetDatanode(), rcc.getReplicaIndex(), scmDeadlineEpochMs);
            }
            if (rcc.getReplicaIndex() > 0) {
                this.getMetrics().incrEcReplicationCmdsSentTotal();
            } else if (rcc.getReplicaIndex() == 0) {
                this.getMetrics().incrReplicationCmdsSentTotal();
            }
        }
    }

    public void updateContainerState(ContainerID containerID, HddsProtos.LifeCycleEvent event) {
        try {
            this.containerManager.updateContainerState(containerID, event);
        }
        catch (IOException | InvalidStateTransitionException e) {
            LOG.error("Failed to update the state of container {}, update Event {}", new Object[]{containerID, event, e});
        }
    }

    int processUnderReplicatedContainer(ContainerHealthResult result) throws IOException {
        UnhealthyReplicationHandler handler;
        ContainerID containerID = result.getContainerInfo().containerID();
        Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(containerID);
        List<ContainerReplicaOp> pendingOps = this.containerReplicaPendingOps.getPendingOps(containerID);
        boolean isEC = ReplicationManager.isEC(result.getContainerInfo().getReplicationConfig());
        if (result.getHealthState() == ContainerHealthResult.HealthState.UNDER_REPLICATED) {
            handler = isEC ? this.ecUnderReplicationHandler : this.ratisUnderReplicationHandler;
        } else if (result.getHealthState() == ContainerHealthResult.HealthState.MIS_REPLICATED) {
            handler = isEC ? this.ecMisReplicationHandler : this.ratisMisReplicationHandler;
        } else {
            throw new IllegalArgumentException("Unexpected health state: " + (Object)((Object)result.getHealthState()));
        }
        return handler.processAndSendCommands(replicas, pendingOps, result, this.getRemainingMaintenanceRedundancy(isEC));
    }

    int processOverReplicatedContainer(ContainerHealthResult result) throws IOException {
        ContainerID containerID = result.getContainerInfo().containerID();
        Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(containerID);
        List<ContainerReplicaOp> pendingOps = this.containerReplicaPendingOps.getPendingOps(containerID);
        boolean isEC = ReplicationManager.isEC(result.getContainerInfo().getReplicationConfig());
        AbstractOverReplicationHandler handler = isEC ? this.ecOverReplicationHandler : this.ratisOverReplicationHandler;
        return handler.processAndSendCommands(replicas, pendingOps, result, this.getRemainingMaintenanceRedundancy(isEC));
    }

    public long getScmTerm() throws NotLeaderException {
        return this.scmContext.getTermOfLeader();
    }

    public void datanodeCommandCountUpdated(DatanodeDetails datanode) {
        LOG.trace("Received a notification that the DN command count has been updated for {}", (Object)datanode);
        this.excludedNodes.computeIfPresent(datanode, (dn, v) -> {
            try {
                if (this.getQueuedReplicationCount((DatanodeDetails)dn) < this.getReplicationLimit((DatanodeDetails)dn)) {
                    return null;
                }
                return 1;
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Unable to find datanode {} in nodeManager. Should not happen.", (Object)datanode);
                return null;
            }
        });
    }

    public Set<DatanodeDetails> getExcludedNodes() {
        return this.excludedNodes.keySet();
    }

    private void addExcludedNode(DatanodeDetails dn) {
        this.excludedNodes.put(dn, 1);
    }

    protected void processContainer(ContainerInfo containerInfo, ReplicationQueue repQueue, ReplicationManagerReport report) throws ContainerNotFoundException {
        this.processContainer(containerInfo, repQueue, report, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean processContainer(ContainerInfo containerInfo, ReplicationQueue repQueue, ReplicationManagerReport report, boolean readOnly) throws ContainerNotFoundException {
        ContainerInfo containerInfo2 = containerInfo;
        synchronized (containerInfo2) {
            ContainerID containerID = containerInfo.containerID();
            boolean isEC = ReplicationManager.isEC(containerInfo.getReplicationConfig());
            Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(containerID);
            List<ContainerReplicaOp> pendingOps = this.containerReplicaPendingOps.getPendingOps(containerID);
            ContainerCheckRequest checkRequest = new ContainerCheckRequest.Builder().setContainerInfo(containerInfo).setContainerReplicas(replicas).setMaintenanceRedundancy(this.getRemainingMaintenanceRedundancy(isEC)).setReport(report).setPendingOps(pendingOps).setReplicationQueue(repQueue).setReadOnly(readOnly).build();
            boolean handled = this.containerCheckChain.handleChain(checkRequest);
            if (!handled) {
                LOG.debug("Container {} had no actions after passing through the check chain", (Object)containerInfo.containerID());
            }
            return handled;
        }
    }

    public void sendCloseContainerReplicaCommand(ContainerInfo container, DatanodeDetails datanode, boolean force) {
        ContainerID containerID = container.containerID();
        CloseContainerCommand closeContainerCommand = new CloseContainerCommand(container.getContainerID(), container.getPipelineID(), force);
        closeContainerCommand.setEncodedToken(this.getContainerToken(containerID));
        try {
            this.sendDatanodeCommand((SCMCommand<?>)closeContainerCommand, container, datanode);
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending close container command, since current SCM is not leader.", (Throwable)nle);
        }
    }

    private String getContainerToken(ContainerID containerID) {
        if (this.scmContext.getScm() instanceof StorageContainerManager) {
            StorageContainerManager scm = (StorageContainerManager)this.scmContext.getScm();
            return scm.getContainerTokenGenerator().generateEncodedToken(containerID);
        }
        return "";
    }

    public ReplicationManagerReport getContainerReport() {
        return this.containerReport;
    }

    private synchronized void run() {
        try {
            while (this.running) {
                this.processAll();
                this.wait(this.rmConf.getInterval().toMillis());
            }
        }
        catch (Throwable t) {
            if (t instanceof InterruptedException) {
                LOG.info("Replication Monitor Thread is stopped");
                Thread.currentThread().interrupt();
            }
            LOG.error("Exception in Replication Monitor Thread.", t);
            ExitUtil.terminate((int)1, (Throwable)t);
        }
    }

    public ContainerReplicaCount getContainerReplicaCount(ContainerID containerID) throws ContainerNotFoundException {
        ContainerInfo container = this.containerManager.getContainer(containerID);
        boolean isEC = ReplicationManager.isEC(container.getReplicationConfig());
        if (!isEC && this.rmConf.isLegacyEnabled()) {
            return this.legacyReplicationManager.getContainerReplicaCount(container);
        }
        return this.getContainerReplicaCount(container, isEC);
    }

    public ContainerHealthResult getContainerReplicationHealth(ContainerInfo containerInfo, Set<ContainerReplica> replicas) {
        boolean isEC = ReplicationManager.isEC(containerInfo.getReplicationConfig());
        ContainerCheckRequest request = new ContainerCheckRequest.Builder().setContainerInfo(containerInfo).setContainerReplicas(replicas).setPendingOps(this.getPendingReplicationOps(containerInfo.containerID())).setMaintenanceRedundancy(this.getRemainingMaintenanceRedundancy(isEC)).build();
        if (isEC) {
            return this.ecReplicationCheckHandler.checkHealth(request);
        }
        return this.ratisReplicationCheckHandler.checkHealth(request);
    }

    public boolean checkContainerStatus(ContainerInfo containerInfo, ReplicationManagerReport report) throws ContainerNotFoundException {
        report.increment(containerInfo.getState());
        return this.processContainer(containerInfo, this.nullReplicationQueue, report, true);
    }

    public List<ContainerReplicaOp> getPendingReplicationOps(ContainerID containerID) {
        return this.containerReplicaPendingOps.getPendingOps(containerID);
    }

    public NodeStatus getNodeStatus(DatanodeDetails datanode) throws NodeNotFoundException {
        return this.nodeManager.getNodeStatus(datanode);
    }

    private boolean isOpenContainerHealthy(ContainerInfo container, Set<ContainerReplica> replicas) {
        HddsProtos.LifeCycleState state = container.getState();
        return replicas.stream().allMatch(r -> ReplicationManager.compareState(state, r.getState()));
    }

    public static boolean compareState(HddsProtos.LifeCycleState containerState, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State replicaState) {
        switch (containerState) {
            case OPEN: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN;
            }
            case CLOSING: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING;
            }
            case QUASI_CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED;
            }
            case CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
            }
        }
        return false;
    }

    ReplicationQueue getQueue() {
        return this.replicationQueue.get();
    }

    @Override
    public void notifyStatusChanged() {
        this.serviceLock.lock();
        try {
            if (this.scmContext.isLeaderReady() && !this.scmContext.isInSafeMode()) {
                if (this.serviceStatus != SCMService.ServiceStatus.RUNNING) {
                    LOG.info("Service {} transitions to RUNNING.", (Object)this.getServiceName());
                    this.lastTimeToBeReadyInMillis = this.clock.millis();
                    this.containerReplicaPendingOps.clear();
                    this.serviceStatus = SCMService.ServiceStatus.RUNNING;
                }
                if (this.rmConf.isLegacyEnabled()) {
                    this.legacyReplicationManager.notifyStatusChanged();
                }
            } else {
                this.serviceStatus = SCMService.ServiceStatus.PAUSING;
            }
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    @Override
    public boolean shouldRun() {
        this.serviceLock.lock();
        try {
            boolean bl = this.serviceStatus == SCMService.ServiceStatus.RUNNING && this.clock.millis() - this.lastTimeToBeReadyInMillis >= this.waitTimeInMillis;
            return bl;
        }
        finally {
            this.serviceLock.unlock();
        }
    }

    @Override
    public String getServiceName() {
        return ReplicationManager.class.getSimpleName();
    }

    public ReplicationManagerMetrics getMetrics() {
        return this.metrics;
    }

    public ReplicationManagerConfiguration getConfig() {
        return this.rmConf;
    }

    public Clock getClock() {
        return this.clock;
    }

    public CompletableFuture<MoveManager.MoveResult> move(ContainerID cid, DatanodeDetails src, DatanodeDetails tgt) throws NodeNotFoundException, ContainerNotFoundException, TimeoutException {
        CompletableFuture<MoveManager.MoveResult> ret = new CompletableFuture<MoveManager.MoveResult>();
        if (!this.isRunning()) {
            ret.complete(MoveManager.MoveResult.FAIL_UNEXPECTED_ERROR);
            LOG.warn("Failing move because Replication Monitor thread's running state is {}", (Object)this.isRunning());
            return ret;
        }
        return this.legacyReplicationManager.move(cid, src, tgt);
    }

    public Map<ContainerID, CompletableFuture<MoveManager.MoveResult>> getInflightMove() {
        return this.legacyReplicationManager.getInflightMove();
    }

    public LegacyReplicationManager.MoveScheduler getMoveScheduler() {
        return this.legacyReplicationManager.getMoveScheduler();
    }

    @VisibleForTesting
    public LegacyReplicationManager getLegacyReplicationManager() {
        return this.legacyReplicationManager;
    }

    public boolean isContainerReplicatingOrDeleting(ContainerID containerID) {
        if (this.rmConf.isLegacyEnabled()) {
            return this.legacyReplicationManager.isContainerReplicatingOrDeleting(containerID);
        }
        return !this.getPendingReplicationOps(containerID).isEmpty();
    }

    private ContainerReplicaCount getContainerReplicaCount(ContainerInfo container, boolean isEC) throws ContainerNotFoundException {
        ContainerID id = container.containerID();
        Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(id);
        List<ContainerReplicaOp> pendingOps = this.containerReplicaPendingOps.getPendingOps(id);
        int redundancy = this.getRemainingMaintenanceRedundancy(isEC);
        return isEC ? new ECContainerReplicaCount(container, replicas, pendingOps, redundancy) : new RatisContainerReplicaCount(container, replicas, pendingOps, redundancy, false);
    }

    public ContainerReplicaPendingOps getContainerReplicaPendingOps() {
        return this.containerReplicaPendingOps;
    }

    private int getReplicationLimit(DatanodeDetails datanode) {
        HddsProtos.NodeOperationalState state = datanode.getPersistedOpState();
        int limit = this.rmConf.getDatanodeReplicationLimit();
        if (DatanodeDetails.isMaintenance((HddsProtos.NodeOperationalState)state) || DatanodeDetails.isDecommission((HddsProtos.NodeOperationalState)state)) {
            limit = this.replicationServerConf.scaleOutOfServiceLimit(limit);
        }
        return limit;
    }

    static NodeStatus getNodeStatus(DatanodeDetails dn, NodeManager nm) {
        try {
            return nm.getNodeStatus(dn);
        }
        catch (NodeNotFoundException e) {
            throw new IllegalStateException("Unable to find NodeStatus for " + dn, e);
        }
    }

    private int getRemainingMaintenanceRedundancy(boolean isEC) {
        return isEC ? this.rmConf.getMaintenanceRemainingRedundancy() : this.rmConf.getMaintenanceReplicaMinimum();
    }

    private static boolean isEC(ReplicationConfig replicationConfig) {
        return replicationConfig.getReplicationType() == HddsProtos.ReplicationType.EC;
    }

    public boolean hasHealthyPipeline(ContainerInfo container) {
        try {
            return this.scmContext.getScm().getPipelineManager().getPipeline(container.getPipelineID()) != null;
        }
        catch (PipelineNotFoundException e) {
            return false;
        }
    }

    @ConfigGroup(prefix="hdds.scm.replication")
    public static class ReplicationManagerConfiguration
    extends ReconfigurableConfig {
        @Config(key="enable.legacy", type=ConfigType.BOOLEAN, defaultValue="false", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="If true, LegacyReplicationManager will handle RATIS containers while ReplicationManager will handle EC containers. If false, ReplicationManager will handle both RATIS and EC.")
        private boolean enableLegacy;
        @Config(key="thread.interval", type=ConfigType.TIME, defaultValue="300s", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="There is a replication monitor thread running inside SCM which takes care of replicating the containers in the cluster. This property is used to configure the interval in which that thread runs.")
        private Duration interval = Duration.ofSeconds(300L);
        @Config(key="under.replicated.interval", type=ConfigType.TIME, defaultValue="30s", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="How frequently to check if there are work to process  on the under replicated queue")
        private Duration underReplicatedInterval = Duration.ofSeconds(30L);
        @Config(key="over.replicated.interval", type=ConfigType.TIME, defaultValue="30s", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="How frequently to check if there are work to process  on the over replicated queue")
        private Duration overReplicatedInterval = Duration.ofSeconds(30L);
        @Config(key="event.timeout", type=ConfigType.TIME, defaultValue="10m", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="Timeout for the container replication/deletion commands sent to datanodes. After this timeout the command will be retried.")
        private long eventTimeout = Duration.ofMinutes(10L).toMillis();
        @Config(key="event.timeout.datanode.offset", type=ConfigType.TIME, defaultValue="30s", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="The amount of time to subtract from hdds.scm.replication.event.timeout to give a deadline on the datanodes which is less than the SCM timeout. This ensures the datanodes will not process a command after SCM believes it should have expired.")
        private long datanodeTimeoutOffset = Duration.ofSeconds(30L).toMillis();
        @Config(key="maintenance.replica.minimum", type=ConfigType.INT, defaultValue="2", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="The minimum number of container replicas which must  be available for a node to enter maintenance. If putting a  node into maintenance reduces the available replicas for any  container below this level, the node will remain in the  entering maintenance state until a new replica is created.")
        private int maintenanceReplicaMinimum = 2;
        @Config(key="maintenance.remaining.redundancy", type=ConfigType.INT, defaultValue="1", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.OZONE}, description="The number of redundant containers in a group which must be available for a node to enter maintenance. If putting a node into maintenance reduces the redundancy below this value , the node will remain in the ENTERING_MAINTENANCE state until a new replica is created. For Ratis containers, the default value of 1 ensures at least two replicas are online, meaning 1 more can be lost without data becoming unavailable. For any EC container it will have at least dataNum + 1 online, allowing the loss of 1 more replica before data becomes unavailable. Currently only EC containers use this setting. Ratis containers use hdds.scm.replication.maintenance.replica.minimum. For EC, if nodes are in maintenance, it is likely reconstruction reads will be required if some of the data replicas are offline. This is seamless to the client, but will affect read performance.")
        private int maintenanceRemainingRedundancy = 1;
        @Config(key="push", type=ConfigType.BOOLEAN, defaultValue="true", tags={ConfigTag.SCM, ConfigTag.DATANODE}, description="If false, replication happens by asking the target to pull from source nodes.  If true, the source node is asked to push to the target node.")
        private boolean push = true;
        @Config(key="datanode.replication.limit", type=ConfigType.INT, defaultValue="20", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.DATANODE}, description="A limit to restrict the total number of replication and reconstruction commands queued on a datanode. Note this is intended to be a temporary config until we have a more dynamic way of limiting load.")
        private int datanodeReplicationLimit = 20;
        @Config(key="datanode.reconstruction.weight", type=ConfigType.INT, defaultValue="3", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.DATANODE}, description="When counting the number of replication commands on a datanode, the number of reconstruction commands is multiplied by this weight to ensure reconstruction commands use more of the capacity, as they are more expensive to process.")
        private int reconstructionCommandWeight = 3;
        @Config(key="datanode.delete.container.limit", type=ConfigType.INT, defaultValue="40", reconfigurable=true, tags={ConfigTag.SCM, ConfigTag.DATANODE}, description="A limit to restrict the total number of delete container commands queued on a datanode. Note this is intended to be a temporary config until we have a more dynamic way of limiting load")
        private int datanodeDeleteLimit = 40;
        @Config(key="inflight.limit.factor", type=ConfigType.DOUBLE, defaultValue="0.75", reconfigurable=true, tags={ConfigTag.SCM}, description="The overall replication task limit on a cluster is the number healthy nodes, times the datanode.replication.limit. This factor, which should be between zero and 1, scales that limit down to reduce the overall number of replicas pending creation on the cluster. A setting of zero disables global limit checking. A setting of 1 effectively disables it, by making the limit equal to the above equation. However if there are many decommissioning nodes on the cluster, the decommission nodes will have a higher than normal limit, so the setting of 1 may still provide some limit in extreme circumstances.")
        private double inflightReplicationLimitFactor = 0.75;

        public boolean isLegacyEnabled() {
            return this.enableLegacy;
        }

        public void setEnableLegacy(boolean enableLegacy) {
            this.enableLegacy = enableLegacy;
        }

        public void setInterval(Duration interval) {
            this.interval = interval;
        }

        public void setEventTimeout(Duration timeout) {
            this.eventTimeout = timeout.toMillis();
        }

        public long getDatanodeTimeoutOffset() {
            return this.datanodeTimeoutOffset;
        }

        public void setDatanodeTimeoutOffset(long val) {
            this.datanodeTimeoutOffset = val;
        }

        public void setMaintenanceReplicaMinimum(int replicaCount) {
            this.maintenanceReplicaMinimum = replicaCount;
        }

        public int getDatanodeReplicationLimit() {
            return this.datanodeReplicationLimit;
        }

        public int getReconstructionCommandWeight() {
            return this.reconstructionCommandWeight;
        }

        public int getDatanodeDeleteLimit() {
            return this.datanodeDeleteLimit;
        }

        public double getInflightReplicationLimitFactor() {
            return this.inflightReplicationLimitFactor;
        }

        public void setInflightReplicationLimitFactor(double factor) {
            this.inflightReplicationLimitFactor = factor;
        }

        public void setDatanodeReplicationLimit(int limit) {
            this.datanodeReplicationLimit = limit;
        }

        public void setMaintenanceRemainingRedundancy(int redundancy) {
            this.maintenanceRemainingRedundancy = redundancy;
        }

        public int getMaintenanceRemainingRedundancy() {
            return this.maintenanceRemainingRedundancy;
        }

        public Duration getInterval() {
            return this.interval;
        }

        public Duration getUnderReplicatedInterval() {
            return this.underReplicatedInterval;
        }

        public void setUnderReplicatedInterval(Duration duration) {
            this.underReplicatedInterval = duration;
        }

        public void setOverReplicatedInterval(Duration duration) {
            this.overReplicatedInterval = duration;
        }

        public Duration getOverReplicatedInterval() {
            return this.overReplicatedInterval;
        }

        public long getEventTimeout() {
            return this.eventTimeout;
        }

        public int getMaintenanceReplicaMinimum() {
            return this.maintenanceReplicaMinimum;
        }

        public boolean isPush() {
            return this.push;
        }

        @PostConstruct
        public void validate() {
            if (this.datanodeTimeoutOffset < 0L) {
                throw new IllegalArgumentException("event.timeout.datanode.offset is set to " + this.datanodeTimeoutOffset + " and must be >= 0");
            }
            if (this.datanodeTimeoutOffset >= this.eventTimeout) {
                throw new IllegalArgumentException("event.timeout.datanode.offset is set to " + this.datanodeTimeoutOffset + " and must be < event.timeout, which is set to " + this.eventTimeout);
            }
            if (this.reconstructionCommandWeight <= 0) {
                throw new IllegalArgumentException("datanode.reconstruction.weight: " + this.reconstructionCommandWeight + " must be > 0");
            }
            if (this.datanodeReplicationLimit < this.reconstructionCommandWeight) {
                throw new IllegalArgumentException("datanode.replication.limit: " + this.datanodeReplicationLimit + " must be >= datanode.reconstruction.weight: " + this.reconstructionCommandWeight);
            }
            if (this.inflightReplicationLimitFactor < 0.0) {
                throw new IllegalArgumentException("inflight.limit.factor is set to " + this.inflightReplicationLimitFactor + " and must be >= 0");
            }
            if (this.inflightReplicationLimitFactor > 1.0) {
                throw new IllegalArgumentException("inflight.limit.factor is set to " + this.inflightReplicationLimitFactor + " and must be <= 1");
            }
        }
    }
}

