/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.replication;

import com.google.common.collect.ImmutableList;
import com.google.protobuf.Message;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.Config;
import org.apache.hadoop.hdds.conf.ConfigGroup;
import org.apache.hadoop.hdds.conf.ConfigTag;
import org.apache.hadoop.hdds.conf.ConfigType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.SCMRatisProtocol;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.ContainerPlacementStatus;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ReplicationManagerReport;
import org.apache.hadoop.hdds.scm.container.balancer.MoveManager;
import org.apache.hadoop.hdds.scm.container.common.helpers.MoveDataNodePair;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaOp;
import org.apache.hadoop.hdds.scm.container.replication.InflightAction;
import org.apache.hadoop.hdds.scm.container.replication.InflightType;
import org.apache.hadoop.hdds.scm.container.replication.LegacyRatisContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.RatisContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerMetrics;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerUtil;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.SCMHAInvocationHandler;
import org.apache.hadoop.hdds.scm.ha.SCMHAManager;
import org.apache.hadoop.hdds.scm.ha.SCMRatisServer;
import org.apache.hadoop.hdds.scm.metadata.DBTransactionBuffer;
import org.apache.hadoop.hdds.scm.metadata.Replicate;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.ClientVersion;
import org.apache.hadoop.ozone.common.statemachine.InvalidStateTransitionException;
import org.apache.hadoop.ozone.protocol.commands.CloseContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.DeleteContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.ReplicateContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.apache.ratis.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LegacyReplicationManager {
    public static final Logger LOG = LoggerFactory.getLogger(LegacyReplicationManager.class);
    private final ContainerManager containerManager;
    private final PlacementPolicy containerPlacement;
    private final EventPublisher eventPublisher;
    private final SCMContext scmContext;
    private final NodeManager nodeManager;
    private final InflightMap inflightReplication;
    private final InflightMap inflightDeletion;
    private final Map<ContainerID, CompletableFuture<MoveManager.MoveResult>> inflightMoveFuture;
    private final ReplicationManager.ReplicationManagerConfiguration rmConf;
    private int minHealthyForMaintenance;
    private final Clock clock;
    private long currentContainerSize;
    private ReplicationManagerMetrics metrics;
    private final MoveScheduler moveScheduler;

    public LegacyReplicationManager(ConfigurationSource conf, ContainerManager containerManager, PlacementPolicy containerPlacement, EventPublisher eventPublisher, SCMContext scmContext, NodeManager nodeManager, SCMHAManager scmhaManager, Clock clock, Table<ContainerID, MoveDataNodePair> moveTable) throws IOException {
        this.containerManager = containerManager;
        this.containerPlacement = containerPlacement;
        this.eventPublisher = eventPublisher;
        this.scmContext = scmContext;
        this.nodeManager = nodeManager;
        this.rmConf = (ReplicationManager.ReplicationManagerConfiguration)((Object)conf.getObject(ReplicationManager.ReplicationManagerConfiguration.class));
        LegacyReplicationManagerConfiguration legacyConf = (LegacyReplicationManagerConfiguration)conf.getObject(LegacyReplicationManagerConfiguration.class);
        this.inflightReplication = new InflightMap(InflightType.REPLICATION, legacyConf.getContainerInflightReplicationLimit());
        this.inflightDeletion = new InflightMap(InflightType.DELETION, legacyConf.getContainerInflightDeletionLimit());
        this.inflightMoveFuture = new ConcurrentHashMap<ContainerID, CompletableFuture<MoveManager.MoveResult>>();
        this.minHealthyForMaintenance = this.rmConf.getMaintenanceReplicaMinimum();
        this.clock = clock;
        this.currentContainerSize = (long)conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        this.metrics = null;
        this.moveScheduler = new MoveSchedulerImpl.Builder().setDBTransactionBuffer(scmhaManager.getDBTransactionBuffer()).setRatisServer(scmhaManager.getRatisServer()).setMoveTable(moveTable).build();
    }

    protected synchronized void clearInflightActions() {
        this.inflightReplication.clear();
        this.inflightDeletion.clear();
    }

    protected synchronized void setMetrics(ReplicationManagerMetrics metrics) {
        this.metrics = metrics;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void processContainer(ContainerInfo container, ReplicationManagerReport report) {
        ContainerID id = container.containerID();
        try {
            ContainerInfo containerInfo = container;
            synchronized (containerInfo) {
                Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(id);
                HddsProtos.LifeCycleState state = container.getState();
                if (state == HddsProtos.LifeCycleState.OPEN) {
                    if (!this.isOpenContainerHealthy(container, replicas)) {
                        report.incrementAndSample(ReplicationManagerReport.HealthState.OPEN_UNHEALTHY, container.containerID());
                        this.eventPublisher.fireEvent(SCMEvents.CLOSE_CONTAINER, (Object)id);
                    }
                    return;
                }
                if (state == HddsProtos.LifeCycleState.CLOSING) {
                    this.setHealthStateForClosing(replicas, container, report);
                    boolean foundHealthy = false;
                    for (ContainerReplica replica : replicas) {
                        if (replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.UNHEALTHY) continue;
                        foundHealthy = true;
                        this.sendCloseCommand(container, replica.getDatanodeDetails(), false);
                    }
                    if (replicas.isEmpty() && container.getNumberOfKeys() == 0L) {
                        this.closeEmptyContainer(container);
                        return;
                    }
                    if (!foundHealthy) {
                        this.containerManager.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.QUASI_CLOSE);
                        LOG.debug("Moved container {} from CLOSING to QUASI_CLOSED because it has only UNHEALTHY replicas: {}.", (Object)container, replicas);
                    }
                    return;
                }
                if (state == HddsProtos.LifeCycleState.QUASI_CLOSED) {
                    if (this.canForceCloseContainer(container, replicas)) {
                        this.forceCloseContainer(container, replicas);
                        return;
                    }
                    report.incrementAndSample(ReplicationManagerReport.HealthState.QUASI_CLOSED_STUCK, container.containerID());
                }
                if (container.getReplicationType() == HddsProtos.ReplicationType.EC) {
                    return;
                }
                this.updateInflightAction(container, this.inflightReplication, action -> replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)action.getDatanode())), () -> this.metrics.incrReplicaCreateTimeoutTotal(), action -> this.updateCompletedReplicationMetrics(container, (InflightAction)action));
                this.updateInflightAction(container, this.inflightDeletion, action -> replicas.stream().noneMatch(r -> r.getDatanodeDetails().equals((Object)action.getDatanode())), () -> this.metrics.incrReplicaDeleteTimeoutTotal(), action -> this.updateCompletedDeletionMetrics(container, (InflightAction)action));
                if (state == HddsProtos.LifeCycleState.DELETING) {
                    this.handleContainerUnderDelete(container, replicas);
                    return;
                }
                if (state == HddsProtos.LifeCycleState.DELETED) {
                    return;
                }
                RatisContainerReplicaCount replicaSet = this.getContainerReplicaCount(container, replicas);
                ContainerPlacementStatus placementStatus = this.getPlacementStatus(replicas, container.getReplicationConfig().getRequiredNodes());
                if (this.isContainerEmpty(container, replicas)) {
                    report.incrementAndSample(ReplicationManagerReport.HealthState.EMPTY, container.containerID());
                    this.deleteContainerReplicas(container, replicas);
                    return;
                }
                if (replicas.isEmpty() && container.getState() == HddsProtos.LifeCycleState.CLOSED && container.getNumberOfKeys() == 0L) {
                    LOG.debug("Container {} appears empty and is closed, but cannot be deleted because it has no replicas. Marking as EMPTY.", (Object)container);
                    report.incrementAndSample(ReplicationManagerReport.HealthState.EMPTY, container.containerID());
                    return;
                }
                boolean sufficientlyReplicated = replicaSet.isSufficientlyReplicated();
                boolean placementSatisfied = placementStatus.isPolicySatisfied();
                ContainerID containerID = container.containerID();
                if (!placementStatus.isPolicySatisfied()) {
                    report.incrementAndSample(ReplicationManagerReport.HealthState.MIS_REPLICATED, containerID);
                }
                if (!replicaSet.isHealthy()) {
                    report.incrementAndSample(ReplicationManagerReport.HealthState.UNHEALTHY, containerID);
                }
                if (!sufficientlyReplicated || !placementSatisfied) {
                    if (!this.inflightReplication.isFull() || !this.inflightDeletion.isFull()) {
                        if (replicaSet.isUnrecoverable()) {
                            report.incrementAndSample(ReplicationManagerReport.HealthState.MISSING, containerID);
                            report.incrementAndSample(ReplicationManagerReport.HealthState.UNDER_REPLICATED, containerID);
                        } else if (replicaSet.getHealthyReplicaCount() == 0 && replicaSet.getUnhealthyReplicaCount() != 0) {
                            this.handleAllReplicasUnhealthy(container, replicaSet, placementStatus, report);
                        } else {
                            this.handleUnderReplicatedHealthy(container, replicaSet, placementStatus, report);
                        }
                    }
                    return;
                }
                List<ContainerReplica> vulnerableUnhealthy = replicaSet.getVulnerableUnhealthyReplicas(dn -> {
                    try {
                        return this.nodeManager.getNodeStatus((DatanodeDetails)dn);
                    }
                    catch (NodeNotFoundException e) {
                        LOG.warn("Exception for datanode {} while getting vulnerable replicas for container {}, with all replicas {}.", new Object[]{dn, container, replicas, e});
                        return null;
                    }
                });
                if (!vulnerableUnhealthy.isEmpty()) {
                    report.incrementAndSample(ReplicationManagerReport.HealthState.UNDER_REPLICATED, container.containerID());
                    this.handleVulnerableUnhealthyReplicas(replicaSet, vulnerableUnhealthy);
                    return;
                }
                if (replicaSet.getReplicas().size() > container.getReplicationConfig().getRequiredNodes()) {
                    if (replicaSet.isHealthy()) {
                        this.handleOverReplicatedHealthy(container, replicaSet, report);
                    } else {
                        this.handleOverReplicatedExcessUnhealthy(container, replicaSet, report);
                    }
                    return;
                }
                if (!replicaSet.isHealthy()) {
                    this.handleContainerWithUnhealthyReplica(container, replicaSet);
                }
            }
        }
        catch (ContainerNotFoundException ex) {
            LOG.warn("Missing container {}.", (Object)id);
        }
        catch (Exception ex) {
            LOG.warn("Process container {} error: ", (Object)id, (Object)ex);
        }
    }

    private void handleVulnerableUnhealthyReplicas(RatisContainerReplicaCount replicaCount, List<ContainerReplica> vulnerableUnhealthy) {
        ContainerInfo container = replicaCount.getContainer();
        LOG.debug("Handling vulnerable UNHEALTHY replicas {} for container {}.", vulnerableUnhealthy, (Object)container);
        int pendingAdds = this.getInflightAdd(container.containerID());
        if (pendingAdds >= vulnerableUnhealthy.size()) {
            LOG.debug("There are {} pending adds for container {}, while the number of UNHEALTHY replicas is {}.", new Object[]{pendingAdds, container.containerID(), vulnerableUnhealthy.size()});
            return;
        }
        Collections.shuffle(vulnerableUnhealthy);
        this.replicateEachSource(container, vulnerableUnhealthy, replicaCount.getReplicas());
    }

    private void updateCompletedReplicationMetrics(ContainerInfo container, InflightAction action) {
        this.metrics.incrReplicasCreatedTotal();
        this.metrics.incrReplicationBytesCompletedTotal(container.getUsedBytes());
        this.metrics.addReplicationTime(this.clock.millis() - action.getTime());
    }

    private void updateCompletedDeletionMetrics(ContainerInfo container, InflightAction action) {
        this.metrics.incrReplicasDeletedTotal();
        this.metrics.incrDeletionBytesCompletedTotal(container.getUsedBytes());
        this.metrics.addDeletionTime(this.clock.millis() - action.getTime());
    }

    private void updateInflightAction(ContainerInfo container, InflightMap inflightActions, Predicate<InflightAction> filter, Runnable timeoutCounter, Consumer<InflightAction> completedCounter) {
        ContainerID id = container.containerID();
        long deadline = this.clock.millis() - this.rmConf.getEventTimeout();
        inflightActions.iterate(id, a -> this.updateInflightAction(container, (InflightAction)a, filter, timeoutCounter, completedCounter, deadline, inflightActions.isReplication()));
    }

    private boolean updateInflightAction(ContainerInfo container, InflightAction a, Predicate<InflightAction> filter, Runnable timeoutCounter, Consumer<InflightAction> completedCounter, long deadline, boolean isReplication) {
        boolean remove = false;
        try {
            boolean isNotInService;
            NodeStatus status = this.nodeManager.getNodeStatus(a.getDatanode());
            boolean isUnhealthy = status.getHealth() != HddsProtos.NodeState.HEALTHY;
            boolean isCompleted = filter.test(a);
            boolean isTimeout = a.getTime() < deadline;
            boolean bl = isNotInService = status.getOperationalState() != HddsProtos.NodeOperationalState.IN_SERVICE;
            if (isCompleted || isUnhealthy || isTimeout || isNotInService) {
                if (isTimeout) {
                    timeoutCounter.run();
                } else if (isCompleted) {
                    completedCounter.accept(a);
                }
                this.updateMoveIfNeeded(isUnhealthy, isCompleted, isTimeout, isNotInService, container, a.getDatanode(), isReplication);
                remove = true;
            }
        }
        catch (ContainerNotFoundException | NodeNotFoundException e) {
            remove = true;
        }
        catch (Exception e) {
            LOG.error("Got exception while updating.", (Throwable)e);
        }
        return remove;
    }

    private void updateMoveIfNeeded(boolean isUnhealthy, boolean isCompleted, boolean isTimeout, boolean isNotInService, ContainerInfo container, DatanodeDetails dn, boolean isInflightReplication) throws SCMException {
        ContainerID id = container.containerID();
        MoveDataNodePair kv = this.moveScheduler.getMoveDataNodePair(id);
        if (kv == null) {
            return;
        }
        boolean isSource = kv.getSrc().equals((Object)dn);
        boolean isTarget = kv.getTgt().equals((Object)dn);
        if (!isSource && !isTarget) {
            return;
        }
        if (isSource && isInflightReplication) {
            this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.FAIL_UNEXPECTED_ERROR);
            LOG.info("Move failed because replication for container {} unexpectedly happened at the source {}, not the target {}.", new Object[]{container, kv.getSrc().getUuidString(), kv.getTgt().getUuidString()});
            this.moveScheduler.completeMove(id.getProtobuf());
            return;
        }
        if (isTarget && !isInflightReplication) {
            this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.FAIL_UNEXPECTED_ERROR);
            LOG.info("Move failed because deletion for container {} unexpectedly happened at the target {}, not the source {}.", new Object[]{container, kv.getTgt().getUuidString(), kv.getSrc().getUuidString()});
            this.moveScheduler.completeMove(id.getProtobuf());
            return;
        }
        if (!isInflightReplication || !isCompleted) {
            if (isInflightReplication) {
                if (isUnhealthy) {
                    this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
                } else if (isNotInService) {
                    this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
                } else {
                    this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.REPLICATION_FAIL_TIME_OUT);
                }
            } else if (isUnhealthy) {
                this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.DELETION_FAIL_NODE_UNHEALTHY);
            } else if (isTimeout) {
                this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.DELETION_FAIL_TIME_OUT);
            } else if (isNotInService) {
                this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.DELETION_FAIL_NODE_NOT_IN_SERVICE);
            } else {
                this.compleleteMoveFutureWithResult(id, MoveManager.MoveResult.COMPLETED);
            }
            this.moveScheduler.completeMove(id.getProtobuf());
        } else {
            this.deleteSrcDnForMove(container, this.containerManager.getContainerReplicas(id));
        }
    }

    public CompletableFuture<MoveManager.MoveResult> move(ContainerID cid, DatanodeDetails src, DatanodeDetails tgt) throws ContainerNotFoundException, NodeNotFoundException {
        return this.move(cid, new MoveDataNodePair(src, tgt));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<MoveManager.MoveResult> move(ContainerID cid, MoveDataNodePair mp) throws ContainerNotFoundException, NodeNotFoundException {
        ContainerInfo cif;
        CompletableFuture<MoveManager.MoveResult> ret = new CompletableFuture<MoveManager.MoveResult>();
        if (!this.scmContext.isLeader()) {
            ret.complete(MoveManager.MoveResult.FAIL_LEADER_NOT_READY);
            return ret;
        }
        DatanodeDetails srcDn = mp.getSrc();
        DatanodeDetails targetDn = mp.getTgt();
        NodeStatus currentNodeStat = this.nodeManager.getNodeStatus(srcDn);
        HddsProtos.NodeState healthStat = currentNodeStat.getHealth();
        HddsProtos.NodeOperationalState operationalState = currentNodeStat.getOperationalState();
        if (healthStat != HddsProtos.NodeState.HEALTHY) {
            ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
            LOG.info("Failing move for container {} because source {} is {}", new Object[]{cid, srcDn.getUuidString(), healthStat.toString()});
            return ret;
        }
        if (operationalState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
            LOG.info("Failing move for container {} because source {} is {}", new Object[]{cid, srcDn.getUuidString(), operationalState.toString()});
            return ret;
        }
        currentNodeStat = this.nodeManager.getNodeStatus(targetDn);
        healthStat = currentNodeStat.getHealth();
        operationalState = currentNodeStat.getOperationalState();
        if (healthStat != HddsProtos.NodeState.HEALTHY) {
            ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
            LOG.info("Failing move for container {} because target {} is {}", new Object[]{cid, targetDn.getUuidString(), healthStat.toString()});
            return ret;
        }
        if (operationalState != HddsProtos.NodeOperationalState.IN_SERVICE) {
            ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
            LOG.info("Failing move for container {} because target {} is {}", new Object[]{cid, targetDn.getUuidString(), operationalState.toString()});
            return ret;
        }
        ContainerInfo containerInfo = cif = this.containerManager.getContainer(cid);
        synchronized (containerInfo) {
            Set<ContainerReplica> currentReplicas = this.containerManager.getContainerReplicas(cid);
            Set replicas = currentReplicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toSet());
            if (replicas.contains(targetDn)) {
                ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_EXIST_IN_TARGET);
                return ret;
            }
            if (!replicas.contains(srcDn)) {
                ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_NOT_EXIST_IN_SOURCE);
                return ret;
            }
            if (this.inflightReplication.containsKey(cid)) {
                ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_INFLIGHT_REPLICATION);
                return ret;
            }
            if (this.inflightDeletion.containsKey(cid)) {
                ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_INFLIGHT_DELETION);
                return ret;
            }
            HddsProtos.LifeCycleState currentContainerStat = cif.getState();
            if (currentContainerStat != HddsProtos.LifeCycleState.CLOSED) {
                ret.complete(MoveManager.MoveResult.REPLICATION_FAIL_CONTAINER_NOT_CLOSED);
                return ret;
            }
            if (!this.isPolicySatisfiedAfterMove(cif, srcDn, targetDn, new ArrayList<ContainerReplica>(currentReplicas))) {
                ret.complete(MoveManager.MoveResult.REPLICATION_NOT_HEALTHY_AFTER_MOVE);
                return ret;
            }
            try {
                this.moveScheduler.startMove(cid.getProtobuf(), mp.getProtobufMessage(ClientVersion.CURRENT_VERSION));
            }
            catch (IOException e) {
                LOG.warn("Exception while starting move for container {}", (Object)cid, (Object)e);
                ret.complete(MoveManager.MoveResult.FAIL_UNEXPECTED_ERROR);
                return ret;
            }
            this.inflightMoveFuture.putIfAbsent(cid, ret);
            this.sendReplicateCommand(cif, targetDn, Collections.singletonList(srcDn));
        }
        LOG.info("receive a move request about container {} , from {} to {}", new Object[]{cid, srcDn.getUuid(), targetDn.getUuid()});
        return ret;
    }

    private boolean isPolicySatisfiedAfterMove(ContainerInfo cif, DatanodeDetails srcDn, DatanodeDetails targetDn, List<ContainerReplica> replicas) {
        HashSet<ContainerReplica> movedReplicas = new HashSet<ContainerReplica>(replicas);
        movedReplicas.removeIf(r -> r.getDatanodeDetails().equals((Object)srcDn));
        movedReplicas.add(ContainerReplica.newBuilder().setDatanodeDetails(targetDn).setContainerID(cif.containerID()).setContainerState(StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED).build());
        ContainerPlacementStatus placementStatus = this.getPlacementStatus(movedReplicas, cif.getReplicationConfig().getRequiredNodes());
        return placementStatus.isPolicySatisfied();
    }

    private int getInflightAdd(ContainerID id) {
        return this.inflightReplication.inflightActionCount(id);
    }

    private int getInflightDel(ContainerID id) {
        return this.inflightDeletion.inflightActionCount(id);
    }

    private boolean isContainerEmpty(ContainerInfo container, Set<ContainerReplica> replicas) {
        return container.getState() == HddsProtos.LifeCycleState.CLOSED && !replicas.isEmpty() && replicas.stream().allMatch(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED && r.isEmpty());
    }

    public ContainerReplicaCount getContainerReplicaCount(ContainerID containerID) throws ContainerNotFoundException {
        ContainerInfo container = this.containerManager.getContainer(containerID);
        return this.getContainerReplicaCount(container);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ContainerReplicaCount getContainerReplicaCount(ContainerInfo container) throws ContainerNotFoundException {
        ContainerInfo containerInfo = container;
        synchronized (containerInfo) {
            Set<ContainerReplica> replica = this.containerManager.getContainerReplicas(container.containerID());
            return this.getReplicaCountOptionallyConsiderUnhealthy(container, replica);
        }
    }

    private RatisContainerReplicaCount getContainerReplicaCount(ContainerInfo container, Set<ContainerReplica> replica) {
        return new LegacyRatisContainerReplicaCount(container, replica, this.getInflightAdd(container.containerID()), this.getInflightDel(container.containerID()), container.getReplicationConfig().getRequiredNodes(), this.minHealthyForMaintenance);
    }

    private RatisContainerReplicaCount getReplicaCountOptionallyConsiderUnhealthy(ContainerInfo container, Set<ContainerReplica> replicas) {
        LegacyRatisContainerReplicaCount withUnhealthy = new LegacyRatisContainerReplicaCount(container, replicas, this.getPendingOps(container.containerID()), this.minHealthyForMaintenance, true);
        if (withUnhealthy.getHealthyReplicaCount() == 0 && withUnhealthy.getUnhealthyReplicaCount() > 0) {
            return withUnhealthy;
        }
        return new LegacyRatisContainerReplicaCount(container, replicas, this.getInflightAdd(container.containerID()), this.getInflightDel(container.containerID()), container.getReplicationConfig().getRequiredNodes(), this.minHealthyForMaintenance);
    }

    private boolean canForceCloseContainer(ContainerInfo container, Set<ContainerReplica> replicas) {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.QUASI_CLOSED ? 1 : 0) != 0);
        int replicationFactor = container.getReplicationConfig().getRequiredNodes();
        long uniqueQuasiClosedReplicaCount = replicas.stream().filter(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED).map(ContainerReplica::getOriginDatanodeId).distinct().count();
        return uniqueQuasiClosedReplicaCount > (long)(replicationFactor / 2);
    }

    private void deleteContainerReplicas(ContainerInfo container, Set<ContainerReplica> replicas) throws IOException, InvalidStateTransitionException {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.CLOSED ? 1 : 0) != 0);
        replicas.stream().forEach(rp -> {
            Preconditions.assertTrue((rp.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED ? 1 : 0) != 0);
            Preconditions.assertTrue((boolean)rp.isEmpty());
            this.sendDeleteCommand(container, rp.getDatanodeDetails(), false);
        });
        this.containerManager.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.DELETE);
        LOG.debug("Deleting empty container replicas for {},", (Object)container);
    }

    private void handleContainerUnderDelete(ContainerInfo container, Set<ContainerReplica> replicas) throws IOException, InvalidStateTransitionException {
        if (replicas.size() == 0) {
            this.containerManager.updateContainerState(container.containerID(), HddsProtos.LifeCycleEvent.CLEANUP);
            LOG.debug("Container {} state changes to DELETED", (Object)container);
        } else {
            List<DatanodeDetails> deletionInFlight = this.inflightDeletion.getDatanodeDetails(container.containerID());
            Set filteredReplicas = replicas.stream().filter(r -> !deletionInFlight.contains(r.getDatanodeDetails())).collect(Collectors.toSet());
            if (filteredReplicas.size() > 0) {
                filteredReplicas.stream().forEach(rp -> this.sendDeleteCommand(container, rp.getDatanodeDetails(), false));
                LOG.debug("Resend delete Container command for {}", (Object)container);
            }
        }
    }

    private void forceCloseContainer(ContainerInfo container, Set<ContainerReplica> replicas) {
        Preconditions.assertTrue((container.getState() == HddsProtos.LifeCycleState.QUASI_CLOSED ? 1 : 0) != 0);
        List quasiClosedReplicas = replicas.stream().filter(r -> r.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED).collect(Collectors.toList());
        Long sequenceId = quasiClosedReplicas.stream().map(ContainerReplica::getSequenceId).max(Long::compare).orElse(-1L);
        LOG.info("Force closing container {} with BCSID {}, which is in QUASI_CLOSED state.", (Object)container.containerID(), (Object)sequenceId);
        quasiClosedReplicas.stream().filter(r -> sequenceId != -1L).filter(replica -> replica.getSequenceId().equals(sequenceId)).forEach(replica -> this.sendCloseCommand(container, replica.getDatanodeDetails(), true));
    }

    private void handleUnderReplicatedHealthy(ContainerInfo container, RatisContainerReplicaCount replicaSet, ContainerPlacementStatus placementStatus, ReplicationManagerReport report) {
        block5: {
            LOG.debug("Handling under-replicated container: {}", (Object)container);
            if (replicaSet.isSufficientlyReplicated() && placementStatus.isPolicySatisfied()) {
                LOG.info("The container {} with replicas {} is sufficiently replicated and is not mis-replicated", (Object)container.getContainerID(), (Object)replicaSet);
                return;
            }
            List<ContainerReplica> allReplicas = replicaSet.getReplicas();
            int numCloseCommandsSent = this.closeReplicasIfPossible(container, allReplicas);
            int replicasNeeded = replicaSet.additionalReplicaNeeded() - numCloseCommandsSent;
            if (replicasNeeded > 0) {
                report.incrementAndSample(ReplicationManagerReport.HealthState.UNDER_REPLICATED, container.containerID());
            }
            StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State matchingReplicaState = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
            if (container.getState() == HddsProtos.LifeCycleState.QUASI_CLOSED) {
                matchingReplicaState = StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED;
            }
            List<ContainerReplica> replicationSources = this.getReplicationSources(container, replicaSet.getReplicas(), matchingReplicaState);
            try {
                this.replicateAnyWithTopology(container, replicationSources, placementStatus, replicasNeeded, replicaSet.getReplicas());
            }
            catch (SCMException e) {
                if (!e.getResult().equals((Object)SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE) || replicasNeeded <= 0) break block5;
                this.deleteUnhealthyReplicaIfNeeded(container, replicaSet);
            }
        }
    }

    private void deleteUnhealthyReplicaIfNeeded(ContainerInfo container, RatisContainerReplicaCount replicaCount) {
        LOG.info("Finding an unhealthy replica to delete for container {} with replicas {} to unblock under replication handling.", (Object)container, replicaCount.getReplicas());
        HashSet<ContainerReplica> replicas = new HashSet<ContainerReplica>(replicaCount.getReplicas());
        ContainerReplica replica = ReplicationManagerUtil.selectUnhealthyReplicaForDelete(container, replicas, this.getInflightDel(container.containerID()), dnd -> {
            try {
                return this.nodeManager.getNodeStatus((DatanodeDetails)dnd);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Exception while finding an unhealthy replica to delete for container {}.", (Object)container, (Object)e);
                return null;
            }
        });
        if (replica == null) {
            LOG.info("Could not find any unhealthy replica to delete when unblocking under replication handling for container {} with replicas {}.", (Object)container, replicas);
        } else {
            this.sendDeleteCommand(container, replica.getDatanodeDetails(), true);
        }
    }

    private void handleOverReplicatedHealthy(ContainerInfo container, RatisContainerReplicaCount replicaSet, ReplicationManagerReport report) {
        ContainerID id = container.containerID();
        int replicationFactor = container.getReplicationConfig().getRequiredNodes();
        int excess = replicaSet.additionalReplicaNeeded() * -1;
        if (excess > 0) {
            LOG.info("Container {} is over replicated. Expected replica count is {}, but found {}.", new Object[]{id, replicationFactor, replicationFactor + excess});
            report.incrementAndSample(ReplicationManagerReport.HealthState.OVER_REPLICATED, container.containerID());
            List<ContainerReplica> deleteCandidates = this.getHealthyDeletionCandidates(container, replicaSet.getReplicas());
            if (container.getState() == HddsProtos.LifeCycleState.CLOSED) {
                this.deleteExcessWithTopology(excess, container, deleteCandidates);
            } else {
                this.deleteExcessWithNonUniqueOriginNodeIDs(container, replicaSet.getReplicas(), deleteCandidates, excess);
            }
        }
    }

    private void handleAllReplicasUnhealthy(ContainerInfo container, RatisContainerReplicaCount replicaSet, ContainerPlacementStatus placementStatus, ReplicationManagerReport report) {
        List<ContainerReplica> replicas = replicaSet.getReplicas();
        LegacyRatisContainerReplicaCount unhealthyReplicaSet = new LegacyRatisContainerReplicaCount(container, new HashSet<ContainerReplica>(replicaSet.getReplicas()), this.getPendingOps(container.containerID()), this.minHealthyForMaintenance, true);
        if (unhealthyReplicaSet.isUnderReplicated()) {
            this.handleUnderReplicatedAllUnhealthy(container, replicas, placementStatus, unhealthyReplicaSet.additionalReplicaNeeded(), report);
        } else if (unhealthyReplicaSet.isOverReplicated()) {
            this.handleOverReplicatedAllUnhealthy(container, replicas, unhealthyReplicaSet.getExcessRedundancy(true), report);
        } else {
            this.closeReplicasIfPossible(container, replicas);
        }
    }

    private List<ContainerReplicaOp> getPendingOps(ContainerID containerID) {
        ArrayList<ContainerReplicaOp> pendingOps = new ArrayList<ContainerReplicaOp>();
        List inflightActions = this.inflightReplication.get(containerID);
        if (inflightActions != null) {
            for (InflightAction a : inflightActions) {
                pendingOps.add(new ContainerReplicaOp(ContainerReplicaOp.PendingOpType.ADD, a.getDatanode(), 0, Long.MAX_VALUE));
            }
        }
        if ((inflightActions = this.inflightDeletion.get(containerID)) != null) {
            for (InflightAction a : inflightActions) {
                pendingOps.add(new ContainerReplicaOp(ContainerReplicaOp.PendingOpType.DELETE, a.getDatanode(), 0, Long.MAX_VALUE));
            }
        }
        return pendingOps;
    }

    private void handleOverReplicatedExcessUnhealthy(ContainerInfo container, RatisContainerReplicaCount replicaSet, ReplicationManagerReport report) {
        List<ContainerReplica> replicas = replicaSet.getReplicas();
        List<ContainerReplica> unhealthyReplicas = this.getUnhealthyDeletionCandidates(container, replicas);
        this.closeReplicasIfPossible(container, unhealthyReplicas);
        if (!unhealthyReplicas.isEmpty()) {
            int excessReplicaCount = replicas.size() - container.getReplicationConfig().getRequiredNodes();
            boolean excessDeleted = false;
            if (container.getState() == HddsProtos.LifeCycleState.CLOSED) {
                this.deleteExcess(container, unhealthyReplicas, excessReplicaCount);
                excessDeleted = true;
            } else {
                excessDeleted = this.deleteExcessWithNonUniqueOriginNodeIDs(container, replicaSet.getReplicas(), unhealthyReplicas, excessReplicaCount);
            }
            if (excessDeleted) {
                LOG.info("Container {} has {} excess unhealthy replicas. Excess unhealthy replicas will be deleted.", (Object)container.getContainerID(), (Object)unhealthyReplicas.size());
                report.incrementAndSample(ReplicationManagerReport.HealthState.OVER_REPLICATED, container.containerID());
            }
        }
    }

    private void handleContainerWithUnhealthyReplica(ContainerInfo container, RatisContainerReplicaCount replicaSet) {
        if (container.getState() == HddsProtos.LifeCycleState.CLOSED) {
            List<ContainerReplica> replicas = replicaSet.getReplicas();
            List<ContainerReplica> replicationSources = this.getReplicationSources(container, replicaSet.getReplicas(), StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED);
            if (replicationSources.isEmpty()) {
                LOG.warn("No healthy CLOSED replica for replication.");
                return;
            }
            ContainerPlacementStatus placementStatus = this.getPlacementStatus(new HashSet<ContainerReplica>(replicationSources), container.getReplicationConfig().getRequiredNodes());
            try {
                this.replicateAnyWithTopology(container, replicationSources, placementStatus, replicas.size() - replicationSources.size(), replicas);
            }
            catch (SCMException e) {
                LOG.warn("Could not fix container {} with replicas {}.", new Object[]{container, replicas, e});
            }
        }
    }

    private List<ContainerReplica> getReplicationSources(ContainerInfo container, List<ContainerReplica> replicas, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State ... validReplicaStates) {
        List<DatanodeDetails> deletionInFlight = this.inflightDeletion.getDatanodeDetails(container.containerID());
        Set validReplicaStateSet = Arrays.stream(validReplicaStates).collect(Collectors.toSet());
        return replicas.stream().filter(r -> this.getNodeStatus(r.getDatanodeDetails()).isHealthy() && !deletionInFlight.contains(r.getDatanodeDetails()) && (validReplicaStateSet.isEmpty() || validReplicaStateSet.contains(r.getState()))).collect(Collectors.toList());
    }

    private List<ContainerReplica> getHealthyDeletionCandidates(ContainerInfo container, List<ContainerReplica> replicas) {
        return this.getDeletionCandidates(container, replicas, true);
    }

    private List<ContainerReplica> getUnhealthyDeletionCandidates(ContainerInfo container, List<ContainerReplica> replicas) {
        return this.getDeletionCandidates(container, replicas, false);
    }

    private List<ContainerReplica> getDeletionCandidates(ContainerInfo container, List<ContainerReplica> replicas, boolean healthy) {
        return replicas.stream().filter(r -> this.getNodeStatus(r.getDatanodeDetails()).isHealthy() && LegacyReplicationManager.compareState(container.getState(), r.getState()) == healthy && r.getDatanodeDetails().getPersistedOpState() == HddsProtos.NodeOperationalState.IN_SERVICE).collect(Collectors.toList());
    }

    private void deleteSrcDnForMove(ContainerInfo cif, Set<ContainerReplica> replicaSet) throws SCMException {
        ContainerID cid = cif.containerID();
        MoveDataNodePair movePair = this.moveScheduler.getMoveDataNodePair(cid);
        if (movePair == null) {
            return;
        }
        DatanodeDetails srcDn = movePair.getSrc();
        RatisContainerReplicaCount replicaCount = this.getContainerReplicaCount(cif, replicaSet);
        if (!replicaSet.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)srcDn))) {
            this.compleleteMoveFutureWithResult(cid, MoveManager.MoveResult.COMPLETED);
            this.moveScheduler.completeMove(cid.getProtobuf());
            return;
        }
        int replicationFactor = cif.getReplicationConfig().getRequiredNodes();
        ContainerPlacementStatus currentCPS = this.getPlacementStatus(replicaSet, replicationFactor);
        HashSet<ContainerReplica> newReplicaSet = new HashSet<ContainerReplica>(replicaSet);
        newReplicaSet.removeIf(r -> r.getDatanodeDetails().equals((Object)srcDn));
        ContainerPlacementStatus newCPS = this.getPlacementStatus(newReplicaSet, replicationFactor);
        if (replicaCount.isOverReplicated() && this.isPlacementStatusActuallyEqual(currentCPS, newCPS)) {
            this.sendDeleteCommand(cif, srcDn, true);
        } else {
            LOG.info("can not remove source replica after successfully replicated to target datanode");
            this.compleleteMoveFutureWithResult(cid, MoveManager.MoveResult.DELETE_FAIL_POLICY);
            this.moveScheduler.completeMove(cid.getProtobuf());
        }
    }

    private boolean isPlacementStatusActuallyEqual(ContainerPlacementStatus cps1, ContainerPlacementStatus cps2) {
        return !cps1.isPolicySatisfied() && cps1.actualPlacementCount() == cps2.actualPlacementCount() || cps1.isPolicySatisfied() && cps2.isPolicySatisfied();
    }

    private ContainerPlacementStatus getPlacementStatus(Set<ContainerReplica> replicas, int replicationFactor) {
        List<DatanodeDetails> replicaDns = replicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
        return this.containerPlacement.validateContainerPlacement(replicaDns, replicationFactor);
    }

    private void sendCloseCommand(ContainerInfo container, DatanodeDetails datanode, boolean force) {
        ContainerID containerID = container.containerID();
        LOG.info("Sending close container command for container {} to datanode {}.", (Object)containerID, (Object)datanode);
        CloseContainerCommand closeContainerCommand = new CloseContainerCommand(container.getContainerID(), container.getPipelineID(), force);
        try {
            closeContainerCommand.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending close container command, since current SCM is not leader.", (Throwable)nle);
            return;
        }
        closeContainerCommand.setEncodedToken(this.getContainerToken(containerID));
        this.eventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)new CommandForDatanode(datanode.getUuid(), (SCMCommand)closeContainerCommand));
    }

    private String getContainerToken(ContainerID containerID) {
        if (this.scmContext.getScm() instanceof StorageContainerManager) {
            StorageContainerManager scm = (StorageContainerManager)this.scmContext.getScm();
            return scm.getContainerTokenGenerator().generateEncodedToken(containerID);
        }
        return "";
    }

    private boolean addInflight(InflightType type, ContainerID id, InflightAction action) {
        boolean added = this.getInflightMap(type).add(id, action);
        if (!added) {
            this.metrics.incrInflightSkipped(type);
        }
        return added;
    }

    private void sendReplicateCommand(ContainerInfo container, DatanodeDetails target, List<DatanodeDetails> sources) {
        ContainerID id = container.containerID();
        long containerID = id.getId();
        ReplicateContainerCommand replicateCommand = ReplicateContainerCommand.fromSources((long)containerID, sources);
        LOG.debug("Trying to send {} to {}", (Object)replicateCommand, (Object)target);
        boolean sent = this.sendAndTrackDatanodeCommand(target, (SCMCommand)replicateCommand, action -> this.addInflight(InflightType.REPLICATION, id, (InflightAction)action));
        if (sent) {
            LOG.info("Sent {} to {}", (Object)replicateCommand, (Object)target);
            this.metrics.incrReplicationCmdsSentTotal();
            this.metrics.incrReplicationBytesTotal(container.getUsedBytes());
        }
    }

    private void sendDeleteCommand(ContainerInfo container, DatanodeDetails datanode, boolean force) {
        LOG.info("Sending delete container command for container {} to datanode {}", (Object)container.containerID(), (Object)datanode);
        ContainerID id = container.containerID();
        DeleteContainerCommand deleteCommand = new DeleteContainerCommand(id.getId(), force);
        boolean sent = this.sendAndTrackDatanodeCommand(datanode, (SCMCommand)deleteCommand, action -> this.addInflight(InflightType.DELETION, id, (InflightAction)action));
        if (sent) {
            this.metrics.incrDeletionCmdsSentTotal();
            this.metrics.incrDeletionBytesTotal(container.getUsedBytes());
        }
    }

    private <T extends Message> boolean sendAndTrackDatanodeCommand(DatanodeDetails datanode, SCMCommand<T> command, Predicate<InflightAction> tracker) {
        try {
            command.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending datanode command, since current SCM is not leader.", (Throwable)nle);
            return false;
        }
        boolean allowed = tracker.test(new InflightAction(datanode, this.clock.millis()));
        if (!allowed) {
            return false;
        }
        CommandForDatanode datanodeCommand = new CommandForDatanode(datanode.getUuid(), command);
        this.eventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)datanodeCommand);
        return true;
    }

    private NodeStatus getNodeStatus(DatanodeDetails dn) {
        try {
            return this.nodeManager.getNodeStatus(dn);
        }
        catch (NodeNotFoundException e) {
            throw new IllegalStateException("Unable to find NodeStatus for " + dn, e);
        }
    }

    public static boolean compareState(HddsProtos.LifeCycleState containerState, StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State replicaState) {
        switch (containerState) {
            case OPEN: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN;
            }
            case CLOSING: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING;
            }
            case QUASI_CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED;
            }
            case CLOSED: {
                return replicaState == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
            }
            case DELETING: {
                return false;
            }
            case DELETED: {
                return false;
            }
        }
        return false;
    }

    private boolean isOpenContainerHealthy(ContainerInfo container, Set<ContainerReplica> replicas) {
        HddsProtos.LifeCycleState state = container.getState();
        return replicas.stream().allMatch(r -> LegacyReplicationManager.compareState(state, r.getState()));
    }

    private void setHealthStateForClosing(Set<ContainerReplica> replicas, ContainerInfo container, ReplicationManagerReport report) {
        if (replicas.size() == 0) {
            report.incrementAndSample(ReplicationManagerReport.HealthState.MISSING, container.containerID());
            report.incrementAndSample(ReplicationManagerReport.HealthState.UNDER_REPLICATED, container.containerID());
            report.incrementAndSample(ReplicationManagerReport.HealthState.MIS_REPLICATED, container.containerID());
        }
    }

    public boolean isContainerReplicatingOrDeleting(ContainerID containerID) {
        return this.inflightReplication.containsKey(containerID) || this.inflightDeletion.containsKey(containerID);
    }

    protected void notifyStatusChanged() {
        this.onLeaderReadyAndOutOfSafeMode();
    }

    private InflightMap getInflightMap(InflightType type) {
        switch (type) {
            case REPLICATION: {
                return this.inflightReplication;
            }
            case DELETION: {
                return this.inflightDeletion;
            }
        }
        throw new IllegalStateException("Unexpected type " + (Object)((Object)type));
    }

    int getInflightCount(InflightType type) {
        return this.getInflightMap(type).containerCount();
    }

    DatanodeDetails getFirstDatanode(InflightType type, ContainerID id) {
        return ((InflightAction)this.getInflightMap(type).get(id).get(0)).getDatanode();
    }

    public Map<ContainerID, CompletableFuture<MoveManager.MoveResult>> getInflightMove() {
        return this.inflightMoveFuture;
    }

    public MoveScheduler getMoveScheduler() {
        return this.moveScheduler;
    }

    private void onLeaderReadyAndOutOfSafeMode() {
        LinkedList needToRemove = new LinkedList();
        this.moveScheduler.getInflightMove().forEach((k, v) -> {
            ContainerInfo cif;
            Set<ContainerReplica> replicas;
            try {
                replicas = this.containerManager.getContainerReplicas((ContainerID)k);
                cif = this.containerManager.getContainer((ContainerID)k);
            }
            catch (ContainerNotFoundException e) {
                needToRemove.add(k.getProtobuf());
                LOG.error("can not find container {} while processing replicated move", k);
                return;
            }
            boolean isSrcExist = replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)v.getSrc()));
            boolean isTgtExist = replicas.stream().anyMatch(r -> r.getDatanodeDetails().equals((Object)v.getTgt()));
            if (isSrcExist) {
                if (isTgtExist) {
                    try {
                        this.deleteSrcDnForMove(cif, replicas);
                    }
                    catch (Exception ex) {
                        LOG.error("Exception while cleaning up excess replicas.", (Throwable)ex);
                    }
                } else {
                    this.sendReplicateCommand(cif, v.getTgt(), Collections.singletonList(v.getSrc()));
                }
            } else {
                needToRemove.add(k.getProtobuf());
            }
        });
        for (HddsProtos.ContainerID containerID : needToRemove) {
            try {
                this.moveScheduler.completeMove(containerID);
            }
            catch (Exception ex) {
                LOG.error("Exception while moving container.", (Throwable)ex);
            }
        }
    }

    private void compleleteMoveFutureWithResult(ContainerID cid, MoveManager.MoveResult mr) {
        if (this.inflightMoveFuture.containsKey(cid)) {
            this.inflightMoveFuture.get(cid).complete(mr);
            this.inflightMoveFuture.remove(cid);
        }
    }

    private int closeReplicasIfPossible(ContainerInfo container, List<ContainerReplica> replicas) {
        if (container.getState() == HddsProtos.LifeCycleState.OPEN) {
            return 0;
        }
        int numCloseCmdsSent = 0;
        Iterator<ContainerReplica> iterator = replicas.iterator();
        while (iterator.hasNext()) {
            ContainerReplica replica = iterator.next();
            StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State state = replica.getState();
            if (state == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.OPEN || state == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSING) {
                this.sendCloseCommand(container, replica.getDatanodeDetails(), false);
                ++numCloseCmdsSent;
                iterator.remove();
                continue;
            }
            if (state != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED || container.getState() != HddsProtos.LifeCycleState.CLOSED || container.getSequenceId() != replica.getSequenceId().longValue()) continue;
            this.sendCloseCommand(container, replica.getDatanodeDetails(), true);
            ++numCloseCmdsSent;
            iterator.remove();
        }
        return numCloseCmdsSent;
    }

    private void handleOverReplicatedAllUnhealthy(ContainerInfo container, List<ContainerReplica> replicas, int excess, ReplicationManagerReport report) {
        List<ContainerReplica> deleteCandidates = this.getUnhealthyDeletionCandidates(container, replicas);
        this.closeReplicasIfPossible(container, deleteCandidates);
        if (deleteCandidates.isEmpty()) {
            return;
        }
        if (excess > 0) {
            boolean excessDeleted = false;
            if (container.getState() == HddsProtos.LifeCycleState.CLOSED) {
                this.deleteExcessLowestBcsIDs(container, deleteCandidates, excess);
                excessDeleted = true;
            } else {
                excessDeleted = this.deleteExcessWithNonUniqueOriginNodeIDs(container, replicas, deleteCandidates, excess);
            }
            if (excessDeleted) {
                report.incrementAndSample(ReplicationManagerReport.HealthState.OVER_REPLICATED, container.containerID());
                int replicationFactor = container.getReplicationFactor().getNumber();
                LOG.info("Container {} has all unhealthy replicas and is over replicated. Expected replica count is {}, but found {}.", new Object[]{container.getContainerID(), replicationFactor, replicationFactor + excess});
            }
        }
    }

    private void handleUnderReplicatedAllUnhealthy(ContainerInfo container, List<ContainerReplica> replicas, ContainerPlacementStatus placementStatus, int additionalReplicasNeeded, ReplicationManagerReport report) {
        report.incrementAndSample(ReplicationManagerReport.HealthState.UNDER_REPLICATED, container.containerID());
        int numCloseCmdsSent = this.closeReplicasIfPossible(container, replicas);
        if (numCloseCmdsSent == 0) {
            LOG.info("Container {} is under replicated missing {} replicas with all replicas unhealthy. Copying unhealthy replicas.", (Object)container.getContainerID(), (Object)additionalReplicasNeeded);
            try {
                this.replicateAnyWithTopology(container, this.getReplicationSources(container, replicas, new StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State[0]), placementStatus, additionalReplicasNeeded, replicas);
            }
            catch (SCMException e) {
                LOG.warn("Could not fix container {} with replicas {}.", new Object[]{container, replicas, e});
            }
        }
    }

    private void deleteExcess(ContainerInfo container, List<ContainerReplica> deleteCandidates, int excess) {
        deleteCandidates.removeIf(r -> r.getDatanodeDetails().getPersistedOpState() != HddsProtos.NodeOperationalState.IN_SERVICE);
        deleteCandidates.stream().limit(excess).forEach(r -> this.sendDeleteCommand(container, r.getDatanodeDetails(), true));
    }

    private void deleteExcessWithTopology(int excess, ContainerInfo container, List<ContainerReplica> eligibleReplicas) {
        if (excess > 0) {
            HashSet<ContainerReplica> eligibleSet = new HashSet<ContainerReplica>(eligibleReplicas);
            int replicationFactor = container.getReplicationConfig().getRequiredNodes();
            ContainerPlacementStatus ps = this.getPlacementStatus(eligibleSet, replicationFactor);
            for (ContainerReplica r : eligibleReplicas) {
                if (excess <= 0) break;
                eligibleSet.remove(r);
                ContainerPlacementStatus nowPS = this.getPlacementStatus(eligibleSet, replicationFactor);
                if (this.isPlacementStatusActuallyEqual(ps, nowPS)) {
                    this.sendDeleteCommand(container, r.getDatanodeDetails(), true);
                    --excess;
                    continue;
                }
                eligibleSet.add(r);
            }
            if (excess > 0) {
                LOG.info("The container {} is over replicated with {} excess replica. The excess replicas cannot be removed without violating the placement policy", (Object)container, (Object)excess);
            }
        }
    }

    private boolean deleteExcessWithNonUniqueOriginNodeIDs(ContainerInfo container, List<ContainerReplica> allReplicas, List<ContainerReplica> deleteCandidates, int excess) {
        boolean deleteCandidatesPresent;
        HashSet<ContainerReplica> allReplicasSet = new HashSet<ContainerReplica>(allReplicas);
        List<ContainerReplica> nonUniqueDeleteCandidates = ReplicationManagerUtil.findNonUniqueDeleteCandidates(allReplicasSet, deleteCandidates, dnd -> {
            try {
                return this.nodeManager.getNodeStatus((DatanodeDetails)dnd);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Exception while finding excess unhealthy replicas to delete for container {} with replicas {}.", new Object[]{container, allReplicas, e});
                return null;
            }
        });
        if (LOG.isDebugEnabled() && nonUniqueDeleteCandidates.size() < excess) {
            LOG.debug("Unable to delete {} excess replicas of container {}. Only {} replicas can be deleted to preserve unique origin node IDs for this unclosed container.", new Object[]{excess, container.getContainerID(), nonUniqueDeleteCandidates.size()});
        }
        boolean bl = deleteCandidatesPresent = !nonUniqueDeleteCandidates.isEmpty();
        if (deleteCandidatesPresent) {
            this.deleteExcess(container, nonUniqueDeleteCandidates, excess);
        }
        return deleteCandidatesPresent;
    }

    private void deleteExcessLowestBcsIDs(ContainerInfo container, List<ContainerReplica> deleteCandidates, int excess) {
        deleteCandidates.sort(Comparator.comparingLong(ContainerReplica::getSequenceId));
        this.deleteExcess(container, deleteCandidates, excess);
    }

    private void replicateAnyWithTopology(ContainerInfo container, List<ContainerReplica> sourceReplicas, ContainerPlacementStatus placementStatus, int additionalReplicasNeeded, List<ContainerReplica> allReplicas) throws SCMException {
        try {
            ContainerID id = container.containerID();
            List<DatanodeDetails> sourceDNs = sourceReplicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
            List<DatanodeDetails> replicationInFlight = this.inflightReplication.getDatanodeDetails(id);
            if (sourceDNs.size() > 0) {
                int replicationFactor = container.getReplicationConfig().getRequiredNodes();
                ArrayList<DatanodeDetails> targetReplicas = new ArrayList<DatanodeDetails>(sourceDNs);
                targetReplicas.addAll(replicationInFlight);
                ContainerPlacementStatus inFlightplacementStatus = this.containerPlacement.validateContainerPlacement(targetReplicas, replicationFactor);
                int misRepDelta = inFlightplacementStatus.misReplicationCount();
                int replicasNeeded = Math.max(additionalReplicasNeeded, misRepDelta);
                if (replicasNeeded <= 0) {
                    LOG.debug("Container {} meets replication requirement with inflight replicas", (Object)id);
                    return;
                }
                List<DatanodeDetails> excludeList = allReplicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
                excludeList.addAll(replicationInFlight);
                List<DatanodeDetails> selectedDatanodes = ReplicationManagerUtil.getTargetDatanodes(this.containerPlacement, replicasNeeded, null, excludeList, this.currentContainerSize, container);
                if (additionalReplicasNeeded > 0) {
                    LOG.info("Container {} is under replicated. Expected replica count is {}, but found {}.", new Object[]{id, replicationFactor, replicationFactor - additionalReplicasNeeded});
                }
                int newMisRepDelta = misRepDelta;
                if (misRepDelta > 0) {
                    LOG.info("Container: {}. {}", (Object)id, (Object)placementStatus.misReplicatedReason());
                    targetReplicas.addAll(selectedDatanodes);
                    newMisRepDelta = this.containerPlacement.validateContainerPlacement(targetReplicas, replicationFactor).misReplicationCount();
                }
                if (additionalReplicasNeeded > 0 || newMisRepDelta < misRepDelta) {
                    for (DatanodeDetails datanode : selectedDatanodes) {
                        this.sendReplicateCommand(container, datanode, sourceDNs);
                    }
                } else {
                    LOG.warn("Container {} is mis-replicated, requiring {} additional replicas. After selecting new nodes, mis-replication hasnot improved. No additional replicas will be scheduled", (Object)id, (Object)misRepDelta);
                }
            } else {
                LOG.warn("Cannot replicate container {}, no healthy datanodes with replica found.", (Object)container.containerID());
            }
        }
        catch (IllegalStateException ex) {
            LOG.warn("Exception while replicating container {}.", (Object)container.getContainerID(), (Object)ex);
        }
    }

    private void replicateEachSource(ContainerInfo container, List<ContainerReplica> sources, List<ContainerReplica> allReplicas) {
        List<DatanodeDetails> excludeList = allReplicas.stream().map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
        for (ContainerReplica replica : sources) {
            List<DatanodeDetails> replicationInFlight = this.inflightReplication.getDatanodeDetails(container.containerID());
            for (DatanodeDetails dn : replicationInFlight) {
                if (excludeList.contains(dn)) continue;
                excludeList.add(dn);
            }
            try {
                List<DatanodeDetails> target = ReplicationManagerUtil.getTargetDatanodes(this.containerPlacement, 1, null, excludeList, this.currentContainerSize, container);
                this.sendReplicateCommand(container, target.iterator().next(), (List<DatanodeDetails>)ImmutableList.of((Object)replica.getDatanodeDetails()));
            }
            catch (SCMException e) {
                LOG.warn("Exception while trying to replicate {} of container {}.", new Object[]{replica, container, e});
            }
        }
    }

    private void closeEmptyContainer(ContainerInfo containerInfo) {
        Duration waitTime = this.rmConf.getInterval().multipliedBy(5L);
        Instant closingTime = containerInfo.getStateEnterTime();
        try {
            if (this.clock.instant().isAfter(closingTime.plus(waitTime))) {
                this.containerManager.updateContainerState(containerInfo.containerID(), HddsProtos.LifeCycleEvent.CLOSE);
            }
        }
        catch (IOException | InvalidStateTransitionException e) {
            LOG.error("Failed to CLOSE the container {}", (Object)containerInfo.containerID(), (Object)e);
        }
    }

    public static final class MoveSchedulerImpl
    implements MoveScheduler {
        private Table<ContainerID, MoveDataNodePair> moveTable;
        private final DBTransactionBuffer transactionBuffer;
        private final Map<ContainerID, MoveDataNodePair> inflightMove;

        private MoveSchedulerImpl(Table<ContainerID, MoveDataNodePair> moveTable, DBTransactionBuffer transactionBuffer) throws IOException {
            this.moveTable = moveTable;
            this.transactionBuffer = transactionBuffer;
            this.inflightMove = new ConcurrentHashMap<ContainerID, MoveDataNodePair>();
            this.initialize();
        }

        @Override
        public void completeMove(HddsProtos.ContainerID contianerIDProto) {
            ContainerID cid = null;
            try {
                cid = ContainerID.getFromProtobuf((HddsProtos.ContainerID)contianerIDProto);
                this.transactionBuffer.removeFromBuffer(this.moveTable, (Object)cid);
            }
            catch (IOException e) {
                LOG.warn("Exception while completing move {}", (Object)cid);
            }
            this.inflightMove.remove(cid);
        }

        @Override
        public void startMove(HddsProtos.ContainerID contianerIDProto, HddsProtos.MoveDataNodePairProto mdnpp) throws IOException {
            ContainerID cid = null;
            MoveDataNodePair mp = null;
            try {
                cid = ContainerID.getFromProtobuf((HddsProtos.ContainerID)contianerIDProto);
                mp = MoveDataNodePair.getFromProtobuf((HddsProtos.MoveDataNodePairProto)mdnpp);
                if (!this.inflightMove.containsKey(cid)) {
                    this.transactionBuffer.addToBuffer(this.moveTable, (Object)cid, (Object)mp);
                    this.inflightMove.putIfAbsent(cid, mp);
                }
            }
            catch (IOException e) {
                LOG.warn("Exception while completing move {}", (Object)cid);
            }
        }

        @Override
        public MoveDataNodePair getMoveDataNodePair(ContainerID cid) {
            return this.inflightMove.get(cid);
        }

        @Override
        public void reinitialize(Table<ContainerID, MoveDataNodePair> mt) throws IOException {
            this.moveTable = mt;
            this.inflightMove.clear();
            this.initialize();
        }

        private void initialize() throws IOException {
            try (TableIterator iterator = this.moveTable.iterator();){
                while (iterator.hasNext()) {
                    Table.KeyValue kv = (Table.KeyValue)iterator.next();
                    ContainerID cid = (ContainerID)kv.getKey();
                    MoveDataNodePair mp = (MoveDataNodePair)kv.getValue();
                    Preconditions.assertNotNull((Object)cid, (String)"moved container id should not be null");
                    Preconditions.assertNotNull((Object)mp, (String)"MoveDataNodePair container id should not be null");
                    this.inflightMove.put(cid, mp);
                }
            }
        }

        @Override
        public Map<ContainerID, MoveDataNodePair> getInflightMove() {
            return this.inflightMove;
        }

        public static class Builder {
            private Table<ContainerID, MoveDataNodePair> moveTable;
            private DBTransactionBuffer transactionBuffer;
            private SCMRatisServer ratisServer;

            public Builder setRatisServer(SCMRatisServer scmRatisServer) {
                this.ratisServer = scmRatisServer;
                return this;
            }

            public Builder setMoveTable(Table<ContainerID, MoveDataNodePair> mt) {
                this.moveTable = mt;
                return this;
            }

            public Builder setDBTransactionBuffer(DBTransactionBuffer trxBuffer) {
                this.transactionBuffer = trxBuffer;
                return this;
            }

            public MoveScheduler build() throws IOException {
                Preconditions.assertNotNull(this.moveTable, (String)"moveTable is null");
                Preconditions.assertNotNull((Object)this.transactionBuffer, (String)"transactionBuffer is null");
                MoveSchedulerImpl impl = new MoveSchedulerImpl(this.moveTable, this.transactionBuffer);
                SCMHAInvocationHandler invocationHandler = new SCMHAInvocationHandler(SCMRatisProtocol.RequestType.MOVE, impl, this.ratisServer);
                return (MoveScheduler)Proxy.newProxyInstance(SCMHAInvocationHandler.class.getClassLoader(), new Class[]{MoveScheduler.class}, (InvocationHandler)invocationHandler);
            }
        }
    }

    public static interface MoveScheduler {
        @Replicate
        public void completeMove(HddsProtos.ContainerID var1) throws SCMException;

        @Replicate
        public void startMove(HddsProtos.ContainerID var1, HddsProtos.MoveDataNodePairProto var2) throws IOException;

        public MoveDataNodePair getMoveDataNodePair(ContainerID var1);

        public void reinitialize(Table<ContainerID, MoveDataNodePair> var1) throws IOException;

        public Map<ContainerID, MoveDataNodePair> getInflightMove();
    }

    @ConfigGroup(prefix="hdds.scm.replication")
    public static class LegacyReplicationManagerConfiguration {
        @Config(key="container.inflight.replication.limit", type=ConfigType.INT, defaultValue="0", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="This property is used to limit the maximum number of inflight replication.")
        private int containerInflightReplicationLimit = 0;
        @Config(key="container.inflight.deletion.limit", type=ConfigType.INT, defaultValue="0", tags={ConfigTag.SCM, ConfigTag.OZONE}, description="This property is used to limit the maximum number of inflight deletion.")
        private int containerInflightDeletionLimit = 0;

        public void setContainerInflightReplicationLimit(int replicationLimit) {
            this.containerInflightReplicationLimit = replicationLimit;
        }

        public void setContainerInflightDeletionLimit(int deletionLimit) {
            this.containerInflightDeletionLimit = deletionLimit;
        }

        public int getContainerInflightReplicationLimit() {
            return this.containerInflightReplicationLimit;
        }

        public int getContainerInflightDeletionLimit() {
            return this.containerInflightDeletionLimit;
        }
    }

    static class InflightMap {
        private final Map<ContainerID, List<InflightAction>> map = new ConcurrentHashMap<ContainerID, List<InflightAction>>();
        private final InflightType type;
        private final int sizeLimit;
        private final AtomicInteger inflightCount = new AtomicInteger();

        InflightMap(InflightType type, int sizeLimit) {
            this.type = type;
            this.sizeLimit = sizeLimit > 0 ? sizeLimit : Integer.MAX_VALUE;
        }

        boolean isReplication() {
            return this.type == InflightType.REPLICATION;
        }

        private List<InflightAction> get(ContainerID id) {
            return this.map.get(id);
        }

        boolean containsKey(ContainerID id) {
            return this.map.containsKey(id);
        }

        int inflightActionCount(ContainerID id) {
            return Optional.ofNullable(this.map.get(id)).map(List::size).orElse(0);
        }

        int containerCount() {
            return this.map.size();
        }

        boolean isFull() {
            return this.inflightCount.get() >= this.sizeLimit;
        }

        void clear() {
            this.map.clear();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        void iterate(ContainerID id, Predicate<InflightAction> processor) {
            List<InflightAction> actions;
            while (true) {
                if ((actions = this.get(id)) == null) {
                    return;
                }
                List<InflightAction> list = actions;
                synchronized (list) {
                    if (this.get(id) == actions) break;
                }
            }
            {
                Iterator<InflightAction> i = actions.iterator();
                while (true) {
                    if (!i.hasNext()) {
                        this.map.computeIfPresent(id, (k, v) -> v == actions && v.isEmpty() ? null : v);
                        return;
                    }
                    boolean remove = processor.test(i.next());
                    if (!remove) continue;
                    i.remove();
                    this.inflightCount.decrementAndGet();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        boolean add(ContainerID id, InflightAction a) {
            List actions;
            int previous = this.inflightCount.getAndUpdate(n -> n < this.sizeLimit ? n + 1 : n);
            if (previous >= this.sizeLimit) {
                return false;
            }
            while (true) {
                List list = actions = this.map.computeIfAbsent(id, key -> new LinkedList());
                synchronized (list) {
                    if (this.get(id) == actions) break;
                }
            }
            {
                boolean added = actions.add(a);
                if (!added) {
                    this.inflightCount.decrementAndGet();
                }
                return added;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        List<DatanodeDetails> getDatanodeDetails(ContainerID id) {
            List<InflightAction> actions;
            while ((actions = this.get(id)) != null) {
                List<InflightAction> list = actions;
                synchronized (list) {
                    if (this.get(id) == actions) {
                        return actions.stream().map(InflightAction::getDatanode).collect(Collectors.toList());
                    }
                }
            }
            return Collections.emptyList();
        }
    }
}

