/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.balancer;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.scm.PlacementPolicyValidateProxy;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaNotFoundException;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancer;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerConfiguration;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerMetrics;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerSelectionCriteria;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerMoveSelection;
import org.apache.hadoop.hdds.scm.container.balancer.FindSourceGreedy;
import org.apache.hadoop.hdds.scm.container.balancer.FindSourceStrategy;
import org.apache.hadoop.hdds.scm.container.balancer.FindTargetGreedyByNetworkTopology;
import org.apache.hadoop.hdds.scm.container.balancer.FindTargetGreedyByUsageInfo;
import org.apache.hadoop.hdds.scm.container.balancer.FindTargetStrategy;
import org.apache.hadoop.hdds.scm.container.balancer.MoveManager;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.node.DatanodeUsageInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.apache.hadoop.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerBalancerTask
implements Runnable {
    public static final Logger LOG = LoggerFactory.getLogger(ContainerBalancerTask.class);
    private NodeManager nodeManager;
    private ContainerManager containerManager;
    private ReplicationManager replicationManager;
    private MoveManager moveManager;
    private OzoneConfiguration ozoneConfiguration;
    private ContainerBalancer containerBalancer;
    private final SCMContext scmContext;
    private double threshold;
    private int totalNodesInCluster;
    private double maxDatanodesRatioToInvolvePerIteration;
    private long maxSizeToMovePerIteration;
    private int countDatanodesInvolvedPerIteration;
    private long sizeScheduledForMoveInLatestIteration;
    private long sizeActuallyMovedInLatestIteration;
    private int iterations;
    private List<DatanodeUsageInfo> unBalancedNodes;
    private List<DatanodeUsageInfo> overUtilizedNodes;
    private List<DatanodeUsageInfo> underUtilizedNodes;
    private List<DatanodeUsageInfo> withinThresholdUtilizedNodes;
    private Set<String> excludeNodes;
    private Set<String> includeNodes;
    private ContainerBalancerConfiguration config;
    private ContainerBalancerMetrics metrics;
    private long clusterCapacity;
    private long clusterRemaining;
    private double clusterAvgUtilisation;
    private PlacementPolicyValidateProxy placementPolicyValidateProxy;
    private NetworkTopology networkTopology;
    private double upperLimit;
    private double lowerLimit;
    private ContainerBalancerSelectionCriteria selectionCriteria;
    private volatile Status taskStatus = Status.RUNNING;
    private final Map<ContainerID, DatanodeDetails> containerToSourceMap;
    private final Map<ContainerID, DatanodeDetails> containerToTargetMap;
    private Set<DatanodeDetails> selectedTargets;
    private Set<DatanodeDetails> selectedSources;
    private FindTargetStrategy findTargetStrategy;
    private FindSourceStrategy findSourceStrategy;
    private Map<ContainerMoveSelection, CompletableFuture<MoveManager.MoveResult>> moveSelectionToFutureMap;
    private IterationResult iterationResult;
    private int nextIterationIndex;
    private boolean delayStart;

    public ContainerBalancerTask(StorageContainerManager scm, int nextIterationIndex, ContainerBalancer containerBalancer, ContainerBalancerMetrics metrics, ContainerBalancerConfiguration config, boolean delayStart) {
        this.nodeManager = scm.getScmNodeManager();
        this.containerManager = scm.getContainerManager();
        this.replicationManager = scm.getReplicationManager();
        this.moveManager = scm.getMoveManager();
        this.moveManager.setMoveTimeout(config.getMoveTimeout().toMillis());
        this.moveManager.setReplicationTimeout(config.getMoveReplicationTimeout().toMillis());
        this.delayStart = delayStart;
        this.ozoneConfiguration = scm.getConfiguration();
        this.containerBalancer = containerBalancer;
        this.config = config;
        this.metrics = metrics;
        this.scmContext = scm.getScmContext();
        this.overUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.underUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.withinThresholdUtilizedNodes = new ArrayList<DatanodeUsageInfo>();
        this.unBalancedNodes = new ArrayList<DatanodeUsageInfo>();
        this.placementPolicyValidateProxy = scm.getPlacementPolicyValidateProxy();
        this.networkTopology = scm.getClusterMap();
        this.nextIterationIndex = nextIterationIndex;
        this.containerToSourceMap = new HashMap<ContainerID, DatanodeDetails>();
        this.containerToTargetMap = new HashMap<ContainerID, DatanodeDetails>();
        this.selectedSources = new HashSet<DatanodeDetails>();
        this.selectedTargets = new HashSet<DatanodeDetails>();
        this.findSourceStrategy = new FindSourceGreedy(this.nodeManager);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            if (this.delayStart) {
                long delayDuration = this.ozoneConfiguration.getTimeDuration("hdds.scm.wait.time.after.safemode.exit", "5m", TimeUnit.SECONDS);
                LOG.info("ContainerBalancer will sleep for {} seconds before starting balancing.", (Object)delayDuration);
                Thread.sleep(Duration.ofSeconds(delayDuration).toMillis());
            }
            this.balance();
        }
        catch (Exception e) {
            LOG.error("Container Balancer is stopped abnormally, ", (Throwable)e);
        }
        finally {
            ContainerBalancerTask containerBalancerTask = this;
            synchronized (containerBalancerTask) {
                this.taskStatus = Status.STOPPED;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        ContainerBalancerTask containerBalancerTask = this;
        synchronized (containerBalancerTask) {
            if (this.taskStatus == Status.RUNNING) {
                this.taskStatus = Status.STOPPING;
            }
        }
    }

    private void balance() {
        this.iterations = this.config.getIterations();
        if (this.iterations == -1) {
            this.iterations = Integer.MAX_VALUE;
        }
        for (int i = this.nextIterationIndex; i < this.iterations && this.isBalancerRunning(); ++i) {
            this.resetState();
            if (this.config.getTriggerDuEnable().booleanValue()) {
                this.nodeManager.refreshAllHealthyDnUsageInfo();
                try {
                    long nodeReportInterval = this.ozoneConfiguration.getTimeDuration("hdds.node.report.interval", "60s", TimeUnit.MILLISECONDS);
                    long sleepTime = 3L * nodeReportInterval;
                    LOG.info("ContainerBalancer will sleep for {} ms while waiting for updated usage information from Datanodes.", (Object)sleepTime);
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException e) {
                    LOG.info("Container Balancer was interrupted while waiting fordatanodes refreshing volume usage info");
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            if (!this.isBalancerRunning()) {
                return;
            }
            if (!this.initializeIteration()) {
                if (!this.isBalancerRunning()) {
                    return;
                }
                this.tryStopWithSaveConfiguration("Could not initialize ContainerBalancer's iteration number " + i);
                return;
            }
            IterationResult iR = this.doIteration();
            this.metrics.incrementNumIterations(1L);
            LOG.info("Result of this iteration of Container Balancer: {}", (Object)iR);
            if (iR == IterationResult.CAN_NOT_BALANCE_ANY_MORE) {
                this.tryStopWithSaveConfiguration(iR.toString());
                return;
            }
            if (iR == IterationResult.ITERATION_COMPLETED) {
                try {
                    this.saveConfiguration(this.config, true, i + 1);
                }
                catch (IOException | TimeoutException e) {
                    LOG.warn("Could not persist next iteration index value for ContainerBalancer after completing an iteration", (Throwable)e);
                }
            }
            if (!this.isBalancerRunning()) {
                return;
            }
            if (i == this.iterations - 1) continue;
            try {
                Thread.sleep(this.config.getBalancingInterval().toMillis());
                continue;
            }
            catch (InterruptedException e) {
                LOG.info("Container Balancer was interrupted while waiting for next iteration.");
                Thread.currentThread().interrupt();
                return;
            }
        }
        this.tryStopWithSaveConfiguration("Completed all iterations.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void tryStopWithSaveConfiguration(String stopReason) {
        ContainerBalancerTask containerBalancerTask = this;
        synchronized (containerBalancerTask) {
            try {
                LOG.info("Save Configuration for stopping. Reason: {}", (Object)stopReason);
                this.saveConfiguration(this.config, false, 0);
                this.stop();
            }
            catch (IOException | TimeoutException e) {
                LOG.warn("Save configuration failed. Reason for stopping: {}", (Object)stopReason, (Object)e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void saveConfiguration(ContainerBalancerConfiguration configuration, boolean shouldRun, int index) throws IOException, TimeoutException {
        if (!this.isValidSCMState()) {
            LOG.warn("Save configuration is not allowed as not in valid State.");
            return;
        }
        ContainerBalancerTask containerBalancerTask = this;
        synchronized (containerBalancerTask) {
            if (this.isBalancerRunning()) {
                this.containerBalancer.saveConfiguration(configuration, shouldRun, index);
            }
        }
    }

    private boolean initializeIteration() {
        if (!this.isValidSCMState()) {
            return false;
        }
        List<DatanodeUsageInfo> datanodeUsageInfos = this.nodeManager.getMostOrLeastUsedDatanodes(true);
        if (datanodeUsageInfos.isEmpty()) {
            LOG.warn("Received an empty list of datanodes from Node Manager when trying to identify which nodes to balance");
            return false;
        }
        this.threshold = this.config.getThresholdAsRatio();
        this.maxDatanodesRatioToInvolvePerIteration = this.config.getMaxDatanodesRatioToInvolvePerIteration();
        this.maxSizeToMovePerIteration = this.config.getMaxSizeToMovePerIteration();
        this.findTargetStrategy = this.config.getNetworkTopologyEnable() != false ? new FindTargetGreedyByNetworkTopology(this.containerManager, this.placementPolicyValidateProxy, this.nodeManager, this.networkTopology) : new FindTargetGreedyByUsageInfo(this.containerManager, this.placementPolicyValidateProxy, this.nodeManager);
        this.excludeNodes = this.config.getExcludeNodes();
        this.includeNodes = this.config.getIncludeNodes();
        datanodeUsageInfos.removeIf(datanodeUsageInfo -> this.shouldExcludeDatanode(datanodeUsageInfo.getDatanodeDetails()));
        this.totalNodesInCluster = datanodeUsageInfos.size();
        this.clusterAvgUtilisation = this.calculateAvgUtilization(datanodeUsageInfos);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Average utilization of the cluster is {}", (Object)this.clusterAvgUtilisation);
        }
        this.upperLimit = this.clusterAvgUtilisation + this.threshold;
        this.lowerLimit = this.clusterAvgUtilisation - this.threshold;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Lower limit for utilization is {} and Upper limit for utilization is {}", (Object)this.lowerLimit, (Object)this.upperLimit);
        }
        long totalOverUtilizedBytes = 0L;
        long totalUnderUtilizedBytes = 0L;
        for (DatanodeUsageInfo datanodeUsageInfo2 : datanodeUsageInfos) {
            if (!this.isBalancerRunning()) {
                return false;
            }
            double utilization = datanodeUsageInfo2.calculateUtilization();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Utilization for node {} with capacity {}B, used {}B, and remaining {}B is {}", new Object[]{datanodeUsageInfo2.getDatanodeDetails().getUuidString(), datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), datanodeUsageInfo2.getScmNodeStat().getScmUsed().get(), datanodeUsageInfo2.getScmNodeStat().getRemaining().get(), utilization});
            }
            if (Double.compare(utilization, this.upperLimit) > 0) {
                this.overUtilizedNodes.add(datanodeUsageInfo2);
                this.metrics.incrementNumDatanodesUnbalanced(1L);
                long overUtilizedBytes = this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), utilization) - this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), this.upperLimit);
                totalOverUtilizedBytes += overUtilizedBytes;
                continue;
            }
            if (Double.compare(utilization, this.lowerLimit) < 0) {
                this.underUtilizedNodes.add(datanodeUsageInfo2);
                this.metrics.incrementNumDatanodesUnbalanced(1L);
                long underUtilizedBytes = this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), this.lowerLimit) - this.ratioToBytes(datanodeUsageInfo2.getScmNodeStat().getCapacity().get(), utilization);
                totalUnderUtilizedBytes += underUtilizedBytes;
                continue;
            }
            this.withinThresholdUtilizedNodes.add(datanodeUsageInfo2);
        }
        this.metrics.incrementDataSizeUnbalancedGB(Math.max(totalOverUtilizedBytes, totalUnderUtilizedBytes) / 0x40000000L);
        Collections.reverse(this.underUtilizedNodes);
        this.unBalancedNodes = new ArrayList<DatanodeUsageInfo>(this.overUtilizedNodes.size() + this.underUtilizedNodes.size());
        this.unBalancedNodes.addAll(this.overUtilizedNodes);
        this.unBalancedNodes.addAll(this.underUtilizedNodes);
        if (this.unBalancedNodes.isEmpty()) {
            LOG.info("Did not find any unbalanced Datanodes.");
            return false;
        }
        LOG.info("Container Balancer has identified {} Over-Utilized and {} Under-Utilized Datanodes that need to be balanced.", (Object)this.overUtilizedNodes.size(), (Object)this.underUtilizedNodes.size());
        if (LOG.isDebugEnabled()) {
            this.overUtilizedNodes.forEach(entry -> LOG.debug("Datanode {} {} is Over-Utilized.", (Object)entry.getDatanodeDetails().getHostName(), (Object)entry.getDatanodeDetails().getUuid()));
            this.underUtilizedNodes.forEach(entry -> LOG.debug("Datanode {} {} is Under-Utilized.", (Object)entry.getDatanodeDetails().getHostName(), (Object)entry.getDatanodeDetails().getUuid()));
        }
        this.selectionCriteria = new ContainerBalancerSelectionCriteria(this.config, this.nodeManager, this.replicationManager, this.containerManager, this.findSourceStrategy);
        return true;
    }

    private boolean isValidSCMState() {
        if (this.scmContext.isInSafeMode()) {
            LOG.error("Container Balancer cannot operate while SCM is in Safe Mode.");
            return false;
        }
        if (!this.scmContext.isLeaderReady()) {
            LOG.warn("Current SCM is not the leader.");
            return false;
        }
        return true;
    }

    private IterationResult doIteration() {
        List<DatanodeUsageInfo> potentialTargets = this.getPotentialTargets();
        this.findTargetStrategy.reInitialize(potentialTargets, this.config, this.upperLimit);
        this.findSourceStrategy.reInitialize(this.getPotentialSources(), this.config, this.lowerLimit);
        this.moveSelectionToFutureMap = new HashMap<ContainerMoveSelection, CompletableFuture<MoveManager.MoveResult>>(this.unBalancedNodes.size());
        boolean isMoveGeneratedInThisIteration = false;
        this.iterationResult = IterationResult.ITERATION_COMPLETED;
        boolean canAdaptWhenNearingLimits = true;
        boolean canAdaptOnReachingLimits = true;
        while (true) {
            DatanodeDetails source;
            if (!this.isBalancerRunning()) {
                this.iterationResult = IterationResult.ITERATION_INTERRUPTED;
                break;
            }
            if (this.reachedMaxSizeToMovePerIteration()) break;
            if (canAdaptWhenNearingLimits && this.adaptWhenNearingIterationLimits()) {
                canAdaptWhenNearingLimits = false;
            }
            if (canAdaptOnReachingLimits && this.adaptOnReachingIterationLimits()) {
                canAdaptOnReachingLimits = false;
                canAdaptWhenNearingLimits = false;
            }
            if ((source = this.findSourceStrategy.getNextCandidateSourceDataNode()) == null) break;
            ContainerMoveSelection moveSelection = this.matchSourceWithTarget(source);
            if (moveSelection != null) {
                if (!this.processMoveSelection(source, moveSelection)) continue;
                isMoveGeneratedInThisIteration = true;
                continue;
            }
            this.findSourceStrategy.removeCandidateSourceDataNode(source);
        }
        this.checkIterationResults(isMoveGeneratedInThisIteration);
        return this.iterationResult;
    }

    private boolean processMoveSelection(DatanodeDetails source, ContainerMoveSelection moveSelection) {
        ContainerInfo containerInfo;
        ContainerID containerID = moveSelection.getContainerID();
        if (this.containerToSourceMap.containsKey(containerID) || this.containerToTargetMap.containsKey(containerID)) {
            LOG.warn("Container {} has already been selected for move from source {} to target {} earlier. Not moving this container again.", new Object[]{containerID, this.containerToSourceMap.get(containerID), this.containerToTargetMap.get(containerID)});
            return false;
        }
        try {
            containerInfo = this.containerManager.getContainer(containerID);
        }
        catch (ContainerNotFoundException e) {
            LOG.warn("Could not get container {} from Container Manager before starting a container move", (Object)containerID, (Object)e);
            return false;
        }
        LOG.info("ContainerBalancer is trying to move container {} with size {}B from source datanode {} to target datanode {}", new Object[]{containerID.toString(), containerInfo.getUsedBytes(), source.getUuidString(), moveSelection.getTargetNode().getUuidString()});
        if (this.moveContainer(source, moveSelection)) {
            this.updateTargetsAndSelectionCriteria(moveSelection, source);
        }
        return true;
    }

    private void checkIterationResults(boolean isMoveGeneratedInThisIteration) {
        if (!isMoveGeneratedInThisIteration) {
            this.iterationResult = IterationResult.CAN_NOT_BALANCE_ANY_MORE;
        } else {
            this.checkIterationMoveResults();
        }
    }

    private void checkIterationMoveResults() {
        this.countDatanodesInvolvedPerIteration = 0;
        CompletableFuture<Void> allFuturesResult = CompletableFuture.allOf(this.moveSelectionToFutureMap.values().toArray(new CompletableFuture[this.moveSelectionToFutureMap.size()]));
        try {
            allFuturesResult.get(this.config.getMoveTimeout().toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            LOG.warn("Container balancer is interrupted");
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException e) {
            long timeoutCounts = this.cancelMovesThatExceedTimeoutDuration();
            LOG.warn("{} Container moves are canceled.", (Object)timeoutCounts);
            this.metrics.incrementNumContainerMovesTimeoutInLatestIteration(timeoutCounts);
        }
        catch (ExecutionException e) {
            LOG.error("Got exception while checkIterationMoveResults", (Throwable)e);
        }
        this.countDatanodesInvolvedPerIteration = this.selectedSources.size() + this.selectedTargets.size();
        this.metrics.incrementNumDatanodesInvolvedInLatestIteration(this.countDatanodesInvolvedPerIteration);
        this.metrics.incrementNumContainerMovesScheduled(this.metrics.getNumContainerMovesScheduledInLatestIteration());
        this.metrics.incrementNumContainerMovesCompleted(this.metrics.getNumContainerMovesCompletedInLatestIteration());
        this.metrics.incrementNumContainerMovesTimeout(this.metrics.getNumContainerMovesTimeoutInLatestIteration());
        this.metrics.incrementDataSizeMovedGBInLatestIteration(this.sizeActuallyMovedInLatestIteration / 0x40000000L);
        this.metrics.incrementDataSizeMovedGB(this.metrics.getDataSizeMovedGBInLatestIteration());
        this.metrics.incrementNumContainerMovesFailed(this.metrics.getNumContainerMovesFailedInLatestIteration());
        LOG.info("Iteration Summary. Number of Datanodes involved: {}. Size moved: {} ({} Bytes). Number of Container moves completed: {}.", new Object[]{this.countDatanodesInvolvedPerIteration, StringUtils.byteDesc((long)this.sizeActuallyMovedInLatestIteration), this.sizeActuallyMovedInLatestIteration, this.metrics.getNumContainerMovesCompletedInLatestIteration()});
    }

    private long cancelMovesThatExceedTimeoutDuration() {
        Set<Map.Entry<ContainerMoveSelection, CompletableFuture<MoveManager.MoveResult>>> entries = this.moveSelectionToFutureMap.entrySet();
        Iterator<Map.Entry<ContainerMoveSelection, CompletableFuture<MoveManager.MoveResult>>> iterator = entries.iterator();
        int numCancelled = 0;
        while (iterator.hasNext()) {
            Map.Entry<ContainerMoveSelection, CompletableFuture<MoveManager.MoveResult>> entry = iterator.next();
            if (entry.getValue().isDone()) continue;
            LOG.warn("Container move timed out for container {} from source {} to target {}.", new Object[]{entry.getKey().getContainerID(), this.containerToSourceMap.get(entry.getKey().getContainerID()).getUuidString(), entry.getKey().getTargetNode().getUuidString()});
            entry.getValue().cancel(true);
            ++numCancelled;
        }
        return numCancelled;
    }

    private ContainerMoveSelection matchSourceWithTarget(DatanodeDetails source) {
        ContainerMoveSelection moveSelection;
        NavigableSet<ContainerID> candidateContainers = this.selectionCriteria.getCandidateContainers(source, this.sizeScheduledForMoveInLatestIteration);
        if (candidateContainers.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ContainerBalancer could not find any candidate containers for datanode {}", (Object)source.getUuidString());
            }
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("ContainerBalancer is finding suitable target for source datanode {}", (Object)source.getUuidString());
        }
        if ((moveSelection = this.findTargetStrategy.findTargetForContainerMove(source, candidateContainers)) == null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ContainerBalancer could not find a suitable target for source node {}.", (Object)source.getUuidString());
            }
            return null;
        }
        LOG.info("ContainerBalancer matched source datanode {} with target datanode {} for container move.", (Object)source.getUuidString(), (Object)moveSelection.getTargetNode().getUuidString());
        return moveSelection;
    }

    private boolean reachedMaxSizeToMovePerIteration() {
        if (this.sizeScheduledForMoveInLatestIteration >= this.maxSizeToMovePerIteration) {
            LOG.warn("Reached max size to move limit. {} bytes have already been scheduled for balancing and the limit is {} bytes.", (Object)this.sizeScheduledForMoveInLatestIteration, (Object)this.maxSizeToMovePerIteration);
            return true;
        }
        return false;
    }

    private boolean adaptWhenNearingIterationLimits() {
        int maxDatanodesToInvolve = (int)(this.maxDatanodesRatioToInvolvePerIteration * (double)this.totalNodesInCluster);
        if (this.countDatanodesInvolvedPerIteration + 1 == maxDatanodesToInvolve) {
            this.findTargetStrategy.resetPotentialTargets(this.selectedTargets);
            LOG.debug("Approaching max datanodes to involve limit. {} datanodes have already been selected for balancing and the limit is {}. Only already selected targets can be selected as targets now.", (Object)this.countDatanodesInvolvedPerIteration, (Object)maxDatanodesToInvolve);
            return true;
        }
        return false;
    }

    private boolean adaptOnReachingIterationLimits() {
        int maxDatanodesToInvolve = (int)(this.maxDatanodesRatioToInvolvePerIteration * (double)this.totalNodesInCluster);
        if (this.countDatanodesInvolvedPerIteration == maxDatanodesToInvolve) {
            this.findTargetStrategy.resetPotentialTargets(this.selectedTargets);
            this.findSourceStrategy.resetPotentialSources(this.selectedSources);
            LOG.debug("Reached max datanodes to involve limit. {} datanodes have already been selected for balancing and the limit is {}. Only already selected sources and targets can be involved in balancing now.", (Object)this.countDatanodesInvolvedPerIteration, (Object)maxDatanodesToInvolve);
            return true;
        }
        return false;
    }

    private boolean moveContainer(DatanodeDetails source, ContainerMoveSelection moveSelection) {
        CompletionStage<MoveManager.MoveResult> future;
        ContainerID containerID = moveSelection.getContainerID();
        try {
            ContainerInfo containerInfo = this.containerManager.getContainer(containerID);
            future = this.replicationManager.getConfig().isLegacyEnabled() ? this.replicationManager.move(containerID, source, moveSelection.getTargetNode()) : this.moveManager.move(containerID, source, moveSelection.getTargetNode());
            this.metrics.incrementNumContainerMovesScheduledInLatestIteration(1L);
            future = ((CompletableFuture)future).whenComplete((result, ex) -> {
                this.metrics.incrementCurrentIterationContainerMoveMetric((MoveManager.MoveResult)((Object)result), 1L);
                if (ex != null) {
                    LOG.info("Container move for container {} from source {} to target {} failed with exceptions.", new Object[]{containerID.toString(), source.getUuidString(), moveSelection.getTargetNode().getUuidString(), ex});
                    this.metrics.incrementNumContainerMovesFailedInLatestIteration(1L);
                } else if (result == MoveManager.MoveResult.COMPLETED) {
                    this.sizeActuallyMovedInLatestIteration += containerInfo.getUsedBytes();
                    LOG.debug("Container move completed for container {} from source {} to target {}", new Object[]{containerID, source.getUuidString(), moveSelection.getTargetNode().getUuidString()});
                } else {
                    LOG.warn("Container move for container {} from source {} to target {} failed: {}", new Object[]{moveSelection.getContainerID(), source.getUuidString(), moveSelection.getTargetNode().getUuidString(), result});
                }
            });
        }
        catch (ContainerNotFoundException e) {
            LOG.warn("Could not find Container {} for container move", (Object)containerID, (Object)e);
            this.metrics.incrementNumContainerMovesFailedInLatestIteration(1L);
            return false;
        }
        catch (TimeoutException | ContainerReplicaNotFoundException | NodeNotFoundException e) {
            LOG.warn("Container move failed for container {}", (Object)containerID, (Object)e);
            this.metrics.incrementNumContainerMovesFailedInLatestIteration(1L);
            return false;
        }
        if (((CompletableFuture)future).isDone()) {
            if (((CompletableFuture)future).isCompletedExceptionally()) {
                return false;
            }
            MoveManager.MoveResult result2 = (MoveManager.MoveResult)((Object)((CompletableFuture)future).join());
            this.moveSelectionToFutureMap.put(moveSelection, (CompletableFuture<MoveManager.MoveResult>)future);
            return result2 == MoveManager.MoveResult.COMPLETED;
        }
        this.moveSelectionToFutureMap.put(moveSelection, (CompletableFuture<MoveManager.MoveResult>)future);
        return true;
    }

    private void updateTargetsAndSelectionCriteria(ContainerMoveSelection moveSelection, DatanodeDetails source) {
        ContainerID containerID = moveSelection.getContainerID();
        DatanodeDetails target = moveSelection.getTargetNode();
        if (!this.selectedSources.contains(source)) {
            ++this.countDatanodesInvolvedPerIteration;
        }
        if (!this.selectedTargets.contains(target)) {
            ++this.countDatanodesInvolvedPerIteration;
        }
        this.incSizeSelectedForMoving(source, moveSelection);
        this.containerToSourceMap.put(containerID, source);
        this.containerToTargetMap.put(containerID, target);
        this.selectedTargets.add(target);
        this.selectedSources.add(source);
        this.selectionCriteria.setSelectedContainers(new HashSet<ContainerID>(this.containerToSourceMap.keySet()));
    }

    private long ratioToBytes(Long nodeCapacity, double utilizationRatio) {
        return (long)((double)nodeCapacity.longValue() * utilizationRatio);
    }

    @VisibleForTesting
    double calculateAvgUtilization(List<DatanodeUsageInfo> nodes) {
        if (nodes.size() == 0) {
            LOG.warn("No nodes to calculate average utilization for in ContainerBalancer.");
            return 0.0;
        }
        SCMNodeStat aggregatedStats = new SCMNodeStat(0L, 0L, 0L, 0L, 0L);
        for (DatanodeUsageInfo node : nodes) {
            aggregatedStats.add(node.getScmNodeStat());
        }
        this.clusterCapacity = aggregatedStats.getCapacity().get();
        this.clusterRemaining = aggregatedStats.getRemaining().get();
        return (double)(this.clusterCapacity - this.clusterRemaining) / (double)this.clusterCapacity;
    }

    private List<DatanodeUsageInfo> getPotentialTargets() {
        return this.underUtilizedNodes;
    }

    private List<DatanodeUsageInfo> getPotentialSources() {
        return this.overUtilizedNodes;
    }

    private boolean shouldExcludeDatanode(DatanodeDetails datanode) {
        if (this.excludeNodes.contains(datanode.getHostName()) || this.excludeNodes.contains(datanode.getIpAddress())) {
            return true;
        }
        if (!this.includeNodes.isEmpty()) {
            return !this.includeNodes.contains(datanode.getHostName()) && !this.includeNodes.contains(datanode.getIpAddress());
        }
        return false;
    }

    private void incSizeSelectedForMoving(DatanodeDetails source, ContainerMoveSelection moveSelection) {
        ContainerInfo container;
        DatanodeDetails target = moveSelection.getTargetNode();
        try {
            container = this.containerManager.getContainer(moveSelection.getContainerID());
        }
        catch (ContainerNotFoundException e) {
            LOG.warn("Could not find Container {} while matching source and target nodes in ContainerBalancer", (Object)moveSelection.getContainerID(), (Object)e);
            return;
        }
        long size = container.getUsedBytes();
        this.sizeScheduledForMoveInLatestIteration += size;
        this.findSourceStrategy.increaseSizeLeaving(source, size);
        this.findTargetStrategy.increaseSizeEntering(target, size);
    }

    private void resetState() {
        this.moveManager.resetState();
        this.clusterCapacity = 0L;
        this.clusterRemaining = 0L;
        this.overUtilizedNodes.clear();
        this.underUtilizedNodes.clear();
        this.unBalancedNodes.clear();
        this.containerToSourceMap.clear();
        this.containerToTargetMap.clear();
        this.selectedSources.clear();
        this.selectedTargets.clear();
        this.countDatanodesInvolvedPerIteration = 0;
        this.sizeScheduledForMoveInLatestIteration = 0L;
        this.sizeActuallyMovedInLatestIteration = 0L;
        this.metrics.resetDataSizeMovedGBInLatestIteration();
        this.metrics.resetNumContainerMovesCompletedInLatestIteration();
        this.metrics.resetNumContainerMovesTimeoutInLatestIteration();
        this.metrics.resetNumDatanodesInvolvedInLatestIteration();
        this.metrics.resetDataSizeUnbalancedGB();
        this.metrics.resetNumDatanodesUnbalanced();
        this.metrics.resetNumContainerMovesFailedInLatestIteration();
    }

    private boolean isBalancerRunning() {
        return this.taskStatus == Status.RUNNING;
    }

    @VisibleForTesting
    List<DatanodeUsageInfo> getUnBalancedNodes() {
        return this.unBalancedNodes;
    }

    @VisibleForTesting
    Map<ContainerID, DatanodeDetails> getContainerToSourceMap() {
        return this.containerToSourceMap;
    }

    @VisibleForTesting
    Map<ContainerID, DatanodeDetails> getContainerToTargetMap() {
        return this.containerToTargetMap;
    }

    @VisibleForTesting
    Set<DatanodeDetails> getSelectedTargets() {
        return this.selectedTargets;
    }

    @VisibleForTesting
    int getCountDatanodesInvolvedPerIteration() {
        return this.countDatanodesInvolvedPerIteration;
    }

    @VisibleForTesting
    public long getSizeScheduledForMoveInLatestIteration() {
        return this.sizeScheduledForMoveInLatestIteration;
    }

    public ContainerBalancerMetrics getMetrics() {
        return this.metrics;
    }

    @VisibleForTesting
    IterationResult getIterationResult() {
        return this.iterationResult;
    }

    @VisibleForTesting
    void setConfig(ContainerBalancerConfiguration config) {
        this.config = config;
    }

    @VisibleForTesting
    void setTaskStatus(Status taskStatus) {
        this.taskStatus = taskStatus;
    }

    public Status getBalancerStatus() {
        return this.taskStatus;
    }

    public String toString() {
        String status = String.format("%nContainer Balancer Task status:%n%-30s %s%n%-30s %b%n", "Key", "Value", "Running", this.isBalancerRunning());
        return status + this.config.toString();
    }

    static enum Status {
        RUNNING,
        STOPPING,
        STOPPED;

    }

    static enum IterationResult {
        ITERATION_COMPLETED,
        ITERATION_INTERRUPTED,
        CAN_NOT_BALANCE_ANY_MORE;

    }
}

