/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.lease;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.ozone.lease.Lease;
import org.apache.hadoop.ozone.lease.LeaseAlreadyExistException;
import org.apache.hadoop.ozone.lease.LeaseCallbackExecutor;
import org.apache.hadoop.ozone.lease.LeaseExpiredException;
import org.apache.hadoop.ozone.lease.LeaseManagerNotRunningException;
import org.apache.hadoop.ozone.lease.LeaseNotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaseManager<T> {
    private static final Logger LOG = LoggerFactory.getLogger(LeaseManager.class);
    private final String name;
    private final long defaultTimeout;
    private Map<T, Lease<T>> activeLeases;
    private Semaphore semaphore = new Semaphore(0);
    private LeaseMonitor leaseMonitor;
    private Thread leaseMonitorThread;
    private boolean isRunning;

    public LeaseManager(String name, long defaultTimeout) {
        this.name = name + "LeaseManager";
        this.defaultTimeout = defaultTimeout;
    }

    public void start() {
        LOG.debug("Starting {} service", (Object)this.name);
        this.activeLeases = new ConcurrentHashMap<T, Lease<T>>();
        this.leaseMonitor = new LeaseMonitor();
        this.leaseMonitorThread = new Thread(this.leaseMonitor);
        this.leaseMonitorThread.setName(this.name + "#LeaseMonitor");
        this.leaseMonitorThread.setDaemon(true);
        this.leaseMonitorThread.setUncaughtExceptionHandler((thread, throwable) -> {
            LOG.error("LeaseMonitor thread encountered an error. Thread: {}", (Object)thread.toString(), (Object)throwable);
            this.leaseMonitorThread.start();
        });
        LOG.debug("Starting {} Thread", (Object)this.leaseMonitorThread.getName());
        this.leaseMonitorThread.start();
        this.isRunning = true;
    }

    public synchronized Lease<T> acquire(T resource) throws LeaseAlreadyExistException {
        return this.acquire(resource, this.defaultTimeout);
    }

    public synchronized Lease<T> acquire(T resource, long timeout) throws LeaseAlreadyExistException {
        this.checkStatus();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquiring lease on {} for {} milliseconds", resource, (Object)timeout);
        }
        if (this.activeLeases.containsKey(resource)) {
            throw new LeaseAlreadyExistException(Lease.messageForResource(resource));
        }
        Lease<T> lease = new Lease<T>(resource, timeout);
        this.activeLeases.put(resource, lease);
        this.semaphore.release();
        return lease;
    }

    public synchronized Lease<T> acquire(T resource, long timeout, Callable<Void> callback) throws LeaseAlreadyExistException, LeaseExpiredException {
        this.checkStatus();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Acquiring lease on {} for {} milliseconds", resource, (Object)timeout);
        }
        if (this.activeLeases.containsKey(resource)) {
            throw new LeaseAlreadyExistException(Lease.messageForResource(resource));
        }
        Lease<T> lease = new Lease<T>(resource, timeout, callback);
        this.activeLeases.put(resource, lease);
        this.semaphore.release();
        return lease;
    }

    public synchronized Lease<T> acquire(T resource, Callable<Void> callback) throws LeaseAlreadyExistException, LeaseExpiredException {
        return this.acquire(resource, this.defaultTimeout, callback);
    }

    public Lease<T> get(T resource) throws LeaseNotFoundException {
        this.checkStatus();
        Lease<T> lease = this.activeLeases.get(resource);
        if (lease != null) {
            return lease;
        }
        throw new LeaseNotFoundException(Lease.messageForResource(resource));
    }

    public synchronized void release(T resource) throws LeaseNotFoundException {
        Lease<T> lease;
        this.checkStatus();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing lease on {}", resource);
        }
        if ((lease = this.activeLeases.remove(resource)) == null) {
            throw new LeaseNotFoundException(Lease.messageForResource(resource));
        }
        lease.invalidate();
    }

    public void shutdown() {
        this.checkStatus();
        LOG.debug("Shutting down LeaseManager service");
        this.leaseMonitor.disable();
        this.semaphore.release();
        this.leaseMonitorThread.interrupt();
        for (T resource : this.activeLeases.keySet()) {
            try {
                this.release(resource);
            }
            catch (LeaseNotFoundException leaseNotFoundException) {}
        }
        this.isRunning = false;
    }

    private void checkStatus() {
        if (!this.isRunning) {
            throw new LeaseManagerNotRunningException("LeaseManager not running.");
        }
    }

    private final class LeaseMonitor
    implements Runnable {
        private final ExecutorService executorService = Executors.newCachedThreadPool();
        private volatile boolean running = true;

        private LeaseMonitor() {
        }

        @Override
        public void run() {
            while (this.running) {
                LOG.debug("{}-LeaseMonitor: checking for lease expiry", (Object)LeaseManager.this.name);
                long sleepTime = Long.MAX_VALUE;
                for (Object resource : LeaseManager.this.activeLeases.keySet()) {
                    try {
                        Lease lease = LeaseManager.this.get(resource);
                        long remainingTime = lease.getRemainingTime();
                        if (remainingTime <= 0L) {
                            List<Callable<Void>> leaseCallbacks = lease.getCallbacks();
                            LeaseManager.this.release(resource);
                            this.executorService.execute(new LeaseCallbackExecutor(resource, leaseCallbacks));
                            continue;
                        }
                        sleepTime = Math.min(remainingTime, sleepTime);
                    }
                    catch (LeaseExpiredException | LeaseNotFoundException leaseException) {}
                }
                try {
                    boolean bl = LeaseManager.this.semaphore.tryAcquire(sleepTime, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    LOG.warn("Lease manager is interrupted. Shutting down...", (Throwable)e);
                    Thread.currentThread().interrupt();
                }
            }
        }

        public void disable() {
            this.running = false;
        }
    }
}

