# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

import os

from conan import ConanFile
from conan.tools.build import check_min_cppstd
from conan.tools.cmake import CMake, CMakeDeps, CMakeToolchain, cmake_layout
from conan.tools.env import VirtualBuildEnv, VirtualRunEnv
from conan.tools.files import copy, get, rmdir, replace_in_file, mkdir
from conan.tools.scm import Version

required_conan_version = ">=2.1"

class OrcRecipe(ConanFile):
    name = "orc"
    description = "ORC is a self-describing type-aware columnar file format designed for Hadoop workloads"
    license = "Apache-2.0"
    homepage = "https://orc.apache.org/"
    url = "https://github.com/conan-io/conan-center-index"
    topics = ("orc", "columnar", "file-format", "hadoop")

    package_type = "library"
    settings = "os", "arch", "compiler", "build_type"
    options = {
        "shared": [True, False],
        "fPIC": [True, False],
        "build_tools": [True, False],
        "build_avx512": [True, False],
    }
    default_options = {
        "shared": False,
        "fPIC": True,
        "build_tools": False,
        "build_avx512": True,
    }

    @property
    def _should_patch_thirdparty_toolchain(self):
        return Version(self.version) < "2.0.0"

    def export_sources(self):
        if self._should_patch_thirdparty_toolchain:
            copy(self, "ConanThirdpartyToolchain.cmake",
                 self.recipe_folder, os.path.join(self.export_sources_folder, "src", "cmake_modules"))

    def config_options(self):
        if self.settings.os == "Windows":
            del self.options.fPIC
        if self.settings.compiler == "apple-clang":
            # AVX support is not enabled by default, might need to add -mavx512f to CXXFLAGS
            del self.options.build_avx512

    def configure(self):
        if self.options.shared:
            self.options.rm_safe("fPIC")

    def layout(self):
        cmake_layout(self, src_folder="src", build_folder="build")

    def requirements(self):
        self.requires("protobuf/3.21.12")
        self.requires("lz4/1.9.4")
        self.requires("snappy/1.1.9")
        self.requires("zlib/[>=1.2.11 <2]")
        self.requires("zstd/[~1.5]")

    def validate(self):
        check_min_cppstd(self, 17)

    def build_requirements(self):
        self.tool_requires("protobuf/<host_version>")

    def source(self):
        # START
        # This block should be removed when we update upstream:
        # https://github.com/conan-io/conan-center-index/tree/master/recipes/orc/
        if not self.version in self.conan_data.get("sources", {}):
            import shutil
            top_level = os.environ.get("ORC_HOME")
            shutil.copytree(os.path.join(top_level, "c++"),
                            os.path.join(self.source_folder, "c++"))
            shutil.copytree(os.path.join(top_level, "cmake_modules"),
                            os.path.join(self.source_folder, "cmake_modules"))
            top_level_files = [
                "CMakeLists.txt",
                "LICENSE",
                "NOTICE",
            ]
            for top_level_file in top_level_files:
                shutil.copy(os.path.join(top_level, top_level_file),
                            self.source_folder)
            return
        # END
        get(self, **self.conan_data["sources"][self.version], strip_root=True)
        self._patch_sources()

    def generate(self):
        VirtualBuildEnv(self).generate()
        VirtualRunEnv(self).generate(scope="build")

        tc = CMakeToolchain(self)
        if Version(self.version) < "2.3.0":
            tc.variables["ORC_PACKAGE_KIND"] = "conan"
        tc.variables["BUILD_JAVA"] = False
        tc.variables["BUILD_CPP_TESTS"] = False
        tc.variables["BUILD_TOOLS"] = self.options.build_tools
        tc.variables["BUILD_LIBHDFSPP"] = False
        tc.variables["BUILD_POSITION_INDEPENDENT_LIB"] = bool(self.options.get_safe("fPIC", True))
        tc.variables["INSTALL_VENDORED_LIBS"] = False
        # AVX512 support is determined by ORC_USER_SIMD_LEVEL env var at runtime, defaults to off
        tc.variables["BUILD_ENABLE_AVX512"] = self.options.get_safe("build_avx512", False)
        tc.variables["STOP_BUILD_ON_WARNING"] = False
        tc.variables["CMAKE_WINDOWS_EXPORT_ALL_SYMBOLS"] = True

        protoc_path = os.path.join(self.dependencies["protobuf"].cpp_info.bindir, "protoc")
        tc.variables["PROTOBUF_EXECUTABLE"] = protoc_path.replace("\\", "/")
        tc.variables["HAS_POST_2038"] = self.settings.os != "Windows"
        tc.variables["HAS_PRE_1970"] = self.settings.os != "Windows"
        tc.generate()

        deps = CMakeDeps(self)
        deps.generate()

    def _patch_sources(self):
        if self._should_patch_thirdparty_toolchain:
            replace_in_file(self, os.path.join(self.source_folder, "CMakeLists.txt"),
                            "ThirdpartyToolchain", "ConanThirdpartyToolchain")
        # Allow shared builds
        replace_in_file(self, os.path.join(self.source_folder, "c++", "src", "CMakeLists.txt"),
                        "add_library (orc STATIC ${SOURCE_FILES})", "add_library (orc ${SOURCE_FILES})")

    def build(self):
        cmake = CMake(self)
        cmake.configure()
        cmake.build()

    def package(self):
        copy(self, "LICENSE", self.source_folder, os.path.join(self.package_folder, "licenses"))
        copy(self, "NOTICE", self.source_folder, os.path.join(self.package_folder, "licenses"))
        cmake = CMake(self)
        cmake.install()
        rmdir(self, os.path.join(self.package_folder, "share"))
        rmdir(self, os.path.join(self.package_folder, "lib", "cmake"))
        if self.settings.os == "Windows" and self.options.shared and Version(self.version) < "2.1.1":
            mkdir(self, os.path.join(self.package_folder, "bin"))
            os.rename(os.path.join(self.package_folder, "lib", "orc.dll"),
                      os.path.join(self.package_folder, "bin", "orc.dll"))

    def package_info(self):
        self.cpp_info.libs = ["orc"]
        if self.settings.os in ["Linux", "FreeBSD"]:
            self.cpp_info.system_libs = ["pthread", "m"]
