/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.stream.core.storage;

import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.util.Dictionary;
import org.apache.kylin.common.util.Pair;
import org.apache.kylin.cube.CubeInstance;
import org.apache.kylin.cube.CubeManager;
import org.apache.kylin.cube.CubeSegment;
import org.apache.kylin.measure.bitmap.BitmapMeasureType;
import org.apache.kylin.metadata.model.SegmentRange;
import org.apache.kylin.metadata.model.TblColRef;
import org.apache.kylin.shaded.com.google.common.base.Function;
import org.apache.kylin.shaded.com.google.common.collect.Lists;
import org.apache.kylin.shaded.com.google.common.collect.Maps;
import org.apache.kylin.stream.core.consumer.IConsumerProvider;
import org.apache.kylin.stream.core.consumer.StreamingConsumerChannel;
import org.apache.kylin.stream.core.dict.StreamingDictionaryClient;
import org.apache.kylin.stream.core.dict.StreamingDistributedDictionary;
import org.apache.kylin.stream.core.model.StreamingMessage;
import org.apache.kylin.stream.core.model.stats.LongLatencyInfo;
import org.apache.kylin.stream.core.model.stats.SegmentStats;
import org.apache.kylin.stream.core.model.stats.SegmentStoreStats;
import org.apache.kylin.stream.core.query.StreamingCubeDataSearcher;
import org.apache.kylin.stream.core.source.ISourcePosition;
import org.apache.kylin.stream.core.source.ISourcePositionHandler;
import org.apache.kylin.stream.core.storage.CheckPoint;
import org.apache.kylin.stream.core.storage.CheckPointStore;
import org.apache.kylin.stream.core.storage.IStreamingSegmentStore;
import org.apache.kylin.stream.core.storage.StreamingCubeSegment;
import org.apache.kylin.stream.core.storage.columnar.ColumnarSegmentStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamingSegmentManager
implements Closeable {
    private static Logger logger = LoggerFactory.getLogger(StreamingSegmentManager.class);
    private final String cubeName;
    private final CubeInstance cubeInstance;
    private AtomicBoolean closed = new AtomicBoolean(false);
    private final long cubeWindow;
    public final long cubeDuration;
    private final long maxCubeDuration;
    private final int checkPointIntervals;
    private final int maxImmutableSegments;
    private final String baseStorePath;
    private final File cubeDataFolder;
    private final CheckPointStore checkPointStore;
    private final IConsumerProvider consumerProvider;
    private final Map<Long, StreamingCubeSegment> activeSegments = new ConcurrentSkipListMap<Long, StreamingCubeSegment>();
    private final Map<Long, StreamingCubeSegment> immutableSegments = new ConcurrentSkipListMap<Long, StreamingCubeSegment>();
    private Map<Long, ISourcePosition> segmentSourceStartPositions = Maps.newConcurrentMap();
    private final ISourcePositionHandler sourcePositionHandler;
    private final ISourcePosition consumePosition;
    private static final long TIME_ZONE_OFFSET = TimeZone.getTimeZone(KylinConfig.getInstanceFromEnv().getTimeZone()).getRawOffset();
    private volatile LongLatencyInfo longLatencyInfo;
    private volatile long nextCheckPoint = 0L;
    private volatile long lastCheckPointCount = 0L;
    private Pair<Long, Long> latestRemoteSegmentRange;
    private AtomicLong ingestCount = new AtomicLong();
    private AtomicLong dropCounts = new AtomicLong();
    private volatile long latestEventTime = 0L;
    private volatile long latestEventIngestTime = 0L;
    private Map<TblColRef, Dictionary<String>> dictionaryMap = new HashMap<TblColRef, Dictionary<String>>();
    private StreamingDictionaryClient streamingDictionaryClient;

    public StreamingSegmentManager(String baseStorePath, CubeInstance cubeInstance, ISourcePositionHandler sourcePosHandler, IConsumerProvider consumerProvider) {
        this.baseStorePath = baseStorePath;
        this.cubeName = cubeInstance.getName();
        this.cubeInstance = cubeInstance;
        this.cubeWindow = (long)cubeInstance.getConfig().getStreamingCubeWindowInSecs() * 1000L;
        this.cubeDuration = (long)cubeInstance.getConfig().getStreamingCubeDurationInSecs() * 1000L;
        this.maxCubeDuration = (long)cubeInstance.getConfig().getStreamingCubeMaxDurationInSecs() * 1000L;
        this.checkPointIntervals = cubeInstance.getConfig().getStreamingCheckPointIntervalsInSecs() * 1000;
        this.maxImmutableSegments = cubeInstance.getConfig().getStreamingMaxImmutableSegments();
        this.consumerProvider = consumerProvider;
        this.cubeDataFolder = new File(baseStorePath, this.cubeName);
        this.sourcePositionHandler = sourcePosHandler;
        this.consumePosition = this.sourcePositionHandler.createEmptyPosition();
        if (!this.cubeDataFolder.exists()) {
            this.cubeDataFolder.mkdirs();
        }
        this.longLatencyInfo = new LongLatencyInfo();
        this.checkPointStore = new CheckPointStore(this.cubeName, this.cubeDataFolder, cubeInstance.getConfig().getStreamingCheckPointFileMaxNum());
        CubeInstance cube = CubeManager.getInstance(KylinConfig.getInstanceFromEnv()).getCube(this.cubeName);
        List bitmapMeasureList = cube.getDescriptor().getMeasures().stream().filter(measureDesc -> measureDesc.getFunction().getMeasureType() instanceof BitmapMeasureType).collect(Collectors.toList());
        if (!bitmapMeasureList.isEmpty()) {
            List<String> realtimeDictColumn = bitmapMeasureList.stream().map(measureDesc -> measureDesc.getFunction().getParameter().getColRef().getIdentity()).collect(Collectors.toList());
            String str = String.join((CharSequence)", ", realtimeDictColumn);
            logger.info("Find these columns {} need to be encoded realtime.", (Object)str);
            List<TblColRef> tblColRefs = bitmapMeasureList.stream().map(measureDesc -> measureDesc.getFunction().getParameter().getColRef()).collect(Collectors.toList());
            this.streamingDictionaryClient = new StreamingDictionaryClient(this.cubeName, realtimeDictColumn.toArray(new String[0]));
            tblColRefs.forEach(col -> this.dictionaryMap.put((TblColRef)col, new StreamingDistributedDictionary(col.getIdentity(), this.streamingDictionaryClient)));
        }
    }

    public void addEvent(StreamingMessage event) {
        long currentTime;
        long eventTime = event.getTimestamp();
        long segmentStart = this.truncateTime(eventTime, this.cubeWindow);
        long segmentEnd = segmentStart + this.cubeWindow;
        StreamingCubeSegment segment = this.activeSegments.get(segmentStart);
        if (segment == null) {
            List<StreamingCubeSegment> toBeImmutableSegments = this.findSegmentsToBeImmutable();
            if (!toBeImmutableSegments.isEmpty()) {
                this.convertImmutable(toBeImmutableSegments);
            }
            if (this.isLongLatencyEvent(segmentStart)) {
                this.longLatencyInfo.incLongLatencyEvent(CubeSegment.makeSegmentName(new SegmentRange.TSRange(segmentStart, segmentEnd), null, this.cubeInstance.getModel()));
                this.ingestCount.incrementAndGet();
                this.consumePosition.update(event.getSourcePosition());
                return;
            }
            if (segment == null) {
                segment = this.createSegment(segmentStart, segmentEnd);
                if (logger.isInfoEnabled()) {
                    logger.info("Create new segment:{}", (Object)segment);
                }
                this.activeSegments.put(segmentStart, segment);
                if (this.activeSegments.size() > 12) {
                    logger.warn("Two many active segments, segments size = " + this.activeSegments.keySet());
                }
                if (this.immutableSegments.size() > this.maxImmutableSegments) {
                    logger.info("Two many immutable segments, segments size:{}, pause the cube consume", (Object)this.immutableSegments.size());
                    this.pauseCubeConsumer();
                }
            }
        }
        this.latestEventIngestTime = currentTime = System.currentTimeMillis();
        segment.addEvent(event);
        segment.setLastUpdateTime(currentTime);
        if (eventTime > this.latestEventTime) {
            this.latestEventTime = eventTime;
        }
        this.ingestCount.incrementAndGet();
        ISourcePosition segmentSourceStartPos = this.segmentSourceStartPositions.get(segment.getDateRangeStart());
        if (segmentSourceStartPos == null) {
            segmentSourceStartPos = this.sourcePositionHandler.createEmptyPosition();
            this.segmentSourceStartPositions.put(segment.getDateRangeStart(), segmentSourceStartPos);
        }
        segmentSourceStartPos.updateWhenPartitionNotExist(event.getSourcePosition());
        this.consumePosition.update(event.getSourcePosition());
        if (this.nextCheckPoint == 0L) {
            this.nextCheckPoint = currentTime + (long)this.checkPointIntervals;
        }
        if (currentTime > this.nextCheckPoint) {
            this.checkpoint();
            this.nextCheckPoint = System.currentTimeMillis() + (long)this.checkPointIntervals;
        }
    }

    public void restoreConsumerStates(ISourcePosition sourcePosition) {
        logger.info("restore consume state:{}", (Object)sourcePosition);
        this.consumePosition.copy(sourcePosition);
    }

    public void restoreSegmentsFromLocal() {
        logger.info("restore segments from local files for cube:{}", (Object)this.cubeName);
        List<File> segmentFolders = this.getSegmentFolders(this.cubeDataFolder);
        Map<Long, String> checkpointStoreStats = null;
        CheckPoint checkpoint = this.checkPointStore.getLatestCheckPoint();
        if (checkpoint != null) {
            checkpointStoreStats = checkpoint.getPersistedIndexes();
            logger.info("checkpoint found for the cube:{}, checkpoint:{}", (Object)this.cubeName, (Object)checkpoint);
            this.ingestCount.set(checkpoint.getTotalCount());
            if (checkpoint.getLongLatencyInfo() != null) {
                this.longLatencyInfo = checkpoint.getLongLatencyInfo();
            }
        }
        CubeSegment latestRemoteSegment = this.cubeInstance.getLatestReadySegment();
        if (checkpointStoreStats == null || checkpointStoreStats.isEmpty()) {
            logger.warn("no checkpoint for cube:{} store state, remove all local segments folders", (Object)this.cubeName);
            for (File segmentFolder : segmentFolders) {
                this.removeSegmentFolder(segmentFolder);
            }
        } else {
            this.restoreSegmentsFromCP(segmentFolders, checkpointStoreStats, checkpoint.getSegmentSourceStartPosition(), latestRemoteSegment);
        }
    }

    private void restoreSegmentsFromCP(List<File> segmentFolders, Map<Long, String> checkpointStoreStats, Map<Long, String> segmentSourceStartPositions, CubeSegment latestRemoteSegment) {
        if (segmentSourceStartPositions != null) {
            this.segmentSourceStartPositions.putAll(Maps.transformValues(segmentSourceStartPositions, new Function<String, ISourcePosition>(){

                @Override
                @Nullable
                public ISourcePosition apply(@Nullable String input) {
                    return StreamingSegmentManager.this.sourcePositionHandler.parsePosition(input);
                }
            }));
        }
        for (File segmentFolder : segmentFolders) {
            try {
                IStreamingSegmentStore segmentStore = this.getSegmentStore(segmentFolder.getName());
                StreamingCubeSegment segment = StreamingCubeSegment.parseSegment(this.cubeInstance, segmentFolder, segmentStore);
                if (latestRemoteSegment != null && segment.getDateRangeEnd() <= (Long)latestRemoteSegment.getTSRange().end.v) {
                    logger.info("remove segment:{} because it is late than remote segment", (Object)segment);
                    this.removeSegmentFolder(segmentFolder);
                    continue;
                }
                if (segment.isImmutable()) {
                    this.immutableSegments.put(segment.getDateRangeStart(), segment);
                    continue;
                }
                String segmentCheckpoint = checkpointStoreStats.get(segment.getDateRangeStart());
                if (segmentCheckpoint == null) {
                    this.removeSegmentFolder(segmentFolder);
                } else {
                    segmentStore.restoreFromCheckpoint(segmentCheckpoint);
                }
                this.activeSegments.put(segment.getDateRangeStart(), segment);
            }
            catch (Exception e) {
                logger.error("fail to restore segment from file:" + segmentFolder.getName(), e);
            }
        }
    }

    private void removeSegmentFolder(File segmentFolder) {
        logger.info("Remove segment folder: {} for the cube:{} ", (Object)segmentFolder, (Object)this.cubeName);
        try {
            FileUtils.deleteDirectory((File)segmentFolder);
        }
        catch (IOException e) {
            logger.error("error happens when purge segment", e);
        }
    }

    private List<File> getSegmentFolders(File cubeDataFolder) {
        File[] segmentFolders = cubeDataFolder.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return file.isDirectory() && !".cp".equals(file.getName());
            }
        });
        if (segmentFolders.length == 0) {
            return Lists.newArrayList();
        }
        return Arrays.asList(segmentFolders);
    }

    public String getCheckPointSourcePosition() {
        CheckPoint latestCheckPoint = this.checkPointStore.getLatestCheckPoint();
        if (latestCheckPoint == null) {
            return null;
        }
        return latestCheckPoint.getSourceConsumePosition();
    }

    private void pauseCubeConsumer() {
        if (this.consumerProvider == null) {
            return;
        }
        StreamingConsumerChannel consumer = this.consumerProvider.getConsumer(this.cubeName);
        if (consumer != null) {
            consumer.pause(false);
        }
    }

    private boolean isLongLatencyEvent(long segmentStart) {
        if (this.immutableSegments.containsKey(segmentStart) || this.latestRemoteSegmentRange != null && segmentStart <= this.latestRemoteSegmentRange.getFirst()) {
            return true;
        }
        if (this.immutableSegments.isEmpty()) {
            return false;
        }
        long maxImmutableSegmentStart = Collections.max(this.immutableSegments.keySet());
        return segmentStart <= maxImmutableSegmentStart;
    }

    private List<StreamingCubeSegment> findSegmentsToBeImmutable() {
        ArrayList<StreamingCubeSegment> toBeImmutableSegments = Lists.newArrayList();
        if (this.activeSegments.size() <= 0) {
            return toBeImmutableSegments;
        }
        for (Map.Entry<Long, StreamingCubeSegment> item : this.activeSegments.entrySet()) {
            StreamingCubeSegment segment = item.getValue();
            if (segment == null) continue;
            if (System.currentTimeMillis() - segment.getLastUpdateTime() > this.cubeDuration) {
                toBeImmutableSegments.add(segment);
                continue;
            }
            if (System.currentTimeMillis() - segment.getCreateTime() > this.maxCubeDuration) {
                logger.warn("OldestSegment:{} pass maxCubeDuration", (Object)segment);
                toBeImmutableSegments.add(segment);
                continue;
            }
            return toBeImmutableSegments;
        }
        return toBeImmutableSegments;
    }

    private StreamingCubeSegment createSegment(long segmentStart, long segmentEnd) {
        String segmentName = CubeSegment.makeSegmentName(new SegmentRange.TSRange(segmentStart, segmentEnd), null, this.cubeInstance.getModel());
        IStreamingSegmentStore store = this.getSegmentStore(segmentName);
        StreamingCubeSegment newSegment = new StreamingCubeSegment(this.cubeInstance, store, segmentStart, segmentEnd);
        return newSegment;
    }

    private IStreamingSegmentStore getSegmentStore(String segmentName) {
        IStreamingSegmentStore segmentStore;
        String storeClassName = this.cubeInstance.getConfig().getStreamingStoreClass();
        try {
            Class<?> clazz = Class.forName(storeClassName);
            Constructor<?> constructor = clazz.getConstructor(String.class, CubeInstance.class, String.class);
            segmentStore = (IStreamingSegmentStore)constructor.newInstance(this.baseStorePath, this.cubeInstance, segmentName);
            segmentStore.addExternalDict(this.dictionaryMap);
        }
        catch (Exception e) {
            logger.warn("Fail to construct an instance for " + storeClassName + ". Will use the default store: ColumnarSegmentStore");
            segmentStore = new ColumnarSegmentStore(this.baseStorePath, this.cubeInstance, segmentName);
        }
        segmentStore.init();
        return segmentStore;
    }

    public Collection<StreamingCubeSegment> getAllSegments() {
        ArrayList<StreamingCubeSegment> result = Lists.newArrayList();
        result.addAll(this.activeSegments.values());
        result.addAll(this.immutableSegments.values());
        return result;
    }

    private Collection<StreamingCubeSegment> getSegmentsLateThan(long segmentStart) {
        ArrayList<StreamingCubeSegment> result = Lists.newArrayList();
        for (StreamingCubeSegment segment : this.immutableSegments.values()) {
            if (segment.getDateRangeStart() <= segmentStart) continue;
            result.add(segment);
        }
        for (StreamingCubeSegment segment : this.activeSegments.values()) {
            if (segment.getDateRangeStart() <= segmentStart) continue;
            result.add(segment);
        }
        return result;
    }

    public StreamingCubeSegment getSegmentByName(String segmentName) {
        Collection<StreamingCubeSegment> allSegments = this.getAllSegments();
        for (StreamingCubeSegment segment : allSegments) {
            if (!segmentName.equals(segment.getSegmentName())) continue;
            return segment;
        }
        return null;
    }

    public Collection<StreamingCubeSegment> getActiveSegments() {
        return this.activeSegments.values();
    }

    public Collection<StreamingCubeSegment> getRequireRemotePersistSegments() {
        Collection<StreamingCubeSegment> allImmutableSegments = this.immutableSegments.values();
        ArrayList<StreamingCubeSegment> result = Lists.newArrayList();
        for (StreamingCubeSegment cubeSegment : allImmutableSegments) {
            if (cubeSegment.isPersistToRemote()) continue;
            result.add(cubeSegment);
        }
        if (result.size() > 1) {
            Collections.sort(result);
        }
        return result;
    }

    public ISourcePosition getSmallestSourcePosition(StreamingCubeSegment currSegment) {
        ArrayList<ISourcePosition> allSegmentStartPositions = Lists.newArrayList();
        for (Map.Entry<Long, ISourcePosition> segmentSourceStartPosEntry : this.segmentSourceStartPositions.entrySet()) {
            if (currSegment.getDateRangeStart() >= segmentSourceStartPosEntry.getKey()) continue;
            allSegmentStartPositions.add(segmentSourceStartPosEntry.getValue());
        }
        return this.sourcePositionHandler.mergePositions(allSegmentStartPositions, ISourcePositionHandler.MergeStrategy.KEEP_SMALL);
    }

    public void purgeSegment(String segmentName) {
        Pair<Long, Long> segmentRange = CubeSegment.parseSegmentName(segmentName);
        StreamingCubeSegment segment = this.activeSegments.remove(segmentRange.getFirst());
        if (segment == null) {
            segment = this.immutableSegments.remove(segmentRange.getFirst());
        }
        this.segmentSourceStartPositions.remove(segmentRange.getFirst());
        if (segment != null) {
            segment.purge();
        }
    }

    public void makeAllSegmentsImmutable() {
        Collection<StreamingCubeSegment> activeSegments = this.getActiveSegments();
        if (activeSegments != null && !activeSegments.isEmpty()) {
            ArrayList<StreamingCubeSegment> segments = Lists.newArrayList(activeSegments);
            for (StreamingCubeSegment segment : segments) {
                this.convertImmutable(segment);
            }
        }
    }

    public void makeSegmentImmutable(String segmentName) {
        StreamingCubeSegment segment = this.getSegmentByName(segmentName);
        this.convertImmutable(segment);
    }

    public List<String> remoteSegmentBuildComplete(String segmentName) {
        Pair<Long, Long> segmentRange = CubeSegment.parseSegmentName(segmentName);
        ArrayList<Long> immutableSegmentStarts = Lists.newArrayList(this.immutableSegments.keySet());
        ArrayList<String> removedSegments = Lists.newArrayList();
        for (Long immutableSegmentStart : immutableSegmentStarts) {
            if (immutableSegmentStart > segmentRange.getFirst()) continue;
            StreamingCubeSegment segment = this.immutableSegments.get(immutableSegmentStart);
            this.immutableSegments.remove(immutableSegmentStart);
            segment.purge();
            removedSegments.add(segment.getSegmentName());
        }
        ArrayList<Long> activeSegmentStarts = Lists.newArrayList(this.activeSegments.keySet());
        for (Long activeSegmentStart : activeSegmentStarts) {
            if (activeSegmentStart > segmentRange.getFirst()) continue;
            StreamingCubeSegment segment = this.activeSegments.get(activeSegmentStart);
            this.activeSegments.remove(activeSegmentStart);
            segment.purge();
            removedSegments.add(segment.getSegmentName());
        }
        this.latestRemoteSegmentRange = segmentRange;
        logger.info("cube:{} segments:{} has been removed", (Object)this.cubeName, (Object)removedSegments);
        return removedSegments;
    }

    public void purgeAllSegments() {
        try {
            FileUtils.deleteDirectory((File)this.cubeDataFolder);
        }
        catch (IOException e) {
            logger.error("error happens when purge cube", e);
        }
    }

    public String getConsumePositionStr() {
        return this.sourcePositionHandler.serializePosition(this.consumePosition);
    }

    public ISourcePosition getConsumePosition() {
        return this.consumePosition;
    }

    public long getIngestCount() {
        return this.ingestCount.get();
    }

    public long getLatestEventTime() {
        return this.latestEventTime;
    }

    public long getLatestEventIngestTime() {
        return this.latestEventIngestTime;
    }

    public LongLatencyInfo getLongLatencyInfo() {
        return this.longLatencyInfo;
    }

    public CubeInstance getCubeInstance() {
        return this.cubeInstance;
    }

    public StreamingCubeDataSearcher getSearcher() {
        return new StreamingCubeDataSearcher(this);
    }

    public Map<String, SegmentStats> getSegmentStats() {
        LinkedHashMap<String, SegmentStats> result = Maps.newLinkedHashMap();
        Collection<StreamingCubeSegment> allSegments = this.getAllSegments();
        for (StreamingCubeSegment segment : allSegments) {
            SegmentStats segmentStats = new SegmentStats();
            segmentStats.setSegmentState(segment.getState().name());
            segmentStats.setSegmentCreateTime(StreamingSegmentManager.resetTimestampByTimeZone(segment.getCreateTime()));
            segmentStats.setSegmentLastUpdateTime(StreamingSegmentManager.resetTimestampByTimeZone(segment.getLastUpdateTime()));
            segmentStats.setLatestEventTime(StreamingSegmentManager.resetTimestampByTimeZone(segment.getLatestEventTimeStamp()));
            segmentStats.setLatestEventLatency(segment.getLatestEventLatecy());
            SegmentStoreStats storeStats = segment.getSegmentStore().getStoreStats();
            segmentStats.setStoreStats(storeStats);
            result.put(segment.getSegmentName(), segmentStats);
        }
        return result;
    }

    @Override
    public void close() {
        if (this.closed.get()) {
            logger.debug("Already close it, skip.");
            return;
        }
        logger.warn("Closing Streaming Cube store, cubeName={}", (Object)this.cubeName);
        this.checkpoint();
        logger.warn("{} ingested {} , dropped {}, long latency {}", this.cubeName, this.ingestCount.get(), this.dropCounts.get(), this.longLatencyInfo.getTotalLongLatencyEventCnt());
        for (StreamingCubeSegment cubesegment : this.getAllSegments()) {
            try {
                cubesegment.close();
            }
            catch (IOException e) {
                logger.error("fail to close cube segment, segment :" + cubesegment.getSegmentName(), e);
            }
        }
        if (this.streamingDictionaryClient != null) {
            this.streamingDictionaryClient.close();
        }
        this.closed.set(true);
    }

    public synchronized void checkpoint() {
        HashMap<Long, String> persistedIndexes = Maps.newHashMap();
        for (Map.Entry<Long, StreamingCubeSegment> segmentEntry : this.activeSegments.entrySet()) {
            IStreamingSegmentStore segmentStore = segmentEntry.getValue().getSegmentStore();
            String largestFragmentID = segmentStore.checkpoint().toString();
            persistedIndexes.put(segmentEntry.getKey(), largestFragmentID);
        }
        this.persistCheckPoint(this.consumePosition, persistedIndexes);
    }

    private void persistCheckPoint(ISourcePosition consumePosition, Map<Long, String> persistedIndexes) {
        CheckPoint cp = new CheckPoint();
        cp.setCheckPointTime(System.currentTimeMillis());
        cp.setSourceConsumePosition(this.sourcePositionHandler.serializePosition(consumePosition));
        cp.setPersistedIndexes(persistedIndexes);
        cp.setTotalCount(this.ingestCount.get());
        cp.setCheckPointCount(this.ingestCount.get() - this.lastCheckPointCount);
        this.lastCheckPointCount = this.ingestCount.get();
        cp.setLongLatencyInfo(this.longLatencyInfo.truncate(100));
        Collection<StreamingCubeSegment> allSegments = this.getAllSegments();
        HashMap<Long, String> segmentSourceStartPosStr = Maps.newHashMap();
        for (StreamingCubeSegment segment : allSegments) {
            segmentSourceStartPosStr.put(segment.getDateRangeStart(), this.sourcePositionHandler.serializePosition(this.segmentSourceStartPositions.get(segment.getDateRangeStart())));
        }
        cp.setSegmentSourceStartPosition(segmentSourceStartPosStr);
        if (logger.isInfoEnabled()) {
            logger.info("Print check point for cube {}", (Object)(this.cubeName + " ," + cp.toString()));
        }
        this.checkPointStore.saveCheckPoint(cp);
    }

    private void convertImmutable(List<StreamingCubeSegment> oldestSegments) {
        for (StreamingCubeSegment segment : oldestSegments) {
            this.convertImmutable(segment);
        }
    }

    private void convertImmutable(StreamingCubeSegment segment) {
        this.activeSegments.remove(segment.getDateRangeStart());
        segment.immutable();
        this.immutableSegments.put(segment.getDateRangeStart(), segment);
        if (logger.isInfoEnabled()) {
            logger.info("Convert active segment to immutable: {}", (Object)segment);
        }
    }

    private long truncateTime(long timestamp, long windowLength) {
        return timestamp / windowLength * windowLength;
    }

    public static long resetTimestampByTimeZone(long ts) {
        return ts + TIME_ZONE_OFFSET;
    }
}

