/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.job.lock.zookeeper;

import java.io.Closeable;
import java.nio.charset.StandardCharsets;
import java.util.Random;
import java.util.concurrent.Executor;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.ACLBackgroundPathAndBytesable;
import org.apache.curator.framework.imps.CuratorFrameworkState;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheListener;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.lock.DistributedLock;
import org.apache.kylin.common.lock.DistributedLockFactory;
import org.apache.kylin.common.util.ZKUtil;
import org.apache.kylin.job.lock.JobLock;
import org.apache.kylin.job.lock.zookeeper.exception.ZkAcquireLockException;
import org.apache.kylin.job.lock.zookeeper.exception.ZkPeekLockException;
import org.apache.kylin.job.lock.zookeeper.exception.ZkPeekLockInterruptException;
import org.apache.kylin.job.lock.zookeeper.exception.ZkReleaseLockException;
import org.apache.kylin.job.lock.zookeeper.exception.ZkReleaseLockInterruptException;
import org.apache.kylin.job.util.ThrowableUtils;
import org.apache.kylin.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZookeeperDistributedLock
implements DistributedLock,
JobLock {
    private static Logger logger = LoggerFactory.getLogger(ZookeeperDistributedLock.class);
    private static final Random random = new Random();
    private CuratorFramework curator;
    final String client;
    final byte[] clientBytes;

    private ZookeeperDistributedLock(CuratorFramework curator, String client) {
        if (client == null) {
            throw new NullPointerException("client must not be null");
        }
        this.curator = curator;
        this.client = client;
        this.clientBytes = client.getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public String getClient() {
        return this.client;
    }

    @Override
    public boolean lock(String lockPath) {
        logger.debug("{} trying to lock {}", (Object)this.client, (Object)lockPath);
        if (this.curator.getState() != CuratorFrameworkState.STARTED) {
            this.curator = ZKUtil.getZookeeperClient(KylinConfig.getInstanceFromEnv());
        }
        this.lockInternal(lockPath);
        try {
            String lockOwner = this.peekLock(lockPath);
            if (this.client.equals(lockOwner)) {
                logger.info("{} acquired lock at {}", (Object)this.client, (Object)lockPath);
                return true;
            }
            logger.debug("{} failed to acquire lock at {}, which is held by {}", this.client, lockPath, lockOwner);
            return false;
        }
        catch (ZkPeekLockInterruptException zpie) {
            logger.error("{} peek owner of lock interrupt while acquire lock at {}, check to release lock", (Object)this.client, (Object)lockPath);
            String lockOwner = this.peekLock(lockPath);
            try {
                this.unlockInternal(lockOwner, lockPath);
            }
            catch (Exception anyEx) {
                logger.warn("Exception caught to release lock when lock operation has been interrupted.", anyEx);
            }
            throw zpie;
        }
    }

    @Override
    public boolean globalPermanentLock(String lockPath) {
        logger.debug("{} trying to lock {}", (Object)this.client, (Object)lockPath);
        if (this.curator.getState() != CuratorFrameworkState.STARTED) {
            this.curator = ZKUtil.getZookeeperClient(KylinConfig.getInstanceFromEnv());
        }
        try {
            ((ACLBackgroundPathAndBytesable)this.curator.create().creatingParentsIfNeeded().withMode(CreateMode.PERSISTENT)).forPath(lockPath, this.clientBytes);
        }
        catch (KeeperException.NodeExistsException ex) {
            logger.debug("{} check {} is already locked", (Object)this.client, (Object)lockPath);
        }
        catch (Exception ex) {
            throw new IllegalStateException("Error while " + this.client + " trying to lock " + lockPath, ex);
        }
        String lockOwner = this.peekLock(lockPath);
        if (this.client.equals(lockOwner)) {
            logger.info("{} acquired lock at {}", (Object)this.client, (Object)lockPath);
            return true;
        }
        logger.debug("{} failed to acquire lock at {}, which is held by {}", this.client, lockPath, lockOwner);
        return false;
    }

    private void lockInternal(String lockPath) {
        try {
            ((ACLBackgroundPathAndBytesable)this.curator.create().creatingParentsIfNeeded().withMode(CreateMode.EPHEMERAL)).forPath(lockPath, this.clientBytes);
        }
        catch (KeeperException.NodeExistsException ex) {
            logger.debug("{} see {} is already locked", (Object)this.client, (Object)lockPath);
        }
        catch (Exception ex) {
            throw new ZkAcquireLockException("Error occurs while " + this.client + " trying to lock " + lockPath, ex);
        }
    }

    @Override
    public boolean lock(String lockPath, long timeout) {
        if (this.lock(lockPath)) {
            return true;
        }
        if (timeout <= 0L) {
            timeout = Long.MAX_VALUE;
        }
        logger.debug("{} will wait for lock path {}", (Object)this.client, (Object)lockPath);
        long waitStart = System.currentTimeMillis();
        long sleep = 10000L;
        while (System.currentTimeMillis() - waitStart <= timeout) {
            try {
                Thread.sleep((long)(1000.0 + (double)sleep * random.nextDouble()));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
            if (!this.lock(lockPath)) continue;
            logger.debug("{} waited {} ms for lock path {}", this.client, System.currentTimeMillis() - waitStart, lockPath);
            return true;
        }
        return false;
    }

    @Override
    public String peekLock(String lockPath) {
        try {
            return this.peekLockInternal(lockPath);
        }
        catch (Exception ex) {
            if (ThrowableUtils.isInterruptedException(ex)) {
                throw new ZkPeekLockInterruptException("Peeking owner of lock was interrupted at" + lockPath, ex);
            }
            throw new ZkPeekLockException("Error while peeking at " + lockPath, ex);
        }
    }

    private String peekLockInternal(String lockPath) throws Exception {
        try {
            byte[] bytes = (byte[])this.curator.getData().forPath(lockPath);
            return new String(bytes, StandardCharsets.UTF_8);
        }
        catch (KeeperException.NoNodeException ex) {
            return null;
        }
    }

    @Override
    public boolean isLocked(String lockPath) {
        return this.peekLock(lockPath) != null;
    }

    @Override
    public boolean isLockedByMe(String lockPath) {
        return this.client.equals(this.peekLock(lockPath));
    }

    @Override
    public void unlock(String lockPath) {
        String owner;
        logger.debug("{} trying to unlock {}", (Object)this.client, (Object)lockPath);
        ZkPeekLockInterruptException peekLockInterruptException = null;
        try {
            owner = this.peekLock(lockPath);
        }
        catch (ZkPeekLockInterruptException zie) {
            owner = this.peekLock(lockPath);
            peekLockInterruptException = zie;
        }
        catch (ZkPeekLockException ze) {
            logger.error("{} failed to peekLock when unlock at {}", this.client, lockPath, ze);
            throw ze;
        }
        ZkReleaseLockInterruptException unlockInterruptException = null;
        try {
            this.unlockInternal(owner, lockPath);
        }
        catch (ZkReleaseLockInterruptException zlie) {
            this.unlockInternal(owner, lockPath);
            unlockInterruptException = zlie;
        }
        catch (Exception ex) {
            throw new ZkReleaseLockException("Error while " + this.client + " trying to unlock " + lockPath, ex);
        }
        if (peekLockInterruptException != null) {
            throw peekLockInterruptException;
        }
        if (unlockInterruptException != null) {
            throw unlockInterruptException;
        }
    }

    private void unlockInternal(String owner, String lockPath) {
        if (owner == null) {
            throw new IllegalStateException(this.client + " cannot unlock path " + lockPath + " which is not locked currently");
        }
        if (!this.client.equals(owner)) {
            throw new IllegalStateException(this.client + " cannot unlock path " + lockPath + " which is locked by " + owner);
        }
        this.purgeLockInternal(lockPath);
        logger.info("{} released lock at {}", (Object)this.client, (Object)lockPath);
    }

    @Override
    public void purgeLocks(String lockPathRoot) {
        try {
            this.purgeLockInternal(lockPathRoot);
            logger.info("{} purged all locks under {}", (Object)this.client, (Object)lockPathRoot);
        }
        catch (ZkReleaseLockException zpe) {
            throw zpe;
        }
        catch (ZkReleaseLockInterruptException zpie) {
            this.purgeLockInternal(lockPathRoot);
            throw zpie;
        }
    }

    @VisibleForTesting
    void purgeLockInternal(String lockPath) {
        try {
            this.curator.delete().guaranteed().deletingChildrenIfNeeded().forPath(lockPath);
        }
        catch (KeeperException.NoNodeException ex) {
            logger.warn("No node found when purge lock in Lock path: {}", (Object)lockPath, (Object)ex);
        }
        catch (Exception e) {
            if (ThrowableUtils.isInterruptedException(e)) {
                throw new ZkReleaseLockInterruptException("Purge lock was interrupted at " + lockPath, e);
            }
            throw new ZkReleaseLockException("Error while " + this.client + " trying to purge " + lockPath, e);
        }
    }

    @Override
    public Closeable watchLocks(String lockPathRoot, Executor executor, final DistributedLock.Watcher watcher) {
        PathChildrenCache cache = new PathChildrenCache(this.curator, lockPathRoot, true);
        try {
            cache.start();
            cache.getListenable().addListener((Object)new PathChildrenCacheListener(){

                public void childEvent(CuratorFramework client, PathChildrenCacheEvent event) throws Exception {
                    switch (event.getType()) {
                        case CHILD_ADDED: {
                            watcher.onLock(event.getData().getPath(), new String(event.getData().getData(), StandardCharsets.UTF_8));
                            break;
                        }
                        case CHILD_REMOVED: {
                            watcher.onUnlock(event.getData().getPath(), new String(event.getData().getData(), StandardCharsets.UTF_8));
                            break;
                        }
                    }
                }
            }, executor);
        }
        catch (Exception ex) {
            logger.error("Error to watch lock path " + lockPathRoot, ex);
        }
        return cache;
    }

    @Override
    public boolean lockJobEngine() {
        String path = this.jobEngineLockPath();
        return this.lock(path, 3000L);
    }

    @Override
    public void unlockJobEngine() {
        this.unlock(this.jobEngineLockPath());
    }

    private String jobEngineLockPath() {
        return "/job_engine/global_job_engine_lock";
    }

    public static class Factory
    extends DistributedLockFactory {
        final CuratorFramework curator;

        public Factory() {
            this(KylinConfig.getInstanceFromEnv());
        }

        public Factory(KylinConfig config) {
            this(ZKUtil.getZookeeperClient(config));
        }

        public Factory(CuratorFramework curator) {
            this.curator = curator;
        }

        @Override
        public DistributedLock lockForClient(String client) {
            return new ZookeeperDistributedLock(this.curator, client);
        }
    }
}

