/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.xml;

import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;
import static org.junit.jupiter.api.Assertions.*;

import java.util.function.*;

import javax.xml.stream.*;
import javax.xml.stream.events.*;
import javax.xml.stream.util.*;

import org.apache.juneau.*;
import org.apache.juneau.commons.reflect.*;
import org.apache.juneau.svl.*;
import org.apache.juneau.xml.annotation.*;
import org.junit.jupiter.api.*;

/**
 * Tests the @XmlConfig annotation.
 */
class XmlConfigAnnotationTest extends TestBase {

	private static void check(String expected, Object o) {
		assertEquals(expected, TO_STRING.apply(o));
	}

	private static final Function<Object,String> TO_STRING = t -> {
		if (t == null)
			return null;
		if (isArray(t))
			return XmlConfigAnnotationTest.TO_STRING.apply(toList(t, Object.class));
		if (t instanceof AA)
			return "AA";
		if (t instanceof AB)
			return "AB";
		if (t instanceof AC)
			return "AC";
		return t.toString();
	};

	static VarResolverSession sr = VarResolver.create().vars(XVar.class).build().createSession();

	//-----------------------------------------------------------------------------------------------------------------
	// Basic tests
	//-----------------------------------------------------------------------------------------------------------------

	public static class AA extends XmlEventAllocator {
		@Override
		public XMLEventAllocator newInstance() {
			return null;
		}
		@Override
		public XMLEvent allocate(XMLStreamReader reader) throws XMLStreamException {
			return null;
		}
		@Override
		public void allocate(XMLStreamReader reader, XMLEventConsumer consumer) throws XMLStreamException {/* no-op */}
	}
	public static class AB extends XmlReporter {
		@Override
		public void report(String message, String errorType, Object relatedInformation, Location location) throws XMLStreamException {/* no-op */}
	}
	public static class AC extends XmlResolver {
		@Override
		public Object resolveEntity(String publicID, String systemID, String baseURI, String namespace) throws XMLStreamException {
			return null;
		}
	}

	@XmlConfig(
		addBeanTypes="$X{true}",
		addNamespaceUrisToRoot="$X{true}",
		disableAutoDetectNamespaces="$X{true}",
		defaultNamespace="$X{foo}",
		enableNamespaces="$X{true}",
		eventAllocator=AA.class,
		namespaces="$X{foo}",
		preserveRootElement="$X{true}",
		reporter=AB.class,
		resolver=AC.class,
		validating="$X{true}"
	)
	static class A {}
	static ClassInfo a = ClassInfo.of(A.class);

	@Test void basicSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(a.getAnnotations()));
		var x = XmlSerializer.create().apply(al).build().getSession();
		check("true", x.isAddBeanTypes());
		check("true", x.isAddNamespaceUrisToRoot());
		check("false", x.isAutoDetectNamespaces());
		check("foo:null", x.getDefaultNamespace());
		check("true", x.isEnableNamespaces());
		check("[foo:null]", x.getNamespaces());
	}

	@Test void basicParser() {
		var al = AnnotationWorkList.of(sr, rstream(a.getAnnotations()));
		var x = XmlParser.create().apply(al).build().getSession();
		check("AA", x.getEventAllocator());
		check("true", x.isPreserveRootElement());
		check("AB", x.getReporter());
		check("AC", x.getResolver());
		check("true", x.isValidating());
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Annotation with no values.
	//-----------------------------------------------------------------------------------------------------------------

	@XmlConfig()
	static class B {}
	static ClassInfo b = ClassInfo.of(B.class);

	@Test void noValuesSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(b.getAnnotations()));
		var x = XmlSerializer.create().apply(al).build().getSession();
		check("false", x.isAddBeanTypes());
		check("false", x.isAddNamespaceUrisToRoot());
		check("true", x.isAutoDetectNamespaces());
		check("juneau:http://www.apache.org/2013/Juneau", x.getDefaultNamespace());
		check("false", x.isEnableNamespaces());
		check("[]", x.getNamespaces());
	}

	@Test void noValuesParser() {
		var al = AnnotationWorkList.of(sr, rstream(b.getAnnotations()));
		var x = XmlParser.create().apply(al).build().getSession();
		check(null, x.getEventAllocator());
		check("false", x.isPreserveRootElement());
		check(null, x.getReporter());
		check(null, x.getResolver());
		check("false", x.isValidating());
	}

	//-----------------------------------------------------------------------------------------------------------------
	// No annotation.
	//-----------------------------------------------------------------------------------------------------------------

	static class C {}
	static ClassInfo c = ClassInfo.of(C.class);

	@Test void noAnnotationSerializer() {
		var al = AnnotationWorkList.of(sr, rstream(c.getAnnotations()));
		var x = XmlSerializer.create().apply(al).build().getSession();
		check("false", x.isAddBeanTypes());
		check("false", x.isAddNamespaceUrisToRoot());
		check("true", x.isAutoDetectNamespaces());
		check("juneau:http://www.apache.org/2013/Juneau", x.getDefaultNamespace());
		check("false", x.isEnableNamespaces());
		check("[]", x.getNamespaces());
	}

	@Test void noAnnotationParser() {
		var al = AnnotationWorkList.of(sr, rstream(c.getAnnotations()));
		var x = XmlParser.create().apply(al).build().getSession();
		check(null, x.getEventAllocator());
		check("false", x.isPreserveRootElement());
		check(null, x.getReporter());
		check(null, x.getResolver());
		check("false", x.isValidating());
	}
}