/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.terminal;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PrintStream;
import java.io.StringReader;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.amoro.api.config.Configurations;
import org.apache.amoro.server.AmoroManagementConf;
import org.apache.amoro.server.dashboard.utils.DesensitizationUtil;
import org.apache.amoro.server.terminal.ExecutionResult;
import org.apache.amoro.server.terminal.ExecutionStatus;
import org.apache.amoro.server.terminal.StatementResult;
import org.apache.amoro.server.terminal.TerminalSession;
import org.apache.amoro.server.terminal.TerminalSessionFactory;
import org.apache.amoro.shade.guava32.com.google.common.collect.Lists;
import org.apache.amoro.table.TableMetaStore;
import org.apache.commons.io.Charsets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TerminalSessionContext {
    private static final Logger LOG = LoggerFactory.getLogger(TerminalSessionContext.class);
    private final String sessionId;
    private final TableMetaStore metaStore;
    private final AtomicReference<ExecutionStatus> status = new AtomicReference<ExecutionStatus>(ExecutionStatus.Created);
    private volatile ExecutionTask task = null;
    private final TerminalSessionFactory factory;
    private final Configurations sessionConfiguration;
    private final List<String> sensitiveConfKeys;
    private volatile TerminalSession session;
    private volatile long lastExecutionTime = System.currentTimeMillis();
    final ThreadPoolExecutor threadPool;

    public TerminalSessionContext(String sessionId, TableMetaStore metaStore, ThreadPoolExecutor executor, TerminalSessionFactory factory, Configurations sessionConfiguration) {
        this.sessionId = sessionId;
        this.metaStore = metaStore;
        this.threadPool = executor;
        this.factory = factory;
        this.sessionConfiguration = sessionConfiguration;
        this.sensitiveConfKeys = Arrays.stream(sessionConfiguration.getString(AmoroManagementConf.TERMINAL_SENSITIVE_CONF_KEYS).split(",")).map(String::trim).collect(Collectors.toList());
    }

    public String getSessionId() {
        return this.sessionId;
    }

    public boolean isReadyToExecute() {
        return this.isStatusReadyToExecute(this.status.get());
    }

    public boolean isIdleStatus() {
        return this.isStatusReadyToExecute(this.status.get());
    }

    private boolean isStatusReadyToExecute(ExecutionStatus status) {
        return ExecutionStatus.Running != status;
    }

    public synchronized void submit(String catalog, String script, int fetchLimit, boolean stopOnError) {
        ExecutionTask task = new ExecutionTask(catalog, script, fetchLimit, stopOnError);
        if (!this.isReadyToExecute()) {
            throw new IllegalStateException("current session is not ready to execute. status: " + this.status.get().name());
        }
        this.status.set(ExecutionStatus.Running);
        ((CompletableFuture)CompletableFuture.supplyAsync(task, this.threadPool).whenComplete((s, e) -> this.status.compareAndSet(ExecutionStatus.Running, (ExecutionStatus)((Object)s)))).thenApply(s -> {
            this.lastExecutionTime = System.currentTimeMillis();
            return this.lastExecutionTime;
        });
        this.task = task;
        String poolInfo = "new sql script submit, current thread pool state. [Active: " + this.threadPool.getActiveCount() + ", PoolSize: " + this.threadPool.getPoolSize() + "]";
        LOG.info(poolInfo);
        task.executionResult.appendLog(poolInfo);
    }

    public synchronized void cancel() {
        if (this.task != null) {
            this.task.cancel();
        }
    }

    public void release() {
        if (this.session != null) {
            this.session.release();
        }
    }

    public ExecutionStatus getStatus() {
        return this.status.get();
    }

    public synchronized List<String> getLogs() {
        if (this.task != null) {
            return this.task.executionResult.getLogs();
        }
        return Lists.newArrayList();
    }

    public synchronized List<StatementResult> getStatementResults() {
        if (this.task != null) {
            return this.task.executionResult.getResults();
        }
        return Lists.newArrayList();
    }

    public long lastExecutionTime() {
        return this.lastExecutionTime;
    }

    public String lastScript() {
        if (this.task == null) {
            return "";
        }
        return this.task.script;
    }

    private synchronized TerminalSession lazyLoadSession(ExecutionTask task) {
        if (this.session != null && !this.session.active()) {
            task.executionResult.appendLog("terminal session is not active, release session");
            try {
                this.session.release();
            }
            catch (Throwable e) {
                LOG.error("error when release session.");
            }
            finally {
                this.session = null;
            }
        }
        if (this.session == null) {
            task.executionResult.appendLog("terminal session dose not exists. create session first");
            this.session = this.factory.create(this.metaStore, this.sessionConfiguration);
            task.executionResult.appendLog("create a new terminal session.");
        }
        return this.session;
    }

    public Configurations sessionConfiguration() {
        return this.sessionConfiguration;
    }

    private class ExecutionTask
    implements Supplier<ExecutionStatus> {
        final String script;
        final ExecutionResult executionResult = new ExecutionResult();
        private final AtomicBoolean canceled = new AtomicBoolean(false);
        private final int fetchLimits;
        private final boolean stopOnError;
        private final String catalog;

        public ExecutionTask(String catalog, String script, int fetchLimits, boolean stopOnError) {
            this.catalog = catalog;
            this.script = script.trim().endsWith(";") ? script : script + ";";
            this.fetchLimits = fetchLimits;
            this.stopOnError = stopOnError;
        }

        @Override
        public ExecutionStatus get() {
            try {
                return (ExecutionStatus)((Object)TerminalSessionContext.this.metaStore.doAs(() -> {
                    TerminalSession session = TerminalSessionContext.this.lazyLoadSession(this);
                    this.executionResult.appendLog("fetch terminal session: " + TerminalSessionContext.this.sessionId);
                    this.executionResult.appendLogs(session.logs());
                    for (String key : session.configs().keySet()) {
                        if (TerminalSessionContext.this.sensitiveConfKeys.contains(key)) {
                            this.executionResult.appendLog("session configuration: " + key + " => " + DesensitizationUtil.desensitize(session.configs().get(key)));
                            continue;
                        }
                        this.executionResult.appendLog("session configuration: " + key + " => " + session.configs().get(key));
                    }
                    return this.execute(session);
                }));
            }
            catch (Throwable t) {
                LOG.error("something error when execute script. ", t);
                this.executionResult.appendLog("something error when execute script.");
                this.executionResult.appendLog(this.getStackTraceAsString(t));
                return ExecutionStatus.Failed;
            }
        }

        public void cancel() {
            this.canceled.set(true);
        }

        ExecutionStatus execute(TerminalSession session) throws IOException {
            String line;
            LineNumberReader reader = new LineNumberReader(new StringReader(this.script));
            StringBuilder statementBuilder = null;
            int no = -1;
            while ((line = reader.readLine()) != null) {
                if (this.canceled.get()) {
                    this.executionResult.appendLog("execution is canceled. ");
                    return ExecutionStatus.Canceled;
                }
                if (statementBuilder == null) {
                    statementBuilder = new StringBuilder();
                }
                if ((line = line.trim()).length() < 1 || line.startsWith("--") || line.startsWith("#")) continue;
                if (line.endsWith(";")) {
                    statementBuilder.append(line);
                    no = this.lineNumber(reader, no);
                    String statement = statementBuilder.substring(0, statementBuilder.length() - 1);
                    boolean success = this.executeStatement(session, statement, no);
                    if (!success && this.stopOnError) {
                        this.executionResult.appendLog("execution stopped for error happened and stop-when-error config.");
                        return ExecutionStatus.Failed;
                    }
                    statementBuilder = null;
                    no = -1;
                    continue;
                }
                statementBuilder.append(line);
                statementBuilder.append(" ");
                no = this.lineNumber(reader, no);
            }
            return ExecutionStatus.Finished;
        }

        int lineNumber(LineNumberReader reader, int no) {
            if (no < 0) {
                return reader.getLineNumber();
            }
            return no;
        }

        boolean executeStatement(TerminalSession session, String statement, int lineNo) {
            this.executionResult.appendLog(" ");
            this.executionResult.appendLog("prepare execute statement, line:" + lineNo);
            this.executionResult.appendLog(statement);
            TerminalSession.ResultSet rs = null;
            long begin = System.currentTimeMillis();
            try {
                rs = session.executeStatement(this.catalog, statement);
                this.executionResult.appendLogs(session.logs());
            }
            catch (Throwable t) {
                this.executionResult.appendLogs(session.logs());
                this.executionResult.appendLog("meet exception during execution.");
                this.executionResult.appendLog(this.getStackTraceAsString(t));
                return false;
            }
            if (rs.empty()) {
                long cost = System.currentTimeMillis() - begin;
                this.executionResult.appendLog("statement execute down, result is empty, execution cost: " + cost + "ms");
                return true;
            }
            StatementResult sr = this.fetchResults(rs, statement, lineNo);
            long cost = System.currentTimeMillis() - begin;
            this.executionResult.appendResult(sr);
            this.executionResult.appendLog("statement execute down, fetch rows:" + sr.getDatas().size() + ", execution cost: " + cost + "ms");
            return sr.isSuccess();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        StatementResult fetchResults(TerminalSession.ResultSet rs, String statement, int lineNo) {
            long count = 0L;
            StatementResult sr = new StatementResult(statement, lineNo, rs.columns());
            try {
                while (rs.next()) {
                    sr.appendRow(rs.rowData());
                    if (++count < (long)this.fetchLimits) continue;
                    this.executionResult.appendLog("meet result set limit " + count + ", ignore rows left.");
                    break;
                }
            }
            catch (Throwable t) {
                this.executionResult.appendLog("meet exception when fetch result data.");
                String log = this.getStackTraceAsString(t);
                sr.withExceptionLog(log);
                this.executionResult.appendLog(log);
            }
            finally {
                try {
                    rs.close();
                }
                catch (Throwable throwable) {}
            }
            return sr;
        }

        String getStackTraceAsString(Throwable t) {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            PrintStream ps = new PrintStream(out);
            t.printStackTrace(ps);
            return new String(out.toByteArray(), Charsets.UTF_8);
        }
    }
}

