/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import io.netty.buffer.ByteBuf;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.apache.bookkeeper.bookie.AbstractLogCompactor;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.EntryLocation;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.util.HardLink;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransactionalEntryLogCompactor
extends AbstractLogCompactor {
    private static final Logger LOG = LoggerFactory.getLogger(TransactionalEntryLogCompactor.class);
    final EntryLogger entryLogger;
    final CompactableLedgerStorage ledgerStorage;
    final List<EntryLocation> offsets = new ArrayList<EntryLocation>();
    static final String COMPACTING_SUFFIX = ".log.compacting";
    static final String COMPACTED_SUFFIX = ".compacted";

    public TransactionalEntryLogCompactor(ServerConfiguration conf, EntryLogger entryLogger, CompactableLedgerStorage ledgerStorage, AbstractLogCompactor.LogRemovalListener logRemover) {
        super(conf, logRemover);
        this.entryLogger = entryLogger;
        this.ledgerStorage = ledgerStorage;
    }

    @Override
    public void cleanUpAndRecover() {
        List<File> ledgerDirs = this.entryLogger.getLedgerDirsManager().getAllLedgerDirs();
        for (File dir : ledgerDirs) {
            File[] compactedPhaseFiles;
            File[] compactingPhaseFiles = dir.listFiles(file -> file.getName().endsWith(COMPACTING_SUFFIX));
            if (compactingPhaseFiles != null) {
                for (File file2 : compactingPhaseFiles) {
                    if (!file2.delete()) continue;
                    LOG.info("Deleted failed compaction file {}", (Object)file2);
                }
            }
            if ((compactedPhaseFiles = dir.listFiles(file -> file.getName().endsWith(COMPACTED_SUFFIX))) == null) continue;
            for (File compactedFile : compactedPhaseFiles) {
                LOG.info("Found compacted log file {} has partially flushed index, recovering index.", (Object)compactedFile);
                UpdateIndexPhase updateIndex = new UpdateIndexPhase(compactedFile, true);
                updateIndex.run();
            }
        }
    }

    @Override
    public boolean compact(EntryLogMetadata metadata) {
        if (metadata != null) {
            LOG.info("Compacting entry log {} with usage {}.", (Object)metadata.getEntryLogId(), (Object)metadata.getUsage());
            ScanEntryLogPhase scanEntryLog = new ScanEntryLogPhase(metadata);
            if (!scanEntryLog.run()) {
                LOG.info("Compaction for entry log {} end in ScanEntryLogPhase.", (Object)metadata.getEntryLogId());
                return false;
            }
            File compactionLogFile = this.entryLogger.getCurCompactionLogFile();
            FlushCompactionLogPhase flushCompactionLog = new FlushCompactionLogPhase(metadata.getEntryLogId());
            if (!flushCompactionLog.run()) {
                LOG.info("Compaction for entry log {} end in FlushCompactionLogPhase.", (Object)metadata.getEntryLogId());
                return false;
            }
            File compactedLogFile = this.getCompactedLogFile(compactionLogFile, metadata.getEntryLogId());
            UpdateIndexPhase updateIndex = new UpdateIndexPhase(compactedLogFile);
            if (!updateIndex.run()) {
                LOG.info("Compaction for entry log {} end in UpdateIndexPhase.", (Object)metadata.getEntryLogId());
                return false;
            }
            LOG.info("Compacted entry log : {}.", (Object)metadata.getEntryLogId());
            return true;
        }
        return false;
    }

    File getCompactedLogFile(File compactionLogFile, long compactingLogId) {
        if (compactionLogFile == null) {
            return null;
        }
        File dir = compactionLogFile.getParentFile();
        String filename = compactionLogFile.getName();
        String newSuffix = ".log." + EntryLogger.logId2HexString(compactingLogId) + COMPACTED_SUFFIX;
        String hardLinkFilename = filename.replace(COMPACTING_SUFFIX, newSuffix);
        return new File(dir, hardLinkFilename);
    }

    class UpdateIndexPhase
    extends CompactionPhase {
        File compactedLogFile;
        File newEntryLogFile;
        private final boolean isInRecovery;

        public UpdateIndexPhase(File compactedLogFile) {
            this(compactedLogFile, false);
        }

        public UpdateIndexPhase(File compactedLogFile, boolean isInRecovery) {
            super("UpdateIndexPhase");
            this.compactedLogFile = compactedLogFile;
            this.isInRecovery = isInRecovery;
        }

        @Override
        void start() throws IOException {
            if (this.compactedLogFile != null && this.compactedLogFile.exists()) {
                File dir = this.compactedLogFile.getParentFile();
                String compactedFilename = this.compactedLogFile.getName();
                this.newEntryLogFile = new File(dir, compactedFilename.substring(0, compactedFilename.indexOf(".log") + 4));
                if (!this.newEntryLogFile.exists()) {
                    HardLink.createHardLink(this.compactedLogFile, this.newEntryLogFile);
                }
                if (this.isInRecovery) {
                    this.recoverEntryLocations(EntryLogger.fileName2LogId(this.newEntryLogFile.getName()));
                }
                if (!TransactionalEntryLogCompactor.this.offsets.isEmpty()) {
                    TransactionalEntryLogCompactor.this.ledgerStorage.updateEntriesLocations(TransactionalEntryLogCompactor.this.offsets);
                    TransactionalEntryLogCompactor.this.ledgerStorage.flushEntriesLocationsIndex();
                }
            } else {
                throw new IOException("Failed to find compacted log file in UpdateIndexPhase");
            }
        }

        @Override
        boolean complete() {
            TransactionalEntryLogCompactor.this.offsets.clear();
            if (this.compactedLogFile != null) {
                if (!this.compactedLogFile.delete()) {
                    LOG.warn("Could not delete compacted log file {}", (Object)this.compactedLogFile);
                }
                String compactedFilename = this.compactedLogFile.getName();
                String oldEntryLogFilename = compactedFilename.substring(compactedFilename.indexOf(".log") + 5);
                long entryLogId = EntryLogger.fileName2LogId(oldEntryLogFilename);
                TransactionalEntryLogCompactor.this.logRemovalListener.removeEntryLog(entryLogId);
            }
            return true;
        }

        @Override
        void abort() {
            TransactionalEntryLogCompactor.this.offsets.clear();
        }

        private void recoverEntryLocations(final long compactedLogId) throws IOException {
            TransactionalEntryLogCompactor.this.entryLogger.scanEntryLog(compactedLogId, new EntryLogger.EntryLogScanner(){

                @Override
                public boolean accept(long ledgerId) {
                    return true;
                }

                @Override
                public void process(long ledgerId, long offset, ByteBuf entry) throws IOException {
                    long lid = entry.getLong(entry.readerIndex());
                    long entryId = entry.getLong(entry.readerIndex() + 8);
                    if (lid != ledgerId || entryId < -1L) {
                        LOG.warn("Scanning expected ledgerId {}, but found invalid entry with ledgerId {} entryId {} at offset {}", new Object[]{ledgerId, lid, entryId, offset});
                        throw new IOException("Invalid entry found @ offset " + offset);
                    }
                    long location = compactedLogId << 32 | offset + 4L;
                    TransactionalEntryLogCompactor.this.offsets.add(new EntryLocation(lid, entryId, location));
                }
            });
            LOG.info("Recovered {} entry locations from compacted log {}", (Object)TransactionalEntryLogCompactor.this.offsets.size(), (Object)compactedLogId);
        }
    }

    class FlushCompactionLogPhase
    extends CompactionPhase {
        private final long compactingLogId;
        private File compactedLogFile;

        FlushCompactionLogPhase(long compactingLogId) {
            super("FlushCompactionLogPhase");
            this.compactingLogId = compactingLogId;
        }

        @Override
        void start() throws IOException {
            File compactionLogFile = TransactionalEntryLogCompactor.this.entryLogger.getCurCompactionLogFile();
            if (compactionLogFile == null || !compactionLogFile.exists()) {
                throw new IOException("Compaction log doesn't exist during flushing");
            }
            TransactionalEntryLogCompactor.this.entryLogger.flushCompactionLog();
        }

        @Override
        boolean complete() throws IOException {
            File compactionLogFile = TransactionalEntryLogCompactor.this.entryLogger.getCurCompactionLogFile();
            if (compactionLogFile == null || !compactionLogFile.exists()) {
                LOG.warn("Compaction log doesn't exist any more after flush");
                return false;
            }
            this.compactedLogFile = TransactionalEntryLogCompactor.this.getCompactedLogFile(compactionLogFile, this.compactingLogId);
            if (this.compactedLogFile != null && !this.compactedLogFile.exists()) {
                HardLink.createHardLink(compactionLogFile, this.compactedLogFile);
            }
            TransactionalEntryLogCompactor.this.entryLogger.removeCurCompactionLog();
            return true;
        }

        @Override
        void abort() {
            TransactionalEntryLogCompactor.this.offsets.clear();
            TransactionalEntryLogCompactor.this.entryLogger.removeCurCompactionLog();
            if (this.compactedLogFile != null && this.compactedLogFile.exists() && !this.compactedLogFile.delete()) {
                LOG.warn("Could not delete compacted log file {}", (Object)this.compactedLogFile);
            }
        }
    }

    class ScanEntryLogPhase
    extends CompactionPhase {
        private final EntryLogMetadata metadata;

        ScanEntryLogPhase(EntryLogMetadata metadata) {
            super("ScanEntryLogPhase");
            this.metadata = metadata;
        }

        @Override
        void start() throws IOException {
            TransactionalEntryLogCompactor.this.entryLogger.createNewCompactionLog();
            TransactionalEntryLogCompactor.this.entryLogger.scanEntryLog(this.metadata.getEntryLogId(), new EntryLogger.EntryLogScanner(){

                @Override
                public boolean accept(long ledgerId) {
                    return ScanEntryLogPhase.this.metadata.containsLedger(ledgerId);
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void process(long ledgerId, long offset, ByteBuf entry) throws IOException {
                    TransactionalEntryLogCompactor.this.throttler.acquire(entry.readableBytes());
                    TransactionalEntryLogCompactor transactionalEntryLogCompactor = TransactionalEntryLogCompactor.this;
                    synchronized (transactionalEntryLogCompactor) {
                        long lid = entry.getLong(entry.readerIndex());
                        long entryId = entry.getLong(entry.readerIndex() + 8);
                        if (lid != ledgerId || entryId < -1L) {
                            LOG.warn("Scanning expected ledgerId {}, but found invalid entry with ledgerId {} entryId {} at offset {}", new Object[]{ledgerId, lid, entryId, offset});
                            throw new IOException("Invalid entry found @ offset " + offset);
                        }
                        long newOffset = TransactionalEntryLogCompactor.this.entryLogger.addEntryForCompaction(ledgerId, entry);
                        TransactionalEntryLogCompactor.this.offsets.add(new EntryLocation(ledgerId, entryId, newOffset));
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Compact add entry : lid = {}, eid = {}, offset = {}", new Object[]{ledgerId, entryId, newOffset});
                        }
                    }
                }
            });
        }

        @Override
        boolean complete() {
            if (TransactionalEntryLogCompactor.this.offsets.isEmpty()) {
                LOG.info("No valid entry is found in entry log after scan, removing entry log now.");
                TransactionalEntryLogCompactor.this.logRemovalListener.removeEntryLog(this.metadata.getEntryLogId());
                TransactionalEntryLogCompactor.this.entryLogger.removeCurCompactionLog();
                return false;
            }
            return true;
        }

        @Override
        void abort() {
            TransactionalEntryLogCompactor.this.offsets.clear();
            TransactionalEntryLogCompactor.this.entryLogger.removeCurCompactionLog();
        }
    }

    static abstract class CompactionPhase {
        private String phaseName = "";

        CompactionPhase(String phaseName) {
            this.phaseName = phaseName;
        }

        boolean run() {
            try {
                this.start();
                return this.complete();
            }
            catch (IOException e) {
                LOG.error("Encounter exception in compaction phase {}. Abort current compaction.", (Object)this.phaseName, (Object)e);
                this.abort();
                return false;
            }
        }

        abstract void start() throws IOException;

        abstract boolean complete() throws IOException;

        abstract void abort();
    }
}

